globals [
  carbon-price        ; ETS carbon price
  government-penalty  ; Penalty value
  RR                  ; Rate of carbon reduction
  delta-output        ; Threshold for output changes
  theta                ; Threshold for selling carbon allowances
]

turtles-own [
  firm-emissions       ; Firm emissions specific to each turtle
  carbon-allowances    ; Carbon allowances specific to each turtle
  mac                  ; Marginal abatement cost specific to each turtle
  emission-trade?      ; Boolean: True if the firm is trading carbon
  emission-reduce?     ; Boolean: True if the firm is reducing emissions
  penalty-accepted?    ; Boolean: True if the firm accepts the penalty
  carbon_int           ; Emission intensity of the firm (tCO2/MWh)
  output               ; Output value of the firm (MWh)
  carbon_emi           ; Yearly carbon emissions (tCO2)
  bal                  ; Carbon balance (carbon emissions minus allowances)
  pro                  ; Current profit
  Cat                  ; Industry category (A, B, C, or D)
  Tec_D                ; Decision to invest in carbon reduction technology
  Trade_D              ; Decision to trade, store, or accept penalty
  A_next               ; Allowance for the next year
  ]

to setup-world
  clear-all
  ;; Create market zones: e.g., green = buy zone, red = penalty zone, blue = sell zone
  ask patches [
    if pxcor < -5 [
      set pcolor green   ; Buy zone
    ]
    if pxcor > 5 [
      set pcolor blue    ; Sell zone
    ]
    if pxcor >= -5 and pxcor <= 5 [
      set pcolor red     ; Penalty zone
    ]
  ]
end

to setup
  clear-all
  setup-world
  set-default-shape turtles "circle"
  set theta 0.1  ; Example threshold for selling allowances
  ;; Create 50 firms (turtles)
  create-turtles 50 [
    set color one-of [green blue red]   ; Random color for now
    set shape "circle"                  ; Set the shape of the firms
    set size 2                          ; Make turtles bigger so they are visible
    setxy random-xcor random-ycor       ; Randomly position turtles
    set Cat one-of ["A" "B" "C" "D"]   ; Set random industry category
    industry-values                    ; Set initial values based on industry category
    set firm-emissions initial-firm-emissions  ; Using slider value
    set carbon-allowances initial-carbon-allowances  ; Using slider value
        set carbon_emi output * carbon_int ; Initial carbon emissions
    set carbon-allowances (carbon_emi * RR)  ; Allowance is based on reduction target
    set bal carbon_emi - carbon-allowances ; Initial carbon balance
    set pro random 1000                ; Placeholder for profit
    set A_next carbon-allowances       ; Initial next year's allowance
    set emission-trade? false
    set emission-reduce? false
    set penalty-accepted? false
    set Tec_D false                    ; Initially no investment in carbon reduction
    set Trade_D "none"                 ; No trading decision at the start
    ;; Set initial RR (for example, starting at 1.0, or 100%)
    set RR 1.0

  ]
  ;; Set default values for carbon price and penalty
    set carbon-price slider-carbon-price  ; Example default carbon price
    set government-penalty slider-government-penalty ; Example penalty for exceeding allowances
  reset-ticks
end

to industry-values
  ; Set the firm-specific values based on the industry category
  if Cat = "A" [
    set mac 392.7 * (5 / 6.6)           ; MAC for Type A
    set carbon_int 0.911                ; Emission intensity for Type A
    set output random 2000000 + 400000         ; Random output for Type A
  ]

  if Cat = "B" [
    set mac 392.7 * (10 / 11.06)        ; MAC for Type B
    set carbon_int 1.011                ; Emission intensity for Type B
    set output random 1500000 + 300000         ; Random output for Type B
  ]

  if Cat = "C" [
    set mac 392.7 * (15 / 15.61)        ; MAC for Type C
    set carbon_int 1.089                ; Emission intensity for Type C
    set output random 1000000 + 200000         ; Random output for Type C
  ]

  if Cat = "D" [
    set mac 392.7 * (20 / 19.25)        ; MAC for Type D
    set carbon_int 1.297                ; Emission intensity for Type D
    set output random 1000000 + 100000         ; Random output for Type D
  ]
end

to go
  ;; Firms make decisions and interact in the market
  ask turtles [
    make-decisions
    calculate-carbon-balance
    make-trade-or-penalty-decisions
  ]
  ;; Reduce RR by 2% every tick
  set RR RR * 0.98  ; Decrease RR by 2%

  ;; Update the movement and visualization based on decisions
  move-firms
  update-firm-visualization

  ;; Gradually adjust emissions (simulate production and reduction over time)
  ask turtles [
    if emission-reduce? [
      set firm-emissions firm-emissions * 0.98  ; Decrease emissions by 2% if reducing emissions
    ]
      set firm-emissions firm-emissions * 1.01  ; Increase emissions by 1% due to production

  ]
  ;; Plot and other updates
  update-carbon-price
  update-decision-plot

  tick
end


; Calculate the carbon emissions and decide whether to reduce emissions
to make-decisions
  let predicted-emissions output * carbon_int  ; Calculate emissions based on output and intensity

  ;; Condition 1: Reduce emissions if MAC is less than carbon price and firm is not already reducing
  if (mac < carbon-price) and (not emission-reduce?) [
    set Tec_D true                     ; Invest in technology to reduce emissions
    set emission-reduce? true          ; Mark that the firm is reducing emissions
    reduce-emissions
  ]
  if (emission-reduce? and firm-emissions <= carbon-allowances) [
    set emission-reduce? false         ; Stop reducing emissions when under allowance
  ]
end

to reduce-emissions
  set firm-emissions (firm-emissions - random-float 2)  ; Reduce emissions by a random factor
end


; Calculate carbon balance (Eq. 4)
to calculate-carbon-balance
  set carbon_emi output * carbon_int   ; Calculate actual carbon emissions
  set carbon-allowances carbon_emi * RR ; Allowances decrease based on RR
  set bal carbon_emi - carbon-allowances ; Calculate carbon balance
end

; Make decisions on trading or accepting penalties based on carbon balance
to make-trade-or-penalty-decisions
  ;; Check if emissions exceed allowance
  if (bal > 0) [
    ;; Calculate purchase cost and penalty cost
    let purchase-cost bal * carbon-price
    let penalty-cost bal * government-penalty  ; Penalty is proportional to excess emissions

    ;; Print the costs for debugging purposes
    print (word "Turtle " who " - Purchase Cost: " purchase-cost " vs Penalty Cost: " penalty-cost)

    ;; Condition 3: Buy allowances if it's cheaper than the penalty
    ifelse (purchase-cost < penalty-cost) [
      set Trade_D "buy"
      trade-carbon
      set penalty-accepted? false
      print (word "Turtle " who " is buying allowances.")
    ] [
      ;; Condition 4: Accept penalty if buying allowances is more expensive
      set penalty-accepted? true
      set Trade_D "none"
      print (word "Turtle " who " is accepting penalty.")
    ]
  ]
  ;; Condition 5 & 6: Sell or store extra allowances if emissions are below allowance
  ifelse ((abs(bal) > (A_next - carbon-allowances)) and (abs(bal) * carbon-price > pro * theta)) [
    set Trade_D "sell"
    sell-allowances
    print (word "Turtle " who " is selling allowances.")
  ] [
    ;; Condition 6: Store allowances for the next period if selling is not profitable
    set Trade_D "store"
    print (word "Turtle " who " is storing allowances.")
  ]
end


; Procedure to trade allowances (buy or sell)
to trade-carbon
  if (Trade_D = "buy") [
    set carbon-allowances (carbon-allowances + random-float 1)  ; Buy allowances
    set pro (pro - carbon-price * bal)  ; Decrease profit by the cost of allowances
  ]
end

to sell-allowances
  set carbon-allowances (carbon-allowances - random-float 1)  ; Sell extra allowances
  set pro (pro + carbon-price * abs(bal))  ; Increase profit by selling allowances
end

to trade-between-firms
  ask turtles with [Trade_D = "buy"] [
    let seller one-of turtles with [Trade_D = "sell"]
    if seller != nobody [
      ; Execute trade (exchange allowances and update profit)
      ask seller [
        set carbon-allowances carbon-allowances - 1
        set pro pro + carbon-price
      ]
      set carbon-allowances carbon-allowances + 1
      set pro pro - carbon-price
    ]
  ]
end

; Apply penalty to firms that exceed their carbon allowances
to apply-penalty
  if (penalty-accepted?) [
    set pro (pro - government-penalty)
  ]
end

; Procedure to update carbon price based on trading activity
to update-carbon-price
  ; Carbon price influenced by trading decisions, emissions, and supply-demand dynamics

  ;; Calculate the number of firms buying and selling allowances
  let total-buyers count turtles with [Trade_D = "buy"]
  let total-sellers count turtles with [Trade_D = "sell"]
  let total-traded total-buyers + total-sellers  ; Total market activity (both buying and selling)

  ;; Adjust carbon price based on market supply and demand (buyers increase, sellers decrease)
  let price-adjustment (total-buyers - total-sellers) * 0.1  ; Small adjustment factor

  ;; Calculate total emissions (sum of all firm emissions)
  let total-emissions sum [firm-emissions] of turtles

  ;; Add random fluctuation to simulate market volatility
  let random-fluctuation (random-float 0.05) - 0.025  ; Random fluctuation between -0.025 and +0.025

  ;; Update the carbon price based on trading, emissions, and market fluctuation
  set carbon-price (carbon-price + price-adjustment + (total-traded * 0.1) + (total-emissions * 0.001) + random-fluctuation)

  ;; Ensure the carbon price stays above a minimum value (e.g., 1.0)
  if carbon-price < 1.0 [ set carbon-price 1.0 ]

  ;; Print the new carbon price for debugging (optional)
  print (word "New carbon price: " carbon-price)
end



to update-visualization
  ask turtles [
    ; Color and shape based on emissions status and decision to reduce emissions
    ifelse (firm-emissions > carbon-allowances) [
      set shape "triangle"   ; Firms with excess emissions become triangles
      set color red          ; Indicating high emissions
    ] [
      set shape "circle"     ; Firms with low emissions stay as circles
      set color green
    ]

    ; Change color if the firm is trading carbon
    if (emission-trade?) [
      set color yellow       ; Firms that are actively trading become yellow
    ]

    ; Indicate if the firm has accepted a penalty
    if (penalty-accepted?) [
      set color gray         ; Firms that have accepted penalties become gray
    ]
  ]
end

to move-firms
  ask turtles [
    ;; Move to buy zone if trading
    if Trade_D = "buy" [
      let target-patch one-of patches with [pcolor = green]
      if target-patch != nobody [
        move-to target-patch
      ]
    ]

    ;; Move to sell zone if selling
    if Trade_D = "sell" [
      let target-patch one-of patches with [pcolor = blue]
      if target-patch != nobody [
        move-to target-patch
      ]
    ]

    ;; Move to penalty zone if accepting penalty
    if penalty-accepted? [
      let target-patch one-of patches with [pcolor = red]
      if target-patch != nobody [
        move-to target-patch
      ]
    ]
  ]
end


to update-firm-visualization
  ask turtles [
    ;; If firm is buying, turn blue and make it a square
    if Trade_D = "buy" [
      set shape "square"
      set color blue
    ]

    ;; If firm is selling, turn green and make it a triangle
    if Trade_D = "sell" [
      set shape "triangle"
      set color green
    ]

    ;; If firm accepts penalty, turn red and make it a circle
    if penalty-accepted? [
      set shape "circle"
      set color red
    ]

    ;; If firm is compliant and not trading, keep it a gray circle
    if not penalty-accepted? and Trade_D = "none" [
      set shape "circle"
      set color gray
    ]
  ]
end



to update-decision-plot
  set-current-plot "Firm Decisions"  ; Select the plot

  ; Pen 1: Plot number of firms trading
  set-current-plot-pen "trade"       ; Set the current pen to "trade"
  let num-trade count turtles with [Trade_D = "buy" or Trade_D = "sell"]
  plot num-trade                     ; Plot the number of firms trading

  ; Pen 2: Plot number of firms reducing emissions
  set-current-plot-pen "reduce"      ; Set the current pen to "reduce"
  let num-reduce count turtles with [emission-reduce? = true]
  plot num-reduce                    ; Plot the number of firms reducing emissions

  ; Pen 3: Plot number of firms accepting penalties
  set-current-plot-pen "penalty"     ; Set the current pen to "penalty"
  let num-penalty count turtles with [penalty-accepted? = true]
  plot num-penalty                   ; Plot the number of firms accepting penalties

  set-current-plot "Carbon Price Over Time"
  plot carbon-price  ; Plot the carbon price over time
end

to update-profit
  ; Profit influenced by emissions, penalties, and trading
  if (emission-trade?) [
    set pro (pro + random-float 100 - (carbon-price * firm-emissions))
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
282
16
719
454
-1
-1
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
1
1
1
ticks
30.0

PLOT
762
21
962
171
Carbon Price Over Time
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"set-current-plot \"Carbon Price Over Time\"" "plot carbon-price"
PENS
"carbon-price" 1.0 0 -16777216 true "" ""

SLIDER
7
43
246
76
initial-firm-emissions
initial-firm-emissions
0
10
1.8
0.1
1
Million tCO2e
HORIZONTAL

SLIDER
7
92
264
125
initial-carbon-allowances
initial-carbon-allowances
0
5
1.7
0.1
1
Million tCO2e
HORIZONTAL

BUTTON
11
314
75
347
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
87
315
150
348
Go
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
765
192
965
342
Firm Decisions
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" "; Calculate the number of firms for each decision\nlet num-trade count turtles with [Trade_D = \"buy\" or Trade_D = \"sell\"]\nlet num-reduce count turtles with [emission-reduce? = true]\nlet num-penalty count turtles with [penalty-accepted? = true]\n\n; Update the \"trade\" pen\nset-current-plot-pen \"trade\"\nplot num-trade\n\n; Update the \"reduce\" pen\nset-current-plot-pen \"reduce\"\nplot num-reduce\n\n; Update the \"penalty\" pen\nset-current-plot-pen \"penalty\"\nplot num-penalty\n"
PENS
"trade" 1.0 0 -13345367 true " set-current-plot-pen \"trade\"       ; Set the current pen to \"trade\"\n  let num-trade count turtles with [Trade_D = \"buy\" or Trade_D = \"sell\"]\n  plot num-trade                     ; Plot the number of firms trading" ""
"reduce" 1.0 0 -10899396 true "" ""
"penalty" 1.0 0 -2674135 true "" ""

BUTTON
162
314
262
347
Go Forever?
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
10
182
213
215
slider-carbon-price
slider-carbon-price
0
100
4.0
1
1
$/tCO2e
HORIZONTAL

SLIDER
10
230
257
263
slider-government-penalty
slider-government-penalty
0
100
14.0
1
1
$/tCO2e
HORIZONTAL

TEXTBOX
290
478
769
610
Circle Company = low emission\nTriangle Company = exceeding emission\nGreen zone = Buying in market\nRed Zone = accepting Penalty\nBlue zone = Selling in market
18
0.0
1

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
