;;Assumptions:
;;
;;  - All turtles are centered on a cell, otherwise it won't work
;;  - worldWidth is even, so that the cells wraps properly         (imposed by slider)
;;  - xHeight    is even, so that cell centres are integral pixels (imposed by slider)
;;
;; PROCEDURE and VARIABLE USAGE
;;
;; setup                     <<< User
;; setup-hex-grid            <<< setup
;; drag                      <<< User
;; drag-hex                  <<< drag
;; selected-hex-reporter     <<< drag-hex, delete-hex-atMouse, recolor-hex-atMouse
;; selected-cell-reporter    <<< selected-hex-reporter, add-hex-atMouse, add-walker-atMouse
;; add-hex-atMouse           <<< User
;; add-hex                   <<< add-hex-atMouse
;; top-hex-reporter          <<< selected-hex-reporter, add-hex
;; next-color-reporter       <<< recolor-hex-atMouse, add-hex
;; recolor-hex-atMouse       <<< User
;; delete-hex-atMouse        <<< User
;; add-walker-atMouse        <<< User
;; add-walker                <<< add-walker-atMouse, setup            (sets             heading of walker)
;; xNeighborsInGrid-reporter <<< add-walker, walk, test-xNeighbors
;; walk                      <<< User                                 (uses and updates heading of walker)
;; xNeighbor-reporter        <<< walk, xNeighborsInGrid-reporter
;; test-xNeighbors           <<< User

globals [xDx xDy cell-size hex-size xNeighbors-xDx xNeighbors-xDy]
breed [cells   cell  ]
breed [hexs    hex   ]
breed [walkers walker]

to setup
  clear-all

  setup-hex-grid                                            ;; create a hex grid of cells

  ;; setup walkers
  set-default-shape walkers "arrow half"
  repeat #walkers [ask one-of cells [add-walker] ]          ;; add #walkers walkers

  reset-ticks
end

to setup-hex-grid                                           ;; set up a grid of hexagonal cells
  ifelse TrueHex [
    set-default-shape cells   "truhex-outline"
    set-default-shape hexs    "truhex"
    set xDx round(xHeight / 2 * sqrt 3)                     ;; pxcor displacement of xNeighbors
    set xDy       xHeight                                   ;; pycor displacement of xNeighbors
    set-patch-size 1
    set cell-size round(2 * xDy / sqrt 3)
    set hex-size cell-size - 2                              ;; so that it just fits within the outine.     Used in add-hex
   ][
    set-default-shape cells   "hex-outline"
    set-default-shape hexs    "hex"
    set xDx 1                                               ;; pxcor displacement of xNeighbors
    set xDy 1                                               ;; pycor displacement of xNeighbors
    set-patch-size xHeight
    set cell-size 4 / 3                                     ;; Make cells fit fully by expanding beyond patch .
    set hex-size cell-size - .04
  ]

  set xNeighbors-xDx map [ ?1 -> xDx * ?1 ] [0  1    1    0  -1   -1  ]  ;; xDx of xNeighbors, listed by xHeading.  Used in xNeighbor-reporter
  set xNeighbors-xDy map [ ?1 -> xDy * ?1 ] [1  0.5 -0.5 -1  -0.5  0.5]  ;; xDy of xNeighbors, listed by xHeading.  Used in xNeighbor-reporter

  ;; resize the world
  no-display
  let pixels worldWidth * xDx * worldHeight * xDy
  if pixels > 999999 [error (word "World too big! It has " pixels " pixels.  You should adjust its parameters.")]
  resize-world 0 (worldWidth * xDx - 1) 0 (worldHeight * xDy   - 1)  ;;subtract 1 is in order to wrap exactly

  ;;create cells to form a hex grid
  foreach   n-values worldWidth  [ ?1 -> ?1 ] [ ?1 -> let cellX ?1
    foreach n-values worldHeight [ ??1 -> ??1 ] [ ??1 -> let cellY ??1
      ;; sprout cell(cellX,cellY) at its central patch. Shift even columns half a cell up/down, so that the cells fit together.
      ask patch (cellX * xDx) (cellY * xDy) [
        sprout-cells 1[
          set size cell-size                                ;; Make cells fit fully by expanding beyond patch
          set color gray
          ;; In order to stay within the World (if bounded). Shift up if TrueHex, else shift down.
          if (xcor / xDx) mod 2 = 0 [set ycor ycor + (ifelse-value TrueHex [1][-1]) * (xDy / 2)]
        ]
      ]
    ]
  ]
  display
end

to drag                                                     ;; User method
  if mouse-down? [drag-hex]
end

to drag-hex
  ask selected-hex-reporter [hatch 1  die]                  ;; re-incarnate moused-hex so that it will be on top of any cell it is dragged over

  ask selected-hex-reporter [                               ;; selected-hex is now the re-incarnated moused-hex
    while [mouse-down?] [
      setxy mouse-xcor mouse-ycor
      display                                               ;; force the display to update, to see the turtle moving around
    ]

    move-to min-one-of cells [distance myself]              ;; snap to nearest cell at end of drag
    display                                                 ;; force the display to update, to see the snap
  ]
end

to-report selected-hex-reporter                             ;; set of the top (newest added) hex that the mouse is over ('empty' if none)
  report turtle-set [top-hex-reporter] of selected-cell-reporter
end

to-report top-hex-reporter                                  ;; cell reporter: top-hex on this cell ('nobody' if none)
  report max-one-of (hexs-on self) [who]
end

to-report selected-cell-reporter                            ;; set of the cell the mouse is over ('empty' if outside world)
  report ifelse-value mouse-inside? [turtle-set min-one-of cells [distancexy mouse-xcor mouse-ycor]]
                                    [no-turtles]
end

to add-hex-atMouse                                          ;; User method: add a hex on selected cell (on top of any others; no action if mouse outside world)
  ask selected-cell-reporter [add-hex]
end

to add-hex                                                  ;; cell method: add a hex on top, with the next color after the previous top hex (or red, if none)
  ;; First, get the current-color of the top hex on this cell, or "none"
  let top-hex top-hex-reporter
  let current-color ifelse-value (is-turtle? top-hex) [ [color] of top-hex ] ["none"]

  ;; Now, add a hex with the next color
  hatch-hexs 1 [
    set size hex-size                                       ;; shrink hex to fit just inside the grid lines
    set color next-color-reporter current-color
  ]
end

to-report next-color-reporter [current-color]               ;; color after current-color (or red, if not in swatch)
  let swatch [red sky yellow lime red]
  let next-swatch-position ifelse-value (member? current-color swatch) [1 + position current-color swatch] [0]
  report item next-swatch-position swatch
end

to recolor-hex-atMouse                                      ;; User method: set color of top moused hex to next color in swatch
  ask selected-hex-reporter [set color (next-color-reporter color)]
end

to delete-hex-atMouse                                       ;; User method: delete the top moused hex
  ask selected-hex-reporter [die]
end

to add-walker-atMouse                                       ;; User method: add a walker on selected cell (no action if mouse outside world)
  ask selected-cell-reporter [add-walker]
end

to add-walker                                               ;; turtle method: create a walker at this turtle
  hatch-walkers 1 [
    set size xDy * 2                                        ;; to extend head of walker to the centre of neighbor cell
    set color pink
    face one-of xNeighborsInGrid-reporter                   ;; face any xNeighbor in the grid (assumes 'no cell is an island')
  ]
  set #walkers count walkers
end

to walk                                                     ;; walker method: step randomly, staying in-grid. Between steps the walker heads from where it came.
  let xNeighbor xNeighbor-reporter ((round(heading / 60) + one-of [1 -1 0 0 0 0]) mod 6)  ;; mainly head straight, turning just +/-60 a third of the time
  if xNeighbor = nobody [set xNeighbor one-of xNeighborsInGrid-reporter]  ;; if not in-grid, set xNeighbor to any in-grid
  face xNeighbor
  move-to xNeighbor
end

to-report xNeighborsInGrid-reporter                         ;; turtle reporter: all xNeighbor cells in the grid
  report turtle-set map [ ?1 -> xNeighbor-reporter ?1 ] [0 1 2 3 4 5]
end

to test-xNeighbors                                          ;; User method: shows the xNeighbors of each individual cell
  ask cells [
    hatch-hexs 1[set color blue]
    ask xNeighborsInGrid-reporter [hatch-hexs 1[set color gray]]
    if user-yes-or-no? "Hit Enter to proceed to next cell" []
    ask hexs [die]
  ]
end

to-report xNeighbor-reporter [xHeadingArg]                  ;; turtle reporter: report xNeighbor cell in given direction, or "nobody" if it's off-grid
  ;; xHeading is the index (0,1,2,3,4 or 5 - clockwise, starting from North) of any xNeighbor of a cell
  report one-of cells-at (item xHeadingArg xNeighbors-xDx) (item xHeadingArg xNeighbors-xDy)
end
@#$#@#$#@
GRAPHICS-WINDOW
135
10
663
619
-1
-1
1.0
1
10
1
1
1
0
1
1
1
0
519
0
599
1
1
1
ticks
30.0

BUTTON
5
229
86
262
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
6
362
88
395
drag-hex
drag
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
6
654
89
687
NIL
ask walkers[pd]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
4
47
131
80
worldHeight
worldHeight
1
50
4.0
1
1
cells
HORIZONTAL

SLIDER
3
10
130
43
worldWidth
worldWidth
2
50
4.0
2
1
cells
HORIZONTAL

SLIDER
4
84
132
117
xHeight
xHeight
2
200
150.0
2
1
pixels
HORIZONTAL

SLIDER
4
120
132
153
#walkers
#walkers
0
10
1.0
1
1
NIL
HORIZONTAL

BUTTON
3
515
118
548
NIL
add-walker-atMouse
NIL
1
T
OBSERVER
NIL
W
NIL
NIL
0

BUTTON
6
691
91
724
Clear walkers
ask walkers[die]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
5
401
120
434
NIL
add-hex-atMouse
NIL
1
T
OBSERVER
NIL
X
NIL
NIL
0

BUTTON
5
439
119
472
NIL
recolor-hex-atMouse
NIL
1
T
OBSERVER
NIL
C
NIL
NIL
0

BUTTON
4
478
117
511
NIL
delete-hex-atMouse
NIL
1
T
OBSERVER
NIL
D
NIL
NIL
0

TEXTBOX
3
600
120
628
The following are purely for test purposes:
11
0.0
1

BUTTON
6
728
92
761
Clear hexs
ask hexs [die]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
6
263
69
296
walk
ask walkers [walk]\ntick
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
6
298
69
331
walk
ask walkers [walk]\ntick
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
6
766
94
799
NIL
test-xNeighbors
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SWITCH
6
158
109
191
TrueHex
TrueHex
0
1
-1000

@#$#@#$#@
## WHAT IS IT?

This demonstrates how to make a model that uses an undistorted hexagonal grid of cells.

## HOW IT WORKS

To achieve an undistorted hexagonal grid, the grid is not based on patches the same height as the cells (as in Uri Wilensky's "Hex Cells Example") because such a grid must be squashed vertically by a factor of 0.866 .  Instead we use pixel patches that are pixels (with `patch size = 1`) and create truly hexagonal cells (a breed of turtles) to form the grid.
  
`setup` resizes the World (according to the user parameters) and creates the hex grid. The user parameters are constained so that the hex grid wraps exactly, and the World origin is set to the bottom-left corner as there must be an even number of patches on each axis.

After `setup`, cells are used instead of patches (with the sole exception of inside the `xNeighbor-reporter`).  For example, `selected-cell-reporter` reports the cell under the mouse, with no reference to patches. In order to conform with this avoidance of using patches, navigation around the grid uses `xNeighbor-reporter` which reports the neighboring cell in the specified direction.  Use of this navigation is exemplified by the `walk` procedure in which walkers are centred on cells and navigate the grid by stepping to an immediate neighbor cell. They do a (mostly straight) random walk, bouncing off any non-wrapping boundary. The model allows for the World to have none-wrapping boundaries, and even for the grid to have holes in it (provided 'no cell is an island').

To demonstrate the use of manipilation of hexes, we define hex turtles to fit into cells exactly.  The Interface tab has mouse and "Action key" combinations to create them, select them, recolor them, drag and drop them between cells, and delete them.  Multiple hexes can be created on the same cell. Each new hex is created on top of the others and hides those below (until it is deleted or dragged away).

Only the `setup-hex-grid` procedure is dependent upon whether the grid geometry is truly hexagonal.  To demonstrate this the `TrueHex` switch allows the use of either undistorted or distorted hexagonal grids.

## RELATED MODELS

Uri Wilensky's "Hex Cells Example".  This uses the distorted hexagonal grid, also modeled here for comparison purposes.

## CREDITS AND REFERENCES

Partly based on Uri Wilensky's "Hex Cells Example" in the Models Library.

## COPYRIGHT AND LICENSE

<a rel="license" href="http://creativecommons.org/licenses/by/4.0/"><img alt="Creative Commons Licence" style="border-width:0" src="https://i.creativecommons.org/l/by/4.0/88x31.png" /></a><br /><span xmlns:dct="http://purl.org/dc/terms/" property="dct:title"></span> by <span xmlns:cc="http://creativecommons.org/ns#" property="cc:attributionName">Ian J Heath</span> is licensed under a Creative Commons Attribution 4.0 International License
http://creativecommons.org/licenses/by/4.0/
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

arrow half
true
0
Polygon -7500403 true true 105 30 150 0 195 30 105 30
Rectangle -7500403 true true 135 30 165 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

hex
false
0
Polygon -7500403 true true 0 150 75 37 225 37 300 150 225 263 75 263

hex-outline
false
0
Polygon -7500403 false true 0 150 75 37 225 37 300 150 225 263 75 263

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

truhex
false
0
Polygon -7500403 true true 0 150 75 20 225 20 300 150 225 280 75 280

truhex-outline
false
0
Polygon -7500403 false true 0 150 75 20 225 20 300 150 225 280 75 280

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
