globals                       ;; Setup global variables
[
  grid-size-x              ;; Number of grids of streets in x direction
  grid-size-y              ;; Number of grids of streets in y direction
  grid-x-inc               ;; the amount of patches in between two roads in the x direction
  grid-y-inc               ;; the amount of patches in between two roads in the y direction
  num-passengers           ;; number of passengers dropped off
  roads                    ;; agentset containing the patches that are roads
  watch-a-car-true?        ;; enables or disables watch a car feature
  chargers                 ;; agentset containing patches which are chargers
  gas_stations             ;; agentset containing patches which are gas_stations
  initial-manual-cost      ;; initial purchase cost for manual vehicles
  initial-abet-cost        ;; initial purchase cost for ABET vehicles
]


breed [abet a-abet]           ;; There is a turtle breed of ABETs
breed [manual a-manual]       ;; There is a turtle breed of manual vehicles


turtles-own                   ;; Properties for all turtles
[
  speed                    ;; the speed of the turtle
  destination              ;; the patch where the passenger wants to go
  passenger                ;; the patch where the passenger is
  goal                     ;; where am I currently headed
  trip-status              ;; trip status (0 = heading to passenger, 1 = heading to destination, 2 = arrived)
  travel-distance          ;; distance traveled by the vehicles
  paid-distance            ;; distance traveled with a passenger
  energy_level             ;; vehicles have an energy level (battery or fuel)
  recharge-cost            ;; cost for vehicle to recharge or refill gas
]


abet-own                      ;; Properties for ABETs
[
  charge                   ;; ABET's have a "charge goal"
  charging_state           ;; ABET's have a "charging state condition"
]


manual-own                    ;; Properties for Manual Vehicles
[
  refuel                   ;; Manual vehicles have a "refuel goal"
  refueling_state          ;; Manual vehicles have a "refueling state condition"
]


patches-own                   ;; Properties for patches
[
  my-row                   ;; the row of the intersection counting from the upper left corner of the world.  -1 for non-intersection patches.
  my-column                ;; the column of the intersection counting from the upper left corner of the world.  -1 for non-intersection patches.
]


to setup                      ;; On setup button, initialize model and create environment and agents

  clear-all
  setup-globals            ;; Setup global variables
  setup-patches            ;; ask the patches to draw themselves and set up a few variables

  if (num-manual + num-abet > count roads) [                   ;; warning message if too many cars are created to fit
    user-message (word "There are too many cars to fit on the road.")
    stop
  ]


  create-manual num-manual [                                                                      ;; Create manual vehicles
    set shape "car"                                                                               ;; sets turtle shape to be a car
    set color blue                                                                                ;; sets car color to blue
    set travel-distance 0                                                                         ;; sets initial travel-distance to 0
    set paid-distance 0                                                                           ;; sets initial paid-distance to 0
    move-to one-of roads with [ not any? turtles-on self ]                                        ;; Find a empty road and place the turtle there
    record-data
    set-passenger-destination
    set goal passenger                                                                            ;; sets goal to be passenger
    set energy_level random (max_manual_range - max_manual_range / 2 + 1) + max_manual_range / 2  ;; set the initial energy level between 50% and 100%
    set refueling_state 0                                                                         ;; sets the initial refueling_state to 0
    set initial-manual-cost manual-veh-price * num-manual                                         ;; calculates initial cost of purchasing manual vehicles
  ]

  create-abet num-abet [                                                                          ;; Create ABET vehicles
    set shape "car"                                                                               ;; sets turtle shape to be a car
    set color red                                                                                 ;; sets car color to red
    set travel-distance 0                                                                         ;; sets initial travel-distance to 0
    set paid-distance 0                                                                           ;; sets initial paid-distance to 0
    move-to one-of roads with [ not any? turtles-on self ]                                        ;; Find a empty road and place the turtle there
    record-data
    set-passenger-destination
    set goal passenger                                                                            ;; sets goal to be passenger
    set energy_level random (max_ABET_range - max_ABET_range / 2 + 1) + max_ABET_range / 2        ;; set the initial energy level between 50% and 100%
    set charging_state 0                                                                          ;; sets the initial charging_state to 0
    set initial-abet-cost abet-veh-price * num-abet                                               ;; calculates initial cost of purchasing ABET vehicles
  ]

  reset-ticks
end


to set-passenger-destination  ;; Function to create passengers and destinations

  set trip-status 0                                 ;; set trip to 0 to indicate heading to pickup passenger

  let goal-candidates patches with [                ;; Find all patches where there can be a be a passenger or destination (not on road)
    pcolor = 38 and any? neighbors with [ pcolor = white ] ]

  set passenger one-of goal-candidates              ;; choose at random a location for the passenger

  set destination one-of goal-candidates with [   ;; choose at random a location for the destination
    self != [ passenger ] of myself ]

  if watch-a-car-true? = 1  ;; if watching vehicle

  [update-labels]

end


to setup-globals              ;; Initialize the global variables to appropriate values
  set watch-a-car-true? 0
  set num-passengers 0
  set grid-size-x 9           ;; road grid-size-x for enviornment
  set grid-size-y 9           ;; road grid-size-y for enviornment
  set grid-x-inc world-width / grid-size-x
  set grid-y-inc world-height / grid-size-y
end


to setup-patches              ;; Set up the roads and charger/gas station agentsets


  ask patches [                                               ;; Initialize the patch-owned variables and color the patches to brown
    set my-row -1
    set my-column -1
    set pcolor brown + 3
  ]

  set roads patches with [                                    ;; Define road patches
    (floor ((pxcor + max-pxcor - floor (grid-x-inc - 1)) mod grid-x-inc) = 0) or
    (floor ((pycor + max-pycor) mod grid-y-inc) = 0)
  ]

  ask roads [ set pcolor white ]                              ;; Colors road patches white

  set chargers patches with [ pxcor = -14 and pycor = -14]    ;; Defines charger with this location (-14, -14)
  ask chargers [ set pcolor red ]                             ;; Colors chargers red

  set gas_stations patches with [ pxcor = 13 and pycor = 14]  ;; Defines gas stations with this location ( 13, 14)
  ask gas_stations [ set pcolor blue ]                        ;; Colors gas stations blue

end


to go                         ;; Runs the simulation

  ask abet [
    if charging_state = 0 [                ;; If ABET is NOT currently charging
      if energy_level <= 0 [               ;; ABET battery has died
        facexy 0 0                         ;; Face car toward 0 0
        fd 0                               ;; Don't move car (dead)
        record-data                        ;; Record data for plotting
      ]

      if energy_level > 0 [                ;; ABET has battery remaining
        face next-patch                    ;; Face car toward next goal
        fd car_speed                       ;; Move car forward at car_speed
        set energy_level energy_level - 1  ;; Battery Level decreases by set amount
        record-data                        ;; Record data for plotting
      ]

    ]

    if charging_state = 1 [               ;; If ABET is currently charging
      facexy 0 0                          ;; Face car toward 0 0
      fd 0                                ;; Don't move car (charging)
      set recharge-cost recharge-cost + (recharge-rate * electricity-cost)  ;; Accumulate recharge cost with electricity rate
      set energy_level energy_level + recharge-rate ;; Energy level increases at recharge rate
      record-data                         ;; Record data for plotting
    ]
  ]

  ask manual [
    if refueling_state = 0 [              ;; If manual vehicle is NOT currently refueling
      if energy_level <= 0 [              ;; Manual vehicle is out of fuel
        facexy 0 0                        ;; Face car toward 0 0
        fd 0                              ;; Don't move car (dead)
        record-data                       ;; Record data for plotting
      ]

      if energy_level > 0 [               ;; Manual vehicle has fuel remaining
        face next-patch                   ;; Face car toward next goal
        fd car_speed                      ;; Move car forward at car_speed
        set energy_level energy_level - 1 ;; Battery Level decreases
        record-data                       ;; Record data for plotting
      ]
    ]

    if refueling_state = 1 [              ;; If manual vehicle is currently refueling
      facexy 0 0                          ;; Face car toward 0 0
      fd 0                                ;; Don't move car (Refueling)
      set recharge-cost recharge-cost + (recharge-rate * 25 * gas-cost)  ;; Accumulate recharge cost with gas rate
      set energy_level energy_level + recharge-rate * 25 ;; Energy level increases at recharge rate assuming gas fillup is 25x faster than recharging
      record-data                         ;; Record data for plotting
    ]
  ]

  label-subject                           ;; If watching a car, label it
  tick

end


to record-data                ;; Data recorder for plots
  set travel-distance travel-distance + 1 ;; While moving, increase distance
  if trip-status = 1 [
    set paid-distance paid-distance + 1 ;; Wile moving with passenger onboard, increase paid distance
  ]
end


to-report next-patch          ;; Define goal of vehicle (passenger, destination, or charge/refuel)

  ask abet [

    if goal = passenger and (member? patch-here [ neighbors4 ] of passenger) and charging_state = 0 [      ;; If goal is passenger and nearby passenger
      set trip-status 1                                          ;; Passenger is onboard
      set goal destination                                       ;; Set goal to destination
    ]

    if goal = destination and (member? patch-here [ neighbors4 ] of destination) and charging_state = 0 [  ;; If goal is destination and nearby destination
      set trip-status 2                                          ;; Off-board passenger
      set num-passengers num-passengers + 1                      ;; Count total number of passengers transported

      set-passenger-destination                                  ;; Execute set-passenger-destination function
      let Passenger_X_Cor [ pxcor ] of passenger                 ;; Find X Coordinate of new passenger
      let Passenger_Y_Cor [ pycor ] of passenger                 ;; Find Y Coordinate of new passenger
      let Destination_X_Cor [ pxcor ] of destination             ;; Find X Coordinate of new destination
      let Destination_Y_Cor [ pycor ] of destination             ;; Find Y Coordinate of new destination
      let Charger_X_Cor [ pxcor ] of one-of chargers             ;; Find X Coordinate of one of chargers
      let Charger_Y_Cor [ pycor ] of one-of chargers             ;; Find Y Coordinate of one of chargers

      let required_range 1.5 * (                                 ;; Find required minimum range for next trip and add 50% safety factor
        abs(Passenger_X_Cor -  [ xcor ] of myself ) +
        abs(Destination_X_Cor - Passenger_X_Cor) +
        abs(Charger_X_Cor - Destination_X_Cor) +
        abs(Passenger_Y_Cor -  [ ycor ] of myself) +
        abs(Destination_Y_Cor - Passenger_Y_Cor) +
        abs(Charger_Y_Cor - Destination_Y_Cor) )

      if required_range < energy_level [set goal passenger]     ;; Find required range is less than energy level, assign passenger goal
      if required_range >= energy_level [                       ;; If required range is greater than energy_level
        set charge one-of chargers                              ;; Set a charger location
        set goal charge ]                                       ;; Set goal to charge
    ]

    if goal = charge and (member? patch-here [ neighbors ] of charge) and energy_level < max_ABET_range [  ;; If goal is to charge and not at full energy level and nearby charger
      set charging_state 1                                      ;; Set charging state to 1
      set goal charge                                           ;; Reset goal to charge
    ]

    if goal = charge and energy_level >= max_ABET_range [       ;; If goal and at full battery level
      set charging_state 0                                      ;; Set charging state to 0
      set goal passenger                                        ;; Set goal to passenger
    ]
  ]

  ask manual [

    if goal = passenger and (member? patch-here [ neighbors4 ] of passenger) and refueling_state = 0 [      ;; If goal is passenger and nearby passenger
      set trip-status 1                                        ;; Passenger is onboard
      set goal destination                                     ;; Set goal to destination
    ]

    if goal = destination and (member? patch-here [ neighbors4 ] of destination) and refueling_state = 0 [  ;; If goal is destination and nearby destination
      set trip-status 2                                        ;; Off-board passenger
      set num-passengers num-passengers + 1                    ;; Count total number of passengers transported

      set-passenger-destination                                ;; Execute set-passenger-destination function
      let Passenger_X_Cor [ pxcor ] of passenger               ;; Find X Coordinate of new passenger
      let Passenger_Y_Cor [ pycor ] of passenger               ;; Find Y Coordinate of new passenger
      let Destination_X_Cor [ pxcor ] of destination           ;; Find X Coordinate of new destination
      let Destination_Y_Cor [ pycor ] of destination           ;; Find Y Coordinate of new destination
      let Gas_Station_X_Cor [ pxcor ] of one-of gas_stations   ;; Find X Coordinate of one of gas stations
      let Gas_Station_Y_Cor [ pycor ] of one-of gas_stations   ;; Find Y Coordinate of one of gas stations

      let required_range 1.5 * (                               ;; Find required minimum range for next trip and add 50% safety factor
        abs(Passenger_X_Cor -  [ xcor ] of myself ) +
        abs(Destination_X_Cor - Passenger_X_Cor) +
        abs(Gas_Station_X_Cor - Destination_X_Cor) +
        abs(Passenger_Y_Cor -  [ ycor ] of myself) +
        abs(Destination_Y_Cor - Passenger_Y_Cor) +
        abs(Gas_Station_Y_Cor - Destination_Y_Cor) )

      if required_range < energy_level [set goal passenger]   ;; Find required range is less than energy level, assign passenger goal
      if required_range >= energy_level [                     ;; If required range is greater than energy_level
        set refuel one-of gas_stations                        ;; Set a charger location
        set goal refuel ]                                     ;; Set goal to charge
    ]

    if goal = refuel and (member? patch-here [ neighbors ] of refuel) and energy_level < max_manual_range [  ;; If goal is to refuel and not at energy level and nearby gas_station
      set refueling_state 1                                   ;; Set refueling state to 1
      set goal refuel                                         ;; Reset goal to refuel
    ]

    if goal = refuel and energy_level >= max_manual_range [
      set refueling_state 0                                   ;; Set refueling state to 0
      set goal passenger                                      ;; Set goal to passenger
    ]
  ]

  let choices neighbors with [ pcolor = white ]                  ;; Define choices agentset which constrains vehicle to only drive on road patches
  let choice min-one-of choices [ distance [ goal ] of myself ]  ;; Choose option which minimizes distance between goal and vehicle
  report choice                                                  ;; Report out chosen patch
end



to watch-a-car-button         ;; Watch a car button

  ifelse watch-a-car-true? = 0
  [set watch-a-car-true? 1
    watch-a-car
  ]
  [set watch-a-car-true? 0
    stop-watching
  ]

end


to watch-a-car                ;; Watch a car function

  stop-watching             ;; In case previously watching another car
  watch one-of turtles
  update-labels
end


to update-labels              ;; Update labels of watched car

  if subject != nobody [
    ask subject [

      ask passenger [
        set pcolor yellow          ;; Color the passenger patch yellow
        set plabel-color yellow    ;; Label the passenger in yellow font
        set plabel "passenger"     ;; Label the passenger
      ]

      ask destination [
        set pcolor orange          ;; Color the destination patch orange
        set plabel-color orange    ;; Label the destination in orange font
        set plabel "destination"   ;; Label the destination
      ]

      set label [ plabel ] of goal ;; car displays its goal
    ]
  ]
end


to stop-watching              ;; Stop watching a car

  ask patches with [ pcolor = yellow or pcolor = orange ] [   ;; Reset patch colors/labels
    stop-inspecting self
    set pcolor 38
    set plabel ""
  ]

  ask turtles [
    set label ""
    stop-inspecting self
  ]
  reset-perspective
end


to label-subject              ;; Label watched vehicle
  if subject != nobody [
    ask subject [
      set label energy_level  ;; Label watched vehicle with energy level
      set label-color black   ;; Color label black
    ]
  ]
end


;; Modified Version of Traffic Grid Goal Model by Uri Wilensky (2008).
;; Modified by David Caples, Cameron Culver
@#$#@#$#@
GRAPHICS-WINDOW
905
44
1553
693
-1
-1
17.3
1
15
1
1
1
0
1
1
1
-18
18
-18
18
1
1
1
ticks
30.0

SLIDER
15
215
185
248
num-manual
num-manual
0
20
10.0
1
1
NIL
HORIZONTAL

BUTTON
105
95
185
128
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
15
95
100
128
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
15
155
100
188
Watch Car
watch-a-car-button
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
105
155
185
188
Stop Watching
stop-watching
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
15
255
185
288
num-abet
num-abet
0
20
10.0
1
1
NIL
HORIZONTAL

MONITOR
645
45
775
90
Passengers Transported
num-passengers
0
1
11

PLOT
549
400
894
690
Total Fleet Distances
Time
Distance
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Paid Distance" 1.0 0 -14439633 true "\n" "plot sum [paid-distance] of turtles"
"Total Distance" 1.0 0 -14070903 true "" "plot sum [travel-distance] of turtles"

PLOT
195
93
540
393
Total Fleet Costs
Time
Costs ($)
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Recharge_Costs" 1.0 0 -5298144 true "" "plot sum [recharge-cost] of turtles"
"Operation_Costs" 1.0 0 -7500403 true "" "plot (operator-cost-per-hour / 60 * ticks * num-manual)"

SLIDER
15
295
185
328
max_ABET_range
max_ABET_range
40
500
600.0
20
1
NIL
HORIZONTAL

SLIDER
15
335
185
368
max_manual_range
max_manual_range
40
500
300.0
20
1
NIL
HORIZONTAL

MONITOR
550
45
640
90
Initial Fleet Cost
manual-veh-price * num-manual + abet-veh-price * num-abet
17
1
11

TEXTBOX
15
195
165
213
Model Inputs\n
11
0.0
1

TEXTBOX
20
135
170
153
View Vehicle Destinations
11
0.0
1

SLIDER
15
415
185
448
recharge-rate
recharge-rate
1
10
3.0
1
1
NIL
HORIZONTAL

SLIDER
15
375
185
408
car_speed
car_speed
.1
2
1.0
.1
1
NIL
HORIZONTAL

TEXTBOX
20
75
170
93
Model Controls
11
0.0
1

PLOT
550
95
895
395
Total Fleet Net Profits
Time
Profit ($)
0.0
10.0
0.0
-9.9999999E7
true
true
"" ""
PENS
"Profitability" 1.0 0 -16777216 true "" "plot sum [paid-distance] of turtles * ride-price-per-distance - (operator-cost-per-hour / 60 * ticks * num-manual ) - sum [recharge-cost] of turtles - initial-abet-cost - initial-manual-cost"

SLIDER
15
455
185
488
manual-veh-price
manual-veh-price
10000
100000
20000.0
1000
1
NIL
HORIZONTAL

SLIDER
15
495
185
528
abet-veh-price
abet-veh-price
20000
100000
40000.0
5000
1
NIL
HORIZONTAL

SLIDER
15
535
185
568
ride-price-per-distance
ride-price-per-distance
0
5
3.0
.1
1
NIL
HORIZONTAL

SLIDER
15
575
185
608
electricity-cost
electricity-cost
0
.05
0.033
.001
1
NIL
HORIZONTAL

SLIDER
15
655
185
688
gas-cost
gas-cost
0
1
0.1
.05
1
NIL
HORIZONTAL

SLIDER
15
615
185
648
operator-cost-per-hour
operator-cost-per-hour
0
25
13.5
.5
1
NIL
HORIZONTAL

MONITOR
780
45
895
90
Profit
sum [paid-distance] of turtles * ride-price-per-distance - (operator-cost-per-hour / 60 * ticks * num-manual ) - sum [recharge-cost] of turtles - initial-abet-cost - initial-manual-cost
0
1
11

PLOT
195
400
540
690
Total Fleet Revenue
Time
Revenue ($)
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Revenue" 1.0 0 -955883 true "" "plot sum [paid-distance] of turtles * ride-price-per-distance"

TEXTBOX
35
10
555
45
Autonomous Battery Electric Taxi ABM Model
25
0.0
1

TEXTBOX
80
45
495
86
AAE560 - David Caples, Cameron Culver, Bryan Jagielo, John Jess
14
0.0
1

TEXTBOX
1420
130
1570
156
Gas\nStation
11
0.0
1

TEXTBOX
990
625
1140
643
Charger
11
0.0
1

@#$#@#$#@
Overview: Model simulates a taxi service with a portfolio composed of manual operated and autonomous battery electric vehicles (ABETs). Each vehicle agent is goal based, seeking to navigate the roads to pickup passengers, and transport them to thier associated destination. Addtionally, each vehicle's internal energy reserve is depleted as it travels, requiring either recharging or refuling. 

Each vehicle agent is assigned a passenger with a random location, and a corresponding random destination, with a new passenger and destination being generated as soon as the previous passenger reaches their destination. Before a ride is accepted by the vehicle agent, the agent assesses the total travel distance for the trip, and decides if it has enough energy to safely complete the trip without becoming stranded. 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
true
0
Polygon -7500403 true true 180 15 164 21 144 39 135 60 132 74 106 87 84 97 63 115 50 141 50 165 60 225 150 285 165 285 225 285 225 15 180 15
Circle -16777216 true false 180 30 90
Circle -16777216 true false 180 180 90
Polygon -16777216 true false 80 138 78 168 135 166 135 91 105 106 96 111 89 120
Circle -7500403 true true 195 195 58
Circle -7500403 true true 195 47 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Taxi_Portfolio_Profitability_Manual_20_ABET_0_T5000" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>count turtles</metric>
    <metric>sum [recharge-cost] of turtles</metric>
    <metric>operator-cost-per-hour / 60 * ticks * num-manual</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance - (operator-cost-per-hour / 60 * ticks * num-manual ) - sum [recharge-cost] of turtles - initial-abet-cost - initial-manual-cost</metric>
    <metric>manual-veh-price * num-manual + abet-veh-price * num-abet</metric>
    <metric>num-passengers</metric>
    <enumeratedValueSet variable="gas-cost">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-abet">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ride-price-per-distance">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="manual-veh-price">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="abet-veh-price">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_ABET_range">
      <value value="200"/>
      <value value="400"/>
      <value value="600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-manual">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="electricity-cost">
      <value value="0.033"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_manual_range">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="operator-cost-per-hour">
      <value value="13.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="car_speed">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recharge-rate">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Taxi_Portfolio_Profitability_Manual_15_ABET_5_T5000" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>count turtles</metric>
    <metric>sum [recharge-cost] of turtles</metric>
    <metric>operator-cost-per-hour / 60 * ticks * num-manual</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance - (operator-cost-per-hour / 60 * ticks * num-manual ) - sum [recharge-cost] of turtles - initial-abet-cost - initial-manual-cost</metric>
    <metric>manual-veh-price * num-manual + abet-veh-price * num-abet</metric>
    <metric>num-passengers</metric>
    <enumeratedValueSet variable="gas-cost">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-abet">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ride-price-per-distance">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="manual-veh-price">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="abet-veh-price">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_ABET_range">
      <value value="200"/>
      <value value="400"/>
      <value value="600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-manual">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="electricity-cost">
      <value value="0.033"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_manual_range">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="operator-cost-per-hour">
      <value value="13.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="car_speed">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recharge-rate">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Taxi_Portfolio_Profitability_Manual_10_ABET_10_T5000" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="5000"/>
    <metric>count turtles</metric>
    <metric>sum [recharge-cost] of turtles</metric>
    <metric>operator-cost-per-hour / 60 * ticks * num-manual</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance - (operator-cost-per-hour / 60 * ticks * num-manual ) - sum [recharge-cost] of turtles - initial-abet-cost - initial-manual-cost</metric>
    <metric>manual-veh-price * num-manual + abet-veh-price * num-abet</metric>
    <metric>num-passengers</metric>
    <enumeratedValueSet variable="gas-cost">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-abet">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ride-price-per-distance">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="manual-veh-price">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="abet-veh-price">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_ABET_range">
      <value value="200"/>
      <value value="400"/>
      <value value="600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-manual">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="electricity-cost">
      <value value="0.033"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_manual_range">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="operator-cost-per-hour">
      <value value="13.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="car_speed">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recharge-rate">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Taxi_Portfolio_Profitability_Manual_3_ABET_7" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="30000"/>
    <metric>count turtles</metric>
    <metric>sum [recharge-cost] of turtles</metric>
    <metric>operator-cost-per-hour / 60 * ticks * num-manual</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance - (operator-cost-per-hour / 60 * ticks * num-manual ) - sum [recharge-cost] of turtles - initial-abet-cost - initial-manual-cost</metric>
    <metric>manual-veh-price * num-manual + abet-veh-price * num-abet</metric>
    <metric>num-passengers</metric>
    <enumeratedValueSet variable="gas-cost">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-abet">
      <value value="7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ride-price-per-distance">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="manual-veh-price">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="abet-veh-price">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_ABET_range">
      <value value="200"/>
      <value value="400"/>
      <value value="600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-manual">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="electricity-cost">
      <value value="0.033"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_manual_range">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="operator-cost-per-hour">
      <value value="13.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="car_speed">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recharge-rate">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Taxi_Portfolio_Profitability_Manual_0_ABET_10" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="30000"/>
    <metric>count turtles</metric>
    <metric>sum [recharge-cost] of turtles</metric>
    <metric>operator-cost-per-hour / 60 * ticks * num-manual</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance</metric>
    <metric>sum [paid-distance] of turtles * ride-price-per-distance - (operator-cost-per-hour / 60 * ticks * num-manual ) - sum [recharge-cost] of turtles - initial-abet-cost - initial-manual-cost</metric>
    <metric>manual-veh-price * num-manual + abet-veh-price * num-abet</metric>
    <metric>num-passengers</metric>
    <enumeratedValueSet variable="gas-cost">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-abet">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="ride-price-per-distance">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="manual-veh-price">
      <value value="20000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="abet-veh-price">
      <value value="40000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_ABET_range">
      <value value="200"/>
      <value value="400"/>
      <value value="600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-manual">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="electricity-cost">
      <value value="0.033"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max_manual_range">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="operator-cost-per-hour">
      <value value="13.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="car_speed">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="recharge-rate">
      <value value="3"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
