; 2 dimensional Cellular Automata: the parity model
; code based on the NetLogo Life model

globals [
    mouse-x     ;; record last position of mouse (for adding cells by hand)
    mouse-y
    steps       ;; number of steps (cycles/generations) completed
    ]

patches-own [
  on?            ;; indicates if the cell is on (black)
  on-neighbours  ;; counts how many neighboring cells are on
]

to setup         ;; start with all cells off (white)
  ask patches
    [ cell-off ]
  set steps 0
end

to cell-on
  set on? true
  set pcolor black
end

to cell-off
  set on? false
  set pcolor white
end

to go
  if mouse-down?
    [ stop ]  ;; wait for user to stop drawing
  ask patches
    [ set on-neighbours count neighbors4 with [on?] ]
  ask patches
    [ ifelse (on-neighbours mod 2 = 0)    ;; set cell on iff an odd number of neighbours are on
        [ cell-off ]
        [ cell-on ]
    ]
  set steps steps + 1
end

to add-cells    ;; observer procedure
;; if the patch under the mouse pointer is off, turn it on, and if it is on, turn it off
  if mouse-down? and not (mouse-x = mouse-xcor and mouse-y = mouse-ycor)
      ;; if the mouse button is down and has moved since last time
    [ ask patch mouse-xcor mouse-ycor
        [ ifelse on?
            [ cell-off ]
            [ cell-on ]
        ]
       set mouse-x mouse-xcor
       set mouse-y mouse-ycor
    ]
    wait 0.1
end

to add-central-patch
;; a one on cell in the middle
    setup
    ask patch 0 0 [ cell-on ]
end

to add-central-block
;; add a block of on cells in the middle
    setup
    ;; draw a rough circle in the middle
    ask patch 0 0 [
        ask patches in-radius 3
            [ cell-on]
        ]
    ;; remove the points to make a square
    ask patch  0  3 [ cell-off ]
    ask patch  3  0 [ cell-off ]
    ask patch  0 -3 [ cell-off ]
    ask patch -3  0 [ cell-off ]
end



;; written by Nigel Gilbert (n.gilbert@soc.surrey.ac.uk) July 2002
;; based loosely on a model with the following copyright:

; ***NetLogo Model Copyright Notice***

; This model was created as part of the project: CONNECTED MATHEMATICS:
; MAKING SENSE OF COMPLEX PHENOMENA THROUGH BUILDING OBJECT-BASED PARALLEL
; MODELS (OBPML).  The project gratefully acknowledges the support of the
; National Science Foundation (Applications of Advanced Technologies
; Program) -- grant numbers RED #9552950 and REC #9632612.

; Copyright 1998 by Uri Wilensky. All rights reserved.
; Converted from StarLogoT to NetLogo, 2001.

; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from Uri Wilensky.
; Contact Uri Wilensky for appropriate licenses for redistribution for
; profit.

@#$#@#$#@
GRAPHICS-WINDOW
321
10
663
373
41
41
4.0
1
10
1
1
1
0
1
1
1
-41
41
-41
41

BUTTON
46
42
127
75
Setup
setup
NIL
1
T
OBSERVER
T
NIL
NIL
T

BUTTON
176
43
257
76
Go
go
T
1
T
OBSERVER
T
NIL
NIL
T

BUTTON
47
194
177
227
Add 'On' cells
add-cells
T
1
T
OBSERVER
T
NIL
NIL
T

BUTTON
46
100
177
133
Add central patch
add-central-patch
NIL
1
T
OBSERVER
T
NIL
NIL
T

BUTTON
46
145
178
178
Add central block
add-central-block
NIL
1
T
OBSERVER
T
NIL
NIL
T

MONITOR
131
273
188
318
Steps
steps
0
1
11

@#$#@#$#@
WHAT IS IT?
-----------
This program is an example of a two-dimensional cellular automaton. A cellular automaton is a computational machine that performs actions based on certain rules.  It can be thought of as a board which is divided into cells (such as square cells of a checkerboard).  Each cell can be either on or off. This is called the "state" of the cell.  According to specified rules, each cell will be on (black) or off (white) at the next time step.

This particular cellular automaton is called the Parity Model. It is one of the simplest, yet the effects it produces are astonishingly complex.  There is only one rule, which all the cells follow.  The rule is that each cell looks at its four neighbours (the ones immediately to its left, right, above and below) and if an odd number of them (that is, 1 or 3) of them are 'on', it switches itself 'on'.  If there are an even number 'on' (2 or 4), it switches itself off.  [Spelling note: 'neighbour' is one of those words that is spelled differently in British and American English.  Since I'm English, I'm using the British spelling].

The rule is applied to all the cells in the grid during one time step.  Then the same rule is applied again (but some of the neighbour cells may have changed from on to off or off to on since last time, so the rule may not yield the same result).  This is repeated indefinitely.

The grid of cells is made endless by connecting the lefthand edge to the righthand edge and the top to the bottom.  This means that the cells 'below' the bottom row of cells are the cells on the top row, and the cells to the left of the left hand column are the right hand column of cells.

HOW TO USE IT
-------------
Press the SETUP button to clear the grid - all the cells turn white (off).

Next you need to seed the grid with a few cells that are on.  There are three possibilities:

a. Press the "Add central patch" button to turn one patch in the middle of the grid black, or

b. Press the "Add central block" button to turn a 7 by 7 square set of patches black, or

c. Press the "Add 'On' cells" button and then 'draw' the patches you want turned black by dragging the mouse over the grid.

Then press GO and watch what happens.

THINGS TO NOTICE
----------------
Notice that an initially very simple pattern (for example, with seed (a), one black patch in the middle) becomes a more and more complicated pattern and then suddenly reverts to multiple copies of the initial simple pattern.

After the pattern has reached the edge of the grid, it 'wraps' round to the opposite edge and starts advancing through the pattern that is being generated from the centre.  The consequence is that you get interference patterns (like the waves reflected off the end of a bath tub when you make a splash at the other end).  What is remarkable is that, despite the complexity of these interference patterns, they still resolve to very simple ones every now and again.

This is because the Parity rule has an important property. The regularity of the patterns you see is because the rule is an example of a linear rule: if two starting patterns are run in separate grids for a number of time steps and the resulting patterns are superimposed, this will be the same pattern as one gets if the starting patterns are run together on the same grid.

THINGS TO TRY
-------------
Try drawing your own starting patterns (seeds) with the "Add 'on' cells" button and see if you can find a seed that generates interesting patterns.

Starting from a regular pattern such as the "central patch" or "central block", see how many steps it takes before you get to a multiple of the starting pattern.  Is this number the same for all seeds?

EXTENDING THE MODEL
-------------------
1. the model defines the neighbourhood of a cell as the four cells above, below, to the left and to the right.  This is called the 'von Neumann' neighbourhood, after an early pioneer of computers.  It is also possible to define the neighbourhood of a cell as consisting of the eight cells surrounding a cell - the Moore neighbourhood.  You can try the model using the Moorre neighbourhood (as before, the cell becomes 'on' if an odd number of the 8 surrounding cells are 'on') by making a very small change to the code:
Find the lines in the 'to go' procedure:
  ask patches
    [ set on-neighbours count neighbors4 with [on?] ]
and change  "neighbors4" to "neighbours".  The former version counts how many of the 4 neighbours are 'on', and the new version counts how many of the 8 surrounding cells are 'on'. What happens?  Compare the patterns before and after making the change.  Remebering that the number of neighbours has changed from 4 to 8, is there anything noticeable about the form of the patterns?
2. Experiment with other rules.  One interesting rule is the 'majority' rule:

For each cell:
Count the number of neighbouring cells that are 'on' among the eight that surround the cell.  If the number is 5 or more, turn the cell on.  If 3 or less, turn it off.  If exactly 4, choose randomly whether to turn it on or off.

Can you prediict what kind of patterns you will see?

3. [Hard] Rewrite the program so that the grid, instead of wrapping round at the edges, extends to infinity  (of course, your computer will not accommodate a grid of infinite size, so you will have to find some way to simulate such a grid).


RELATED MODELS
--------------
Life
Rumor Mill
Cellular Automata 1D - a model that shows all 256 possible simple 1D cellular automata
CA 1D Totalistic - a model that shows all 2,187 possible 1D 3-color totalistic cellular automata
CA 1D Rule 30 - the basic rule 30 model
CA 1D Rule 30 Turtle - the basic rule 30 model implemented using turtles
CA 1D Rule 90 - the basic rule 90 model
CA 1D Rule 110 - the basic rule 110 model
CA 1D Rule 250 - the basic rule 250 model


CREDITS AND REFERENCES
----------------------
There is a large literature on cellular automata.

A recent and critically acclaimed book by one of the pioneers of cellular automata is: Stephen Wolfram (2002) A new kind of science.  Wolfram Media Incorporated.

A clear account of some simple but important Cellular Automata can be found in T. Toffoli and N. Margolus (1987) Cellular Automata Machines.  MIT Press.

The parity model is used as an illustration in Chapter 7 of Nigel Gilbert and Klaus G. Troitzsch (1999) Simulation for the Social Scientist.  Open University Press.


This version of the Partity Model was written by Nigel Gilbert (based on the Life model by Uri Wilensky) for the annual ZUMA summer school of Simulation for the Social Sciences, 2002.

Nigel Gilbert, University of Surrey, UK; n.gilbert@soc.surrey.ac.uk
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

ant
true
0
Polygon -7566196 true true 136 61 129 46 144 30 119 45 124 60 114 82 97 37 132 10 93 36 111 84 127 105 172 105 189 84 208 35 171 11 202 35 204 37 186 82 177 60 180 44 159 32 170 44 165 60
Polygon -7566196 true true 150 95 135 103 139 117 125 149 137 180 135 196 150 204 166 195 161 180 174 150 158 116 164 102
Polygon -7566196 true true 149 186 128 197 114 232 134 270 149 282 166 270 185 232 171 195 149 186
Polygon -7566196 true true 225 66 230 107 159 122 161 127 234 111 236 106
Polygon -7566196 true true 78 58 99 116 139 123 137 128 95 119
Polygon -7566196 true true 48 103 90 147 129 147 130 151 86 151
Polygon -7566196 true true 65 224 92 171 134 160 135 164 95 175
Polygon -7566196 true true 235 222 210 170 163 162 161 166 208 174
Polygon -7566196 true true 249 107 211 147 168 147 168 150 213 150

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bee
true
0
Polygon -256 true false 152 149 77 163 67 195 67 211 74 234 85 252 100 264 116 276 134 286 151 300 167 285 182 278 206 260 220 242 226 218 226 195 222 166
Polygon -16777216 true false 150 149 128 151 114 151 98 145 80 122 80 103 81 83 95 67 117 58 141 54 151 53 177 55 195 66 207 82 211 94 211 116 204 139 189 149 171 152
Polygon -7566196 true true 151 54 119 59 96 60 81 50 78 39 87 25 103 18 115 23 121 13 150 1 180 14 189 23 197 17 210 19 222 30 222 44 212 57 192 58
Polygon -16777216 true false 70 185 74 171 223 172 224 186
Polygon -16777216 true false 67 211 71 226 224 226 225 211 67 211
Polygon -16777216 true false 91 257 106 269 195 269 211 255
Line -1 false 144 100 70 87
Line -1 false 70 87 45 87
Line -1 false 45 86 26 97
Line -1 false 26 96 22 115
Line -1 false 22 115 25 130
Line -1 false 26 131 37 141
Line -1 false 37 141 55 144
Line -1 false 55 143 143 101
Line -1 false 141 100 227 138
Line -1 false 227 138 241 137
Line -1 false 241 137 249 129
Line -1 false 249 129 254 110
Line -1 false 253 108 248 97
Line -1 false 249 95 235 82
Line -1 false 235 82 144 100

bird1
false
0
Polygon -7566196 true true 2 6 2 39 270 298 297 298 299 271 187 160 279 75 276 22 100 67 31 0

bird2
false
0
Polygon -7566196 true true 2 4 33 4 298 270 298 298 272 298 155 184 117 289 61 295 61 105 0 43

boat1
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6524078 true false 150 32 157 162
Polygon -16776961 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7566196 true true 158 33 230 157 182 150 169 151 157 156
Polygon -7566196 true true 149 55 88 143 103 139 111 136 117 139 126 145 130 147 139 147 146 146 149 55

boat2
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6524078 true false 150 32 157 162
Polygon -16776961 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7566196 true true 157 54 175 79 174 96 185 102 178 112 194 124 196 131 190 139 192 146 211 151 216 154 157 154
Polygon -7566196 true true 150 74 146 91 139 99 143 114 141 123 137 126 131 129 132 139 142 136 126 142 119 147 148 147

boat3
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6524078 true false 150 32 157 162
Polygon -16776961 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7566196 true true 158 37 172 45 188 59 202 79 217 109 220 130 218 147 204 156 158 156 161 142 170 123 170 102 169 88 165 62
Polygon -7566196 true true 149 66 142 78 139 96 141 111 146 139 148 147 110 147 113 131 118 106 126 71

box
true
0
Polygon -7566196 true true 45 255 255 255 255 45 45 45

butterfly1
true
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8

circle
false
0
Circle -7566196 true true 35 35 230

person
false
0
Circle -7566196 true true 155 20 63
Rectangle -7566196 true true 158 79 217 164
Polygon -7566196 true true 158 81 110 129 131 143 158 109 165 110
Polygon -7566196 true true 216 83 267 123 248 143 215 107
Polygon -7566196 true true 167 163 145 234 183 234 183 163
Polygon -7566196 true true 195 163 195 233 227 233 206 159

spacecraft
true
0
Polygon -7566196 true true 150 0 180 135 255 255 225 240 150 180 75 240 45 255 120 135

thin-arrow
true
0
Polygon -7566196 true true 150 0 0 150 120 150 120 293 180 293 180 150 300 150

truck-down
false
0
Polygon -7566196 true true 225 30 225 270 120 270 105 210 60 180 45 30 105 60 105 30
Polygon -8716033 true false 195 75 195 120 240 120 240 75
Polygon -8716033 true false 195 225 195 180 240 180 240 225

truck-left
false
0
Polygon -7566196 true true 120 135 225 135 225 210 75 210 75 165 105 165
Polygon -8716033 true false 90 210 105 225 120 210
Polygon -8716033 true false 180 210 195 225 210 210

truck-right
false
0
Polygon -7566196 true true 180 135 75 135 75 210 225 210 225 165 195 165
Polygon -8716033 true false 210 210 195 225 180 210
Polygon -8716033 true false 120 210 105 225 90 210

turtle
true
0
Polygon -7566196 true true 138 75 162 75 165 105 225 105 225 142 195 135 195 187 225 195 225 225 195 217 195 202 105 202 105 217 75 225 75 195 105 187 105 135 75 142 75 105 135 105

wolf-left
false
3
Polygon -6524078 true true 117 97 91 74 66 74 60 85 36 85 38 92 44 97 62 97 81 117 84 134 92 147 109 152 136 144 174 144 174 103 143 103 134 97
Polygon -6524078 true true 87 80 79 55 76 79
Polygon -6524078 true true 81 75 70 58 73 82
Polygon -6524078 true true 99 131 76 152 76 163 96 182 104 182 109 173 102 167 99 173 87 159 104 140
Polygon -6524078 true true 107 138 107 186 98 190 99 196 112 196 115 190
Polygon -6524078 true true 116 140 114 189 105 137
Rectangle -6524078 true true 109 150 114 192
Rectangle -6524078 true true 111 143 116 191
Polygon -6524078 true true 168 106 184 98 205 98 218 115 218 137 186 164 196 176 195 194 178 195 178 183 188 183 169 164 173 144
Polygon -6524078 true true 207 140 200 163 206 175 207 192 193 189 192 177 198 176 185 150
Polygon -6524078 true true 214 134 203 168 192 148
Polygon -6524078 true true 204 151 203 176 193 148
Polygon -6524078 true true 207 103 221 98 236 101 243 115 243 128 256 142 239 143 233 133 225 115 214 114

wolf-right
false
3
Polygon -6524078 true true 170 127 200 93 231 93 237 103 262 103 261 113 253 119 231 119 215 143 213 160 208 173 189 187 169 190 154 190 126 180 106 171 72 171 73 126 122 126 144 123 159 123
Polygon -6524078 true true 201 99 214 69 215 99
Polygon -6524078 true true 207 98 223 71 220 101
Polygon -6524078 true true 184 172 189 234 203 238 203 246 187 247 180 239 171 180
Polygon -6524078 true true 197 174 204 220 218 224 219 234 201 232 195 225 179 179
Polygon -6524078 true true 78 167 95 187 95 208 79 220 92 234 98 235 100 249 81 246 76 241 61 212 65 195 52 170 45 150 44 128 55 121 69 121 81 135
Polygon -6524078 true true 48 143 58 141
Polygon -6524078 true true 46 136 68 137
Polygon -6524078 true true 45 129 35 142 37 159 53 192 47 210 62 238 80 237
Line -16777216 false 74 237 59 213
Line -16777216 false 59 213 59 212
Line -16777216 false 58 211 67 192
Polygon -6524078 true true 38 138 66 149
Polygon -6524078 true true 46 128 33 120 21 118 11 123 3 138 5 160 13 178 9 192 0 199 20 196 25 179 24 161 25 148 45 140
Polygon -6524078 true true 67 122 96 126 63 144

@#$#@#$#@
NetLogo 4.0pre5
@#$#@#$#@
setup
add-central-block
repeat 90 [ go ]
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
