globals ;; used to track the three size classes
[red-count
 blue-count
 yellow-count
]

breed [
  fishes fish]

fishes-own
[total-length     ;; length or size of fish
 density          ;; number of neighbors within a radius
 fish-age         ;; age of fish at each tick
]

to setup
  ca
  set-default-shape fishes "fish"
  create-fishes 1000
  [setxy random-xcor random-ycor
  set total-length 0
  set fish-age 0
  set color yellow
  ]
  reset-ticks
end

to go
  if not any? fishes [user-message "All Fish Die" stop] ;; stops the model when all fish die
  if count fishes > 10000 [user-message "Population Explosion Model Stops" stop] ;; stops the model when population bigger than a potential carrying capacity
  ask fishes
  [
    move             ;;spreads fish out and creates density parameters for growing season and idealized as occuring spring/summer
    reproduce        ;;idealized as occurring that spring
    determine-grow   ;;fish don't grow if they are at their maximum size
    identify-density ;;quanitfies how many fish there are around the individual fish
    size-class       ;;based on the fishes' total-length a size-class is assigned (juvenile, sub-adult, adult), this would be assigned as the size-class in the fall
    age              ;;fish ages a year and in the fish world the fish is the next year older starting January 1
    death            ;;occurs as a probability or after it has reached its oldest possible age
  ]
  tick
end

to death;;;;
  if fish-age = 15 [die]
  if color = yellow [if random-float 100 < 75 [die]]                              ;; mortality rate of young of year (value has not been validated)
  if color = red [if random-float 100 < 40 [die]]                                 ;; mortality rate of juveniles
  if color = blue [if random-float 100 < adult-mortality [die]]                   ;; mortality rate of adults set by user

end

to reproduce
  if total-length > 300
  [if random-float 100 < 50          ;; assumes a 1:1 male female ratio so half of population (i.e., females) would have the ability to reproduce
    [hatch 5                       ;; number of young of year produced by each female (value has not been validated)
      [rt random-float 360 fd 5      ;; randomly moves the hatched fish
       set total-length 10           ;; young of year size set to 10 mm
       set color yellow              ;; young of year color set to yellow
       set size 1                    ;; set pixel size
       set fish-age 0]                    ;; all reproduced fish are 0 years of age
      ]
  ]
end

to move                ;; fish move randomly within environment as fish would normally do and this changes the density in which they find themselves
  rt random 0.5
  lt random 0.5
  fd random 10
end

to determine-grow
  if total-length < 810 [grow]      ;; fish stop growing when the reach 810 mm in length
end

to identify-density                 ;; counts the number of fish including self within the surrounding 10 patches
  set density count other fishes in-radius 10 with [color = blue or color = red]
end

to grow
  if density < Density-threshold [set total-length 810 * (1 - exp (-0.089 * (fish-age - (-2.378))))]                                  ;;Density-threshold set by user
  if density > Density-threshold [set total-length 810 * (1 - exp (-(K-coeff) * (fish-age - (-2.378))))]                ;;K coefficient set by user
end

to size-class                      ;; sets size class color based on size/total-length
  if total-length >= 300 [set color blue]
  if total-length >= 300 [set size 2]
  if total-length < 300 and total-length >= 150 [set color red]
  if total-length < 300 and total-length >= 150 [set size 1.5]
  if total-length < 150 [set color yellow]
  if total-length < 150 [set size 0.5]
end



to age
  set fish-age fish-age + 1
end
@#$#@#$#@
GRAPHICS-WINDOW
370
45
953
629
-1
-1
17.42424242424243
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
25
145
88
178
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
170
145
233
178
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
26
218
232
251
Density-threshold
Density-threshold
0
500
150.0
50
1
NIL
HORIZONTAL

SLIDER
25
269
230
302
K-coeff
K-coeff
0
0.1
0.05
0.01
1
NIL
HORIZONTAL

SLIDER
25
70
235
103
Adult-mortality
Adult-mortality
0
100
30.0
5
1
%
HORIZONTAL

MONITOR
965
660
1085
705
Total Adults
count turtles with [color = blue]
17
1
11

MONITOR
1105
660
1225
705
Total Juveniles
count turtles with [color = red]
17
1
11

MONITOR
1240
660
1362
705
Total  Young of Year
count turtles with [color = blue]
17
1
11

PLOT
965
45
1360
355
Plot1
Time
Population Total
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Adults" 1.0 0 -13345367 true "" "plot count turtles with [color = blue]"
"Juveniles" 1.0 0 -2674135 true "" "plot count turtles with [color = red]"
"Young of year" 1.0 0 -1184463 true "" "plot count turtles with [color = yellow]"

PLOT
965
365
1360
650
Plot2
Time
Population Total
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Adults" 1.0 0 -13345367 true "" "plot count turtles with [color = blue]"
"Juveniles" 1.0 0 -2674135 true "" "plot count turtles with [color = red]"

PLOT
30
320
365
630
Average Density
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot (sum [density] of turtles)/ (count turtles with [color = blue or color = red])"

@#$#@#$#@
## WHAT IS IT?

This model explores the effect of density on growth rates and age class populations. Each tick is analagous to a year. Fish increase in size (i.e,.total length) each year, which places the fish into a size class (adult, juvenile, or young of year). Adults can reproduce annually and when they do, they produce young of year fish. ADULT-MORTALITY is defined by the user except when an adult reaches 15 years of age, they die of old age. Juvenile and young of year mortality is set within the code.

## HOW IT WORKS

The program was developed to explore two main variables. 

The first vairable is related to density dependent growth effects on populations of population size.  To explore this, the user defines "high density" and then ascribes a growth rate once that density has been exceeded. 

The model counts the number of fish within a radius of 10 patches. If this value exceeds the DEFINE-HIGH-DENSITY then the DENSITY-DEPENDENT-GROWTH-RATE is used. The fish then increases in total length by that value. When user defined density is not exceeded (i.e., a low density environment) a fish will grow based on a Von Bertalanffy growth equation that is specific to Channel Catfish from the San Juan River (Pennock et al. 2018).

The second variable that can be explored, is changing the ADULT-MORTALITY rate.  This rate can be kept constant throughout a run or changed while the model is running to assess how mortality rates effect the overall population and age class distribution.

The two plots, track the number of fish in each age class overtime. The second plot does not include the young of year totals. The number of fish in each age-class is also enumerated in monitor boxes.

## HOW TO USE IT

SETUP produces 1000 young of year fish
GO causes each fish to:
-move
-adults reproduce based on a probability, 
-each individual identifies the number of fish within its vicinity
-each individual increases in size based on whether there is a low or high number of fish within the individual's vicinity
-each individual has a probability of dying
-each individual ages one year for each tick. 

The model will halt when the population dies out or when it exceeds 10,000 individuals.

## THINGS TO NOTICE

The population is highly sensitive to ADULT-MORTALITY and values greater than ~35% tend to crash the population.

## THINGS TO TRY

Find the adult mortality rates that result in an increasing population, stability, or crash it.  Then figure out the effect of density dependent growth rates on these patterns.

Set the model running and then change DEFINE-HIGH-DENSITY and see what crashes the population, what causes it to explode, and what results in stability.  Do the same for changes in the DENSITY-DEPENDENT-GROWTH-RATE.

## EXTENDING THE MODEL

-Reduce adult mortality as adult density varies (i.e., make it harder for fishermen to remove adults from the populaiton when there are less adults to find)
-Add code so  10% of age 3, 50% of age 4, and 75% of >= age 5 female fish reproduce
-Add a function of fecundity and size 
example: log10 (fecundity) = [log10(total-length) x 2.443] - 2.318
-Recode to run the fish in parallel like the termites model

## NETLOGO FEATURES

I used "set density count other fishes in-radius 10" and there may be better ways to determine the density of fish around a given individual

## RELATED MODELS

Wolf Sheep Predation model.

## CREDITS AND REFERENCES

Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

Pennock, C. A., Durst, S. L., Duran, B. R., Hines, B. A., Cathcart, C. N., Davis, J. E., ... & Franssen, N. R. (2018). Predicted and observed responses of a nonnative Channel Catfish population following managed removal to aid the recovery of endangered fishes. North American Journal of Fisheries Management, 38(3), 565-578.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>count turtles with [color = blue]</metric>
    <steppedValueSet variable="Adult-mortality" first="10" step="5" last="25"/>
    <steppedValueSet variable="Percent-growth" first="0.1" step="0.1" last="0.9"/>
    <steppedValueSet variable="Density-threshold" first="50" step="50" last="500"/>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
