breed [star]

breed [planet]

turtles-own
[ fx     ;; x-component of force vector
  fy     ;; y-component of force vector
  vx     ;; x-component of velocity vector
  vy     ;; y-component of velocity vector
  xc     ;; real x-coordinate (in case particle leaves world)
  yc     ;; real y-coordinate (in case particle leaves world)
  r-sqrd ;; square of the distance to the mouse
  mass   ;; mass of object
]

patches-own [fade]

globals
[ star-xc  ;; x-coordinate of acting mass
  star-yc  ;; y-coordinate of acting mass
  g     ;; Gravitational Constant to slow the acceleration
]

;--------------------------------------------
;--------------------------------------------

to setup
  ca
  reset-ticks
  set g gravitational-constant
  
  create-star 1 [ ; central acting mass
    set size 15
    set shape "sun"
    set color yellow
    set star-xc xcor
    set star-yc ycor]
  
  if planet1? [ ; if allowed, create planet 1
    create-planet 1 [
      set size 7
      set shape "circle"
      set color blue
      setxy 0 (0 - initial-radius1)
      set vx initial-planet-x-velocity1
      set vy initial-planet-y-velocity1
      set xc xcor
      set yc ycor
      set mass mass1]]
  
  if planet2? [ ; if allowed, create planet 2
    create-planet 1 [
      set size 6
      set shape "circle"
      set color red
      setxy 0 (0 - initial-radius2)
      set vx initial-planet-x-velocity2
      set vy initial-planet-y-velocity2
      set xc xcor
      set yc ycor
      set mass mass2]]
  
  if extra-objects? [ ; if allowed, create extra objects
    create-planet extra-objects [
      set size 3
      set shape "circle"
      set color 9 - random-float 6 ; grayish
      setxy 0 0
      fd random 100 ; like setxy random-xcor random-ycor, but limited to 100 away from sun
      ifelse random 2 = 0 ; random positive/negative x-velocity (-0.5 < vx < 0.5)
        [set vx random-float .5]
        [set vx random-float -.5]
      ifelse random 2 = 0 ; random positive/negative y-velocity (-0.5 < vy < 0.5)
        [set vy random-float .5]
        [set vy random-float -.5]
      set xc xcor
      set yc ycor
      set mass extra-objects-mass]]
end

;--------------------------------------------
;--------------------------------------------

to go
  if count planet = 0 [stop]; if no objects orbiting, stop
  ask planet [
    orbit
    let planetcolor color
    if trail? [ask patches in-radius .5 [set pcolor planetcolor set fade time-till-decay]] ; if trail? true, create line around object that can decay, unlike pd
    if die-if-too-close-to-star? and any? star in-radius 5 [die] ; if too close to star and variable true, die
    if ycor = -100 [print "hi"]
  ]
  ask patches [; makes orbit trail decay if decay variable true
    ifelse trail?
      [if fade > 0 and orbit-trail-decay? [
        set fade fade - 1
      if fade = 0 [
        set pcolor black]]]
      [if pcolor != black [set pcolor black]]
  ]
  tick
end

to orbit ;; Turtle Procedure
  update-force
  update-velocity
  update-position
end

to update-force ;; Turtle Procedure
  ;; Similar to 'distancexy', except using an unbounded plane so it's possible to keep track of turtle when goes offscreen.
  set r-sqrd (((xc - star-xc) * (xc - star-xc)) + ((yc - star-yc) * (yc - star-yc)))

  ;; prevents divide by zero
  ifelse (r-sqrd != 0)
  [
    ;; Calculate component forces using inverse square law
    set fx ((cos (atan (star-yc - yc) (star-xc - xc))) * (mass / r-sqrd))
    set fy ((sin (atan (star-yc - yc) (star-xc - xc))) * (mass / r-sqrd))
  ]
  [
    ;; if r-sqrd = 0, then it's at the mass, thus there's no force.
    set fx 0
    set fy 0
  ]
end

to update-velocity ;; Turtle Procedure
  ;; Now we update each particle's velocity, by taking the old velocity and
  ;; adding the force to it.
  set vx (vx + (fx * g))
  set vy (vy + (fy * g))
end 

to update-position ;; Turtle Procedure
  set xc (xc + vx) ; update where turtle should go to
  set yc (yc + vy)
  
  ifelse patch-at (xc - xcor) (yc - ycor) != nobody ; if patch exists,
  [
    setxy xc yc ; go to patch
    show-turtle
   ]
  [ hide-turtle ] ; hide turtle but keep working
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
822
643
150
150
2.0
1
10
1
1
1
0
0
0
1
-150
150
-150
150
0
0
1
ticks
30.0

BUTTON
31
12
94
45
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
115
11
178
44
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
833
95
1005
128
mass1
mass1
0
100
50
.1
1
NIL
HORIZONTAL

SLIDER
831
137
1018
170
initial-planet-x-velocity1
initial-planet-x-velocity1
-2
2
0.5
.01
1
NIL
HORIZONTAL

SLIDER
833
53
1005
86
initial-radius1
initial-radius1
0
125
100
1
1
NIL
HORIZONTAL

SLIDER
21
127
193
160
gravitational-constant
gravitational-constant
0
2
0.5
.1
1
NIL
HORIZONTAL

SLIDER
832
180
1019
213
initial-planet-y-velocity1
initial-planet-y-velocity1
-2
2
0
.01
1
NIL
HORIZONTAL

SWITCH
23
88
192
121
die-if-too-close-to-star?
die-if-too-close-to-star?
0
1
-1000

TEXTBOX
79
64
152
82
All Objects
11
0.0
1

TEXTBOX
844
28
994
46
Planet 1
11
0.0
1

TEXTBOX
89
237
239
255
Oribt Trail
11
0.0
1

SWITCH
36
291
186
324
orbit-trail-decay?
orbit-trail-decay?
0
1
-1000

TEXTBOX
841
228
991
246
Planet 2
11
0.0
1

SLIDER
837
254
1009
287
initial-radius2
initial-radius2
0
125
70
1
1
NIL
HORIZONTAL

SLIDER
838
297
1010
330
mass2
mass2
0
100
32
.1
1
NIL
HORIZONTAL

SLIDER
833
337
1015
370
initial-planet-x-velocity2
initial-planet-x-velocity2
-2
2
-0.4
.01
1
NIL
HORIZONTAL

SLIDER
834
377
1016
410
initial-planet-y-velocity2
initial-planet-y-velocity2
-2
2
-0.25
.01
1
NIL
HORIZONTAL

SWITCH
897
218
1027
251
planet2?
planet2?
0
1
-1000

TEXTBOX
838
425
988
443
Extra Objects
11
0.0
1

SLIDER
839
452
1011
485
extra-objects
extra-objects
0
100
5
1
1
NIL
HORIZONTAL

SLIDER
838
489
1010
522
extra-objects-mass
extra-objects-mass
0
100
25
1
1
NIL
HORIZONTAL

SWITCH
912
416
1034
449
extra-objects?
extra-objects?
0
1
-1000

SWITCH
893
16
1029
49
planet1?
planet1?
0
1
-1000

SWITCH
57
256
160
289
trail?
trail?
0
1
-1000

SLIDER
23
328
195
361
time-till-decay
time-till-decay
1
1000
400
1
1
NIL
HORIZONTAL

@#$#@#$#@
# Orbiting Bodies

### Made by Luke Elissiry using parts from Uri Wilensky's Gravitation model.

## WHAT IS IT?

This model attempts to simulate an aritificial star system where the only central acting mass is the central star. This project displays the common natural phenomenon expressed by the inverse-square law. Essentially this model displays what happens when the strength of the force between two objects varies inversely with the square of the distance between these two objects.

## HOW IT WORKS

In this model the formula used to guide each object's behavior is the standard formula for the Law of Gravitational Attraction:

>(m1 * m2 * G) / r2

This is a single force 'n-body' model, where we have a certain number of small particles, and one large acting mass (the star). The force is entirely one-way: the large mass remains unaffected by the smaller particles around it. And the smaller particles remain unaffected by each other as well. (Note that this is purely for purposes of simulation. In the real world, a force such as gravity acts on all bodies around it.)

Gravity is the best example of such a force. You can watch the particles form elliptic orbits around the star, or watch them slingshot around it, similar to how a comet streaks past our sun. Think of the individual objects as planets or other solar bodies, and see how they react to various masses that move or remain stationary.

## HOW TO USE IT

For a basic working model, press the SETUP button and the GO button.
For different settings, adjust the following before SETUP:

  - ALL OBJECTS section: all objects share these variables
    - DIE-IF-TOO-CLOSE-TO-STAR? - if an object is within 5 patches of sun, it dies
    - GRAVITATIONAL-CONSTANT - universal constant to control the acceleration


  - ORIBT TRAIL section: for the objects's trails
    - TRAIL? - if true, objects leave a trail of where they've been
    - ORBIT-TRAIL-DECAY? - if trail? true and this true, objects's trails decay over time
    - TIME-TILL-DECAY - if if trail? true and orbit-trail-decay? true, objects's trails decay over amount of time specified


  - PLANET 1 and PLANET 2 section: variables only these objects have
    - PLANET_? - turns planet on/off
    - INITIAL-RADIUS_ - how far from star planet will spawn
    - MASS_ - mass of planet
    - INITIAL-PLANET-X-VELOCITY_ and INITIAL-PLANET-Y-VELOCITY_ - the objects initial velocity


  - EXTRA-OBJECTS section: variables only the additional "asteroids" have (many asteroid variables are random such as: location, initial velocities, and color)
    - EXTRA-OBJECTS - number of additional bodies
    - EXTRA-OBJECTS-MASS - mass of all additional bodies


## THINGS TO NOTICE

The most important thing to observe is the behavior of the objects. Notice the objects's orbits around the star. TRAIL? on is recommended.

## THINGS TO TRY

- Add in more extra bodies
- Change masses and initial radii
- Change velocities
- Change the gravitational constant

## EXTENDING THE MODEL

- Add additional planets to control (easy)
- Have planets affect eachother (difficult)
- Have everything affect eachother, like the [N-Bodies] (http://modelingcommons.org/browse/one_model/1332#model_tabs_browse_applet) model, but try to make more usable/stable (difficult)

## NETLOGO FEATURES

This model creates the illusion of a plane of infinite size, to better model the behavior of the particles. Notice that with path marking you can see most of the ellipse a particle draws, even though the particle periodically shoots out of bounds. This is done through a combination of the basic turtle primitives `hide-turtle` and `show-turtle`, keeping every turtle's true coordinates as special turtle variables `xc` and `yc`, and calculations similar to the `distance` primitive but using `xc` and `yc` instead of `xcor` and `ycor`.

When you examine the procedure window, take note that standard turtle commands like `set heading` and `fd 1` aren't used here. Everything is done directly to the x-coordinates and y-coordinates of the turtles.

pd does not have the ability to fade, so workaround was to have the patches under object change color and have the patches own a cooldown time.

## RELATED MODELS

- [Gravitation] (http://modelingcommons.org/browse/one_model/1330#model_tabs_browse_applet) by Uri Wilensky - Central body is mouse, hard to see orbit, and few controls

- [N-Bodies] (http://modelingcommons.org/browse/one_model/1332#model_tabs_browse_applet) by Uri Wilensky - more accurate without central acting mass, but difficult to use and regular setup works for only 100 ticks

## CREDITS AND REFERENCES

Code for the orbit procedure and information was taken from Uri Wilenskys [Gravitation] (http://modelingcommons.org/browse/one_model/1330#model_tabs_browse_applet) model.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

sun
false
0
Circle -7500403 true true 75 75 150
Polygon -7500403 true true 300 150 240 120 240 180
Polygon -7500403 true true 150 0 120 60 180 60
Polygon -7500403 true true 150 300 120 240 180 240
Polygon -7500403 true true 0 150 60 120 60 180
Polygon -7500403 true true 60 195 105 240 45 255
Polygon -7500403 true true 60 105 105 60 45 45
Polygon -7500403 true true 195 60 240 105 255 45
Polygon -7500403 true true 240 195 195 240 255 255

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
