;;-----------------------------------------------------------------------------|
;; SECTION A – AUTHOR IDENTIFICATION AND CODE ABSTRACT
;;-----------------------------------------------------------------------------|
;;
;; File Name: TpLab_V2.07.nlogo
;; By Orrery Software
;; Dated: 2017-03-13
;; Author contact:
;;   Garvin H Boyle
;;   orrery@rogers.com
;;   http://orrery-software.webs.com

;; As the author, I welcome questions, discussion of issues and suggestions
;;   for improvements.

;;-----------------------------------------------------------------------------|
;; This TpLab app is a laboratory in which students can study aspects 
;;   of the phenomenon of Teleological Pruning (TP) as described in my 
;;   associated diary notes.

;; In this model, nrg arrives in a steady stream from the Sun and is captured
;;   in plants that produce fruit, which appears randomly in dales in the 
;;   forest.  Nomadic seekers roaming the forest according to their own 
;;   heuristic strategy seek the dales currently having food, and harvest it.
;;   The strategies start as ineffective bland heuristics, and evolve to be 
;;   much more sophisticated. In the most simple scenario, all actions are 
;;   instinctive.

;; But an ability to have beliefs influence behaviour is built into the genes,
;;   and belief systems evolve within the society.  These teleological belief
;;   systems can over-rule instinctual behaviour and stubbornly resist the
;;   need to change.  What happens to seekers that hold beliefs too tightly?
;;   Can the seekers of fruit become seekers of wisdom?  What happens to 
;;   societies that share a dilectic across multiple belief systems?

;;-----------------------------------------------------------------------------|
;; SECTION B – INITIAL DECLARATIONS OF GLOBALS AND BREEDS
;;-----------------------------------------------------------------------------|
;;
;;-----------------------------------------------------------------------------|
;; This program was developed on NetLogo Version 5.0.5
;;

;;-----------------------------------------------------------------------------|
;; code-determined global variables
globals
[
  ;; The version should be coded in this global variable to be included in
  ;;   output files.
  gs-Version
  
  ;; Note: Some global variables are declared inside of switches, sliders and 
  ;;   choosers when the interface is constructed and are not declared here.
  ;;   For the sake of clarity and completeness, they are noted here.
  
  ;; There are several uses of global variables:
  ;;  - Toggles (switches), and choosers which enable or disable features;
  ;;  - Numbers (in variables or sliders) which act as parameters;
  ;;  - Numbers (in variables) which collect data.
  ;;
  ;; Those marked as 'native Boolean' have values of true or false.
  ;; Those marked as 'numeric Boolean' have values of 1 or 0.
   
  ;;---------------------
  ;; MODELING ENVIRONMENT
  ;;---------------------
  
  ;; Assumed “Model Settings” on startup
  ;; horizontal wrap: on
  ;; vertical wrap: on
  ;; location of origin: centre
  ;; patch size: 5 pixels
  ;;-------------------------------------------------------------------------|
  ;; Implicit global variables due to model settings – patch locations
  ;; min-pxcor  -40
  ;; max-pxcor   40
  ;; min-pycor  -40
  ;; max-pycor   40
  
  ;;----------------------------
  ;; SCENARIO SELECTION CONTROLS
  ;;----------------------------
  
  ;; gs-scenario         ;; Chooser, string converts to a scenario number
  g-scenario-number      ;; scenario no., 0, 1 or 2; interp. of gs-scenario
  ;; Glogal enumeration variables - There are 3 scenarios possible.
  ge-scenario-0-beliefs  ;; scenario 0
  ge-scenario-8-beliefs  ;; scenario 1
  ge-scenario-8-shares   ;; scenario 2
  ge-scenario-n-tribes   ;; scenario 3
  
  ;; To halt a scenario at a pre-determined tick.
  ;; g-halt-at-tick      ;; Has it's own input box

  ;; Initialize the Pseudo Random Number Generator (PRNG).
  ;; g-use-this-seed     ;; Slider [1,1,100,7]

  ;;-----------------------------------------------
  ;; BIOPHYSICAL SUB-SYSTEM CONTROLS AND PARAMETERS
  ;;-----------------------------------------------

  ;; Biophyscial life function parameters, seekers.
  ;; g-c2-dat-parm  ;; The death age threshold
  ;; g-c2-det-parm  ;; The death nrg threshold
  ;; g-c2-rat-parm  ;; The reproductive age threshold
  ;; g-c2-ret-parm  ;; The reproductive nrg threshold
  ;; g-c2-epm-parm  ;; The nrg per move
  ;; g-c2-epa-parm  ;; The maximum nrg an agent may hold
  
  ;; The global list of possible heading deltas for moves.
  gl-heading-list  ;; List of heading deltas.
  gl-index-list    ;; List of numbers 0-7 in order
  gl-base-factors  ;; List of factors, used to mutate bases
  gl-colour-list   ;; List of colours, for agents and graphs
  
  ;; Nrg control variables
  ;; g-sun-nrg-per-tick    ;; Nrg arriving per tick [1000,40,12000,6000]
  ;; g-nrg-per-block       ;; Nrg deposited per Dale [1,1,100,40]
  ;; g-prob-of-deposit     ;; Prob nrg will be deposited [0,.001,1,.1]
  ;; g-heuristic-delta  ;; Amount added to heuristic on success [0,.001,2,1]
  ;; g-prob-of-genetic-mutation ;; As it says [0,.001,1,.1]
  ;; g-dt-for-eroi      ;; Delta time, used for EROI [40,40,400,40]

  ;; Nrg accounting variables
  g-nrg-in-sunshine     ;; Nrg held in the sunshine, to be placed as fruit.
  g-nrg-in-fruit        ;; Nrg held in the fruit.
  gl-nrg-by-belief      ;; Nrg held, seekers, by belief (a list of 8 numbers).
  g-total-nrg-in-system ;; Total nrg in the system.
  gl-nrg-by-tribe       ;; Nrg held, seekers, by tribe (a list of 10 numbers)
  
  ;; Nrg in Sunshine takeup accounting variables
  g-sun-takeup-maximum   ;; Maximum possible, = [empty-cells]x[size-of-deposit]
  g-sun-takeup-expected  ;; Expected, = [Max]x[prob-of-deposit] 
  g-sun-takeup-actual    ;; As measured

  ;; Biophyscial life function parameters, seekers.
  ;; g-no-of-tribes      ;; seekers, per belief type, at startup [1,1,100,5]
  ;; g-c1-belief-value   ;; Default value [0,.1,3,2]

  ;; Global enumeration (ge-) codes for cause of death.
  ge-cod-none
  ge-cod-hunger
  ge-cod-fission
  ge-cod-oldage

  ;; List to hold counts of cause of death.
  gl-causes-of-death-per-tick
  gl-causes-of-death-cumulative

  ;;-------------------------------------
  ;; END OF MODEL PARAMETERS AND CONTROLS
  ;;-------------------------------------
  
  ;;-------------------------------------
  ;; DATA COLLECTION AND DISPLAY CONTROLS
  ;;-------------------------------------
  
  ;; The following global variables are not model controls or paramaters, 
  ;;   but, rather, are variables used to collect data about the model 
  ;;   for display in the user interface, in some fashion (monitors or plots),
  ;;   or used to manage all of the debug routines and output.
  
  ;; Global enumeration (ge-) codes.
  ge-sinktype-discard   ;; Discarded sunlight
  ge-sinktype-move-EPM  ;; Seekers EPM
  ge-sinktype-die-DET   ;; Remaining nrg of seeker on death.
  ge-sinktype-die-DAT   ;; Remaining nrg of seeker on death.

  ;; SYSTEM-WIDE AGGREGATES
  ;; System of nrg sinks.
  gl-sinks-per-tick
  gl-sinks-cumulative

  ;;---------------------------------------------------------------------------|
  ;; The following agent sets, counts and averages are for data collection 
  ;;   and display in monitors and plots.
  
  ;; Global counts
  g-no-of-patches          ;; count of all patches
  g-no-of-dales            ;; count of all dales with fruit
  gl-no-of-seekers         ;; count of all seekers; list
  g-no-of-seekers          ;; count of all seekers
  g-no-with-this-belief    ;; count of those with this belief.
  
  ;; Global EROI/ETA system-wide calculations.
  g-sys-nrg-returned      ;; Total nrg returned within delta T.
  g-sys-nrg-invested      ;; Total nrg invested within delta T.
  g-sys-nrg-income        ;; Total nrg income (flowing) within delta T.
  gl-sys-nrg-returned     ;; List of changes.
  gl-sys-nrg-invested     ;; List of changes.
  g-sys-eroi              ;; System-wide EROI, per tick (=B/C).
  g-sys-eta               ;; System-wide ETA, per tick  =(B/I).

  ;; Averages for seekers
  g-ave-age               ;; age of seekers
  g-ave-nrg               ;; nrg of seekers
  g-ind-min-eroi          ;; min eroi of individual seekers = (B/C)
  g-ind-ave-eroi          ;; ave eroi of individual seekers = (B/C)
  g-ind-max-eroi          ;; max eroi of individual seekers = (B/C)
  g-ind-min-eta           ;; min eta of individual seekers  = (B/I)
  g-ind-ave-eta           ;; ave eta of individual seekers  = (B/I)
  g-ind-max-eta           ;; max eta of individual seekers  = (B/I)

  g-ave-C1-b0             ;; c1, base character, gene-0
  g-ave-C1-b1             ;; c1, base character, gene-1
  g-ave-C1-b2             ;; c1, base character, gene-2
  g-ave-C1-b3             ;; c1, base character, gene-3
  g-ave-C1-b4             ;; c1, base character, gene-4
  g-ave-C1-b5             ;; c1, base character, gene-5
  g-ave-C1-b6             ;; c1, base character, gene-6
  g-ave-C1-b7             ;; c1, base character, gene-7

  g-ave-C1-g0             ;; c1, genotypic character, gene-0
  g-ave-C1-g1             ;; c1, genotypic character, gene-1
  g-ave-C1-g2             ;; c1, genotypic character, gene-2
  g-ave-C1-g3             ;; c1, genotypic character, gene-3
  g-ave-C1-g4             ;; c1, genotypic character, gene-4
  g-ave-C1-g5             ;; c1, genotypic character, gene-5
  g-ave-C1-g6             ;; c1, genotypic character, gene-6
  g-ave-C1-g7             ;; c1, genotypic character, gene-7

  g-ave-C1-l0             ;; c1, learned character, gene-0
  g-ave-C1-l1             ;; c1, learned character, gene-1
  g-ave-C1-l2             ;; c1, learned character, gene-2
  g-ave-C1-l3             ;; c1, learned character, gene-3
  g-ave-C1-l4             ;; c1, learned character, gene-4
  g-ave-C1-l5             ;; c1, learnedores character, gene-5
  g-ave-C1-l6             ;; c1, learned character, gene-6
  g-ave-C1-l7             ;; c1, learned character, gene-7

  g-ave-C1-s0             ;; c1, strength character, gene-0
  g-ave-C1-s1             ;; c1, strength character, gene-1
  g-ave-C1-s2             ;; c1, strength character, gene-2
  g-ave-C1-s3             ;; c1, strength character, gene-3
  g-ave-C1-s4             ;; c1, strength character, gene-4
  g-ave-C1-s5             ;; c1, strength character, gene-5
  g-ave-C1-s6             ;; c1, strength character, gene-6
  g-ave-C1-s7             ;; c1, strength character, gene-7

  g-ave-C1-p0             ;; c1, phenotypic character, gene-0
  g-ave-C1-p1             ;; c1, phenotypic character, gene-1
  g-ave-C1-p2             ;; c1, phenotypic character, gene-2
  g-ave-C1-p3             ;; c1, phenotypic character, gene-3
  g-ave-C1-p4             ;; c1, phenotypic character, gene-4
  g-ave-C1-p5             ;; c1, phenotypic character, gene-5
  g-ave-C1-p6             ;; c1, phenotypic character, gene-6
  g-ave-C1-p7             ;; c1, phenotypic character, gene-7

  ;; SWITCHES - These are declared in the switch itself, and so are 
  ;;   commented out here.  They are all native Booleans, having values of
  ;;   true or false.
  ;; gb-plot-data              ;; Enables plotting

  ;; Other - built-in or declared implicitly in plot interface items
  ;; See each plot design dialogue.

  ;;---------------
  ;; DEBUG CONTROLS
  ;;---------------
  
  gb-debug-on                 ;; Numeric Boolean, opens debug log file, 0 or 1.
  gs-debug-status             ;; for monitor, '1 (On)' or '0 (Off)', 
  ;; gs-debug-step-chooser    ;; Chooser, used with gb-debug-flow-on
  gb-debug-flow-on            ;; Numeric Boolean, in association with chooser, 
  gs-log-file-name            ;; name of the debug log file
                              ;;   opens flow to log file
]

  ;;-------------------------------------
  ;; DEFINING PATCHES AND BREEDS
  ;;-------------------------------------
  
;;-----------------------------------------------------------------------------|
;; Attributes of patches
patches-own 
[
  ;; BUILT-IN ATTRIBUTES 
  ;; pxcor        ;; min-pxcor <= pxcor < max-pxcor
  ;; pycor        ;; min-pxcor <= pxcor < max-pxcor 
  ;; pcolor       ;; color of this patch ( 0 <= color < 140 ) 
  ;; plabel       ;; label of this patch
  ;; plabel-color ;; color of this patch's label ( 0 <= label-color < 140 ) 
   
  ;; TpLab-DETERMINED ATTRIBUTES
  fruit
]

;;-----------------------------------------------------------------------------|
;; Attributes of links
;; nil
;; I don't understand links and did not use any.

;;-----------------------------------------------------------------------------|
;; Turtles and breeds
breed [ seekers seeker ]

;;-----------------------------------------------------------------------------|
;; Attributes of seekers
seekers-own 
[
  ;; BUILT-IN ATTRIBUTES
  ;; who         ;; fixed id number
  ;; breed       ;; to which breed this turtle belongs [seeker]
  ;; heading     ;; 0 <= heading < 360, 0 = north
  ;; xcor        ;; min-pxcor <= xcor < max-pxcor
  ;; ycor        ;; min-pxcor <= xcor < max-pxcor
  ;; size        ;; size relative to a patch, default is 1
  ;; shape       ;; a shape chosen from the shape library
  ;; color       ;; color of this turtle ( 0 <= color < 140 )
  ;; pen-mode    ;; "up" or "down"
  ;; pen-size    ;; in pixels
  ;; hidden?     ;; true or false
  ;; label       ;; label of this turtle
  ;; label-color ;; color of this turtle's label ( 0 <= label-color < 140 )
  
  ;; USER-DETERMINED ATTRIBUTES
  ;; The chromosome 1 (c1) genes are used to distinguish behaviours.
  c1-bases    ;; c1 - list of 8 [B]ases for genes (P=B^(G+L))
  c1-genes    ;; c1 - list of 8 heading delta [G]enes
  c1-learn    ;; c1 - list of 8 [L]earned beliefs
  c1-stren    ;; c1 - list of 8 [S]trengths
  c1-pheno    ;; c1 - list of 8 [P]henotypic characters
  belief-affiliation ;; The belief (0-7) held by this agent.
  tribal-affiliation ;; the social affiliation of the seeker.
  
  ;; The chromosome 2 (C2) genes are static in this model.
  DAT         ;; Death Age Threshold.
  DET         ;; Death Energy Threshold.
  RAT         ;; Reproductive Age Threshold.
  RET         ;; Reproductive Energy Threshold.
  EPM         ;; Energy Per Move.
  EPA         ;; Maximum Energy Per Agent.
  
  ;; Other variable characteritics.
  mas-who              ;; serial number of parent agent.
  age                  ;; age of the agent in ticks
  nrg                  ;; nrg in this agent
  cause-of-death       ;; for statistical purposes
  b-is-ready-to-move      ;; 0 = no; 1 = ready to move
  b-is-ready-to-reproduce ;; mature (in age) and healthy (in nrg)
  b-is-ready-to-die       ;; old (in age) or starved (in nrg)
  
  ;; Variables for calculating personal EROI and ETA.
  nrg-returned         ;; Numerator of EROI - an aggregate = Benefits of (B/C)
  nrg-invested         ;; Denominator of EROI - an aggregate = Costs of (B/C)
  nrg-income           ;; Denominator of ETA  - an aggregate = Income of (B/I)
  eroi                 ;; Nrg returned on nrg invested = (B/C)
  eta                  ;; Nrg efficiency = (B/I) where Income = (B+C)
  l-er                 ;; A list of delta ERs
  l-ei                 ;; A list of delta EIs
  counter-eroi         ;; For tracking time up to delta T.
]

;;-----------------------------------------------------------------------------|
;; SECTION C – INITIALIZATION OR SETUP PROCEDURE( S )
;;-----------------------------------------------------------------------------|

;;-----------------------------------------------------------------------------|
;; The 'autostart' startup routine
to startup
  ;; This routine is to be executed by the observer.

  ;; The manual describes this routine as follows:
  ;; This procedure, if it exists, will be called when a model is first loaded
  ;;   in the NetLogo application.  Startup does not run when a model is run 
  ;;   headless from the command line, or by parallel BehaviorSpace.

  ;; On loading the model, the debug feature is always off.
  set gb-debug-on 0
  set gs-debug-status "0 (Off)"
  
  ;; On loading the model, the model, the choosers, switches and sliders are
  ;;   always reset to the values that are known to work by here invoking 
  ;;   the f-reset-default-parameters routine.  Only the chooser
  ;;   for the scenario is not reset.  The last saved 
  ;;   selection of scenario will therefore be persistant.  This allows the 
  ;;  'Reset Defaults' button to NOT reset the scenario number, but to reset 
  ;;  correct parameters for the scenario.
  f-reset-default-parameters

  ;; Run the setup routine to initialize other globals.
  setup
end

;;-----------------------------------------------------------------------------|
;; Reset the default values for the interface-declared items.
to f-reset-default-parameters 
  ;; The observer executes this routine.

  ;; Switches, sliders and choosers implicitly declare global variables.  The
  ;;   values in these variables are parameters for the model, and many 
  ;;   combinations of those parameters are not sustainable.  However, the
  ;;   values in those user interface devices are stored with the model and
  ;;   are persistant across a save/load action.  The default values must
  ;;   be reset on load, or available to a user as a parameter set.  The
  ;;   purpose of this routine is to store at least one viable set of 
  ;;   parameter values.
  
  ;; DO NOT re-initialize the gs-scenario chooser.  The selected scenario
  ;;   is intended to be persistent, and not subject to a default setting.
  
  ;; Initialize the Pseudo Random Number Generator (PRNG).
  set g-use-this-seed 7          ;; [1,1,100,7]
  
  set gb-plot-data true          ;; Turn plotting on.
   
  ;;-----------------------------------------------
  ;; BIOPHYSICAL SUB-SYSTEM CONTROLS AND PARAMETERS
  ;;-----------------------------------------------

  ;; Slider range settings are shown as (Min,Inc,Max,Default)
  set g-no-of-tribes          10 ;; [1,1,100,5]
  set g-c1-belief-value        2 ;; [0,.1,3,2]
  set g-sun-nrg-per-tick    6000 ;; [1000,40,12000,6000] 
  set g-nrg-per-deposit       40 ;; [1,1,100,40]
  set g-prob-of-deposit      0.1 ;; [0,.001,1,.1]
  set g-prob-of-mutation     0.5 ;; [0,.001,1,.5]
  set g-dt-for-eroi           40 ;; [40,40,400,40]
  set g-gene-to-adjust         0 ;; [0,1,7,0]

  ;; Static chromosome 2 (C2) biophysical controls - borrowed from PSoup model.
  set g-c2-dat-parm        1600  ;; [100,10,3200,1600]
  set g-c2-det-parm           4  ;; [4,4,40,4]
  set g-c2-rat-parm         800  ;; [50,10,3200,800]
  set g-c2-ret-parm        1000  ;; [200,1,1600,1000]
  set g-c2-epm-parm           4  ;; [1,1,40,4]
  set g-c2-epa-parm        1600  ;; [1600,100,3000,1600]
  
  ;; End of f-reset-default-parameters
end

;;-----------------------------------------------------------------------------|
;; The setup button(s)
to setup
  ;; This routine is to be executed by the observer.

  ;; NOTE: The contents of switches, sliders, and choosers seem to be 
  ;;   immune to these 'clear' commands.
  clear-ticks
  clear-turtles
  clear-patches
  clear-drawing
  clear-all-plots
  clear-output
  ;; clear-globals   ;; Suppressed to make gb-debug-on value persistent.
  ;; NOTE: Instead of 'clear-globals', you must ensure all globals are 
  ;;   initialized properly in 'setup'.
  
  ;; import-drawing "01-B OrrSW.jpg"
  
  ;; Set the nrg (encoded in the variable fruit) in all of the patches to zero.
  ask patches
  [
    set fruit 0
    set pcolor brown
  ]
  
  ;; The version should be coded in this global variable to be included in
  ;;   output files.
  set gs-Version "TpLab_V2.07" 

  ;; Debug features may be off or on depending on history.
  ;;   - Perhaps 'setup' was called by 'to startup'.
  ;;   - Perhaps 'setup' was called during a 'BehaviorSpace' run.
  ;;   - Perhaps 'setup' was called by a user-pushed 'setup' button.
  ;; Setup needs to handle some quasi-persistant values correctly regardless of
  ;;   the history.  For gb-debug-on, in particular, I want it to be 
  ;;   persistant so I can have debug output from the 'setup' routine routed
  ;;   to the debug log file, or to the command centre.
  
  ;; 'startup' automatically sets gb-debug-on to 0 when the application is first
  ;;   loaded.  I want to be able to (A) toggle debug on, then, (B) press 
  ;;   'setup' and watch the debug output of the 'setup' command.  The 
  ;;   gb-debug-on must be persistant through the above 'clear' commands.  The 
  ;;   debug log file name and status, however, should not be persistent and 
  ;;   must be reset when setup runs, if appropriate.
  ifelse ( gb-debug-on = 1 )
  [
    ;; Debug is on due to user setting, so file name and status should be 
    ;;   reset.  I do this by turning the feature off then on.
    ;; First toggle it off, closing any remnant log file, if needed.
    f-toggle-debug
    ;; Then toggle it back on, opening a new time-stamped log file.
    f-toggle-debug
  ]
  ;; else
  [
    ;; Debug is off, possibly due to startup execution, possibly due to user 
    ;;   choice.
    ;; Ensure associated variables have compatible settings.
    set gb-debug-on 0              ;; Redundant but ensures consistency.
    set gs-debug-status "0 (Off)"  ;; Redundant but ensures consistency.
    set gb-debug-flow-on 0         ;; Step-specific flow is off.
    file-close-all                 ;; Close the debug log file.
    set gs-log-file-name "dummyname"
  ]
  
  ;; Now, do the standard check that is done at the start of each debuggable 
  ;;   routine.  This must follow the clear commands, which reset everything 
  ;;   except globals, switches, sliders and choosers.
  if( gb-debug-on = 1 )
  [
    ifelse( ( gs-debug-step-chooser = "all" ) or ( gs-debug-step-chooser = "setup" ) )
    [ set gb-debug-flow-on 1 LOG-TO-FILE "" LOG-TO-FILE word "Do-setup: Debug on; tick = " 0 ]
    [ set gb-debug-flow-on 0 ]
  ]

  ;; g-use-this-seed comes from a slider, and is persistant during setup.
  ;; However it is NOT persistent in a 'reset-defaults' call.
  random-seed g-use-this-seed      ;; Tells the PRNG to use this seed.
  
  ;; Establish the list of allowed headings, each 45 degrees from the last.
  ;;   These are the possible deltas that will be added to the current heading
  ;;   based on which of the 8 genes is expressed during a move.
  set gl-heading-list [ 0 45 90 135 180 225 270 315 ] 
  set gl-index-list   [ 0 1 2 3 4 5 6 7 ] 

  ;; The factors used to mutate the base values of the genes need to be 
  ;;   calculated.
  let prime-list [ 7 11 13 17 ]
  let factor-list ( map [ 1 + ( 1 / ? ) ] prime-list )
  let inverse-list ( map [ 1 - ( 1 / ? ) ] prime-list )
  set gl-base-factors ( sentence factor-list inverse-list )

  ;; Identify one colour for each possible value of preferred index.
  set gl-colour-list  [ 65 56 96 26 15 24 84 54 ]
  
  ;; Glogal enumeration variables - There are 4 scenarios possible.
  set ge-scenario-0-beliefs 0  ;; society not affected by beliefs
  set ge-scenario-8-beliefs 1  ;; society with 8 belief systems
  set ge-scenario-8-shares  2  ;; wisdom shared within belief system
  set ge-scenario-n-tribes  3  ;; wisdom also shared within familial group
  ;; Use the input from the chooser gs-scenario to invoke the selected scenario.
  f-set-scenario-number
  
  ;; For debugging the setup procedure, log the values of the globals.
  LOG-TO-FILE ( word "  Do-set: Scenario number    - " g-scenario-number ) 
  LOG-TO-FILE ( word "  Do-set: Scenario name      - " gs-scenario )
  LOG-TO-FILE ( word "  Do-set: Random seed        - " g-use-this-seed )

  ;; Declare values of hidden declarations from sliders.
  LOG-TO-FILE ( word "  Do-set: g-no-of-tribes     - " g-no-of-tribes )
  LOG-TO-FILE ( word "  Do-set: g-c1-belief-value  - " g-c1-belief-value )
  LOG-TO-FILE ( word "  Do-set: g-sun-nrg-per-tick - " g-sun-nrg-per-tick )
  LOG-TO-FILE ( word "  Do-set: g-nrg-per-deposit  - " g-nrg-per-deposit )
  LOG-TO-FILE ( word "  Do-set: g-prob-of-deposit  - " g-prob-of-deposit )
  LOG-TO-FILE ( word "  Do-set: g-prob-of-mutation - " g-prob-of-mutation )
  LOG-TO-FILE ( word "  Do-set: g-dt-for-eroi      - " g-dt-for-eroi )

  ;; Nrg accounting variables
  set g-nrg-in-sunshine     0 ;; Nrg held in the sunshine, to be placed as fruit.
  set g-nrg-in-fruit        0 ;; Nrg held in the fruit.
  set gl-nrg-by-belief  ( n-values  8 [0] ) ;; Nrg held, by belief affiliation.
  set g-total-nrg-in-system 0 ;; Nrg in the system.
  set gl-nrg-by-tribe   ( n-values 10 [0] ) ;; Nrg held, by tribal affiliation.
  
  ;; Nrg in Sunshine takeup accounting variables
  set g-sun-takeup-maximum   0 ;; = [empty-cells]x[size-of-deposit]
  set g-sun-takeup-expected  0 ;; = [Max]x[prob-of-deposit] 
  set g-sun-takeup-actual    0 ;; As measured

  ;; Global enumeration (ge-) codes for cause of death.
  set ge-cod-none     0
  set ge-cod-hunger   1
  set ge-cod-fission  2
  set ge-cod-oldage   3
  
  ;; List to hold counts of cause of death.
  set gl-causes-of-death-per-tick   ( n-values 4 [0] )
  set gl-causes-of-death-cumulative ( n-values 4 [0] )

  ;; Global enumeration (ge-) codes for sinktype.
  set ge-sinktype-discard     0 ;; Discarded sunlight
  set ge-sinktype-move-EPM    1 ;; Seeker EPM
  set ge-sinktype-die-DET     2 ;; Remaining nrg of seeker on death.
  set ge-sinktype-die-DAT     3 ;; Remaining nrg of seeker on death.
  
  ;; System of nrg sinks.
  set gl-sinks-per-tick   ( n-values 4 [0] )
  set gl-sinks-cumulative ( n-values 4 [0] )

  ;; Global EROI system-wide calculations.
  set g-sys-nrg-returned    0 ;; Total nrg returned within delta T.
  set g-sys-nrg-invested    0 ;; Total nrg invested within delta T.
  set g-sys-nrg-income      0 ;; Total nrg income within delta T.
  set gl-sys-nrg-returned  [] ;; List of changes.
  set gl-sys-nrg-invested  [] ;; List of changes.
  set g-sys-eroi          1.0 ;; System-wide EROI, per tick = (B/C).
  set g-sys-eta           0.5 ;; System-wide ETA, per tick  = (B/I).

  ;;---------------------------------------------------------------------------|
  ;; The following agent sets, counts and averages are for data collection 
  ;;   and display in monitors and plots.
  
  ;; Counts
  set g-no-of-patches          0 ;; count of all patches
  set g-no-of-dales            0 ;; count of all dales with fruit
  ;; Counts of all seekers, by belief-affiliation.
  set gl-no-of-seekers [ 0 0 0 0 0 0 0 0 ] 
  set g-no-of-seekers          0 ;; counts of all seekers
  set g-no-with-this-belief    0 ;; count of those with this belief.


  ;; Averages for seekers
  set g-ave-age      0        ;; age of seekers
  set g-ave-nrg      0        ;; nrg of seekers
  set g-ind-min-eroi 1.0      ;; min individual eroi of seekers = (B/C)
  set g-ind-ave-eroi 1.0      ;; ave individual eroi of seekers = (B/C)
  set g-ind-max-eroi 1.0      ;; max individual eroi of seekers = (B/C)
  set g-ind-min-eta  0.5      ;; min individual eta of seekers = (B/I)
  set g-ind-ave-eta  0.5      ;; ave individual eta of seekers = (B/I)
  set g-ind-max-eta  0.5      ;; max individual eta of seekers = (B/I)

  set g-ave-C1-b0   0         ;; c1, base character, gene-0
  set g-ave-C1-b1   0         ;; c1, base character, gene-1
  set g-ave-C1-b2   0         ;; c1, base character, gene-2
  set g-ave-C1-b3   0         ;; c1, base character, gene-3
  set g-ave-C1-b4   0         ;; c1, base character, gene-4
  set g-ave-C1-b5   0         ;; c1, base character, gene-5
  set g-ave-C1-b6   0         ;; c1, base character, gene-6
  set g-ave-C1-b7   0         ;; c1, base character, gene-7

  set g-ave-C1-g0   0         ;; c1, genotypic character, gene-0
  set g-ave-C1-g1   0         ;; c1, genotypic character, gene-1
  set g-ave-C1-g2   0         ;; c1, genotypic character, gene-2
  set g-ave-C1-g3   0         ;; c1, genotypic character, gene-3
  set g-ave-C1-g4   0         ;; c1, genotypic character, gene-4
  set g-ave-C1-g5   0         ;; c1, genotypic character, gene-5
  set g-ave-C1-g6   0         ;; c1, genotypic character, gene-6
  set g-ave-C1-g7   0         ;; c1, genotypic character, gene-7

  set g-ave-C1-l0   0         ;; c1, learned character, gene-0
  set g-ave-C1-l1   0         ;; c1, learned character, gene-1
  set g-ave-C1-l2   0         ;; c1, learned character, gene-2
  set g-ave-C1-l3   0         ;; c1, learned character, gene-3
  set g-ave-C1-l4   0         ;; c1, learned character, gene-4
  set g-ave-C1-l5   0         ;; c1, learned character, gene-5
  set g-ave-C1-l6   0         ;; c1, learned character, gene-6
  set g-ave-C1-l7   0         ;; c1, learned character, gene-7

  set g-ave-C1-s0   0         ;; c1, strength character, gene-0
  set g-ave-C1-s1   0         ;; c1, strength character, gene-1
  set g-ave-C1-s2   0         ;; c1, strength character, gene-2
  set g-ave-C1-s3   0         ;; c1, strength character, gene-3
  set g-ave-C1-s4   0         ;; c1, strength character, gene-4
  set g-ave-C1-s5   0         ;; c1, strength character, gene-5
  set g-ave-C1-s6   0         ;; c1, strength character, gene-6
  set g-ave-C1-s7   0         ;; c1, strength character, gene-7

  set g-ave-C1-p0   0         ;; c1, phenotypic character, gene-0
  set g-ave-C1-p1   0         ;; c1, phenotypic character, gene-1
  set g-ave-C1-p2   0         ;; c1, phenotypic character, gene-2
  set g-ave-C1-p3   0         ;; c1, phenotypic character, gene-3
  set g-ave-C1-p4   0         ;; c1, phenotypic character, gene-4
  set g-ave-C1-p5   0         ;; c1, phenotypic character, gene-5
  set g-ave-C1-p6   0         ;; c1, phenotypic character, gene-6
  set g-ave-C1-p7   0         ;; c1, phenotypic character, gene-7

  ;; For debugging the debug feature!!!  Suppressed now.
  ;; show ( word "SETUP: Debug Is " gb-debug-on )
  ;; show ( word "SETUP: Debug Status Is " gs-debug-status )
  ;; show ( word "SETUP: Step Chooser Is " gs-debug-step-chooser )
  ;; show ( word "SETUP: Flow Control Is " gb-debug-flow-on )

  set-default-shape seekers  "arrow" ;; pulled from shapes library

  ask patches 
  [ 
    set pcolor brown
  ]
   
  reset-ticks ;; restarts tick counter, runs setup commands within plots
  set gb-plot-data true ;; Enables all plotting calls.
  
  ;; Initialize the seekers.
  ;; This differs for each scenario.
  ;; Injects nrg, and establishes belief systems.
  f-initialize-seekers
  
  ;; This call must follow 'reset-ticks' and initialization of seekers.
  f-update-aggregates  ;; Totals and averages.
 
  ;; Clears unwanted zeros in plots.
  clear-all-plots
  setup-plots
  
  ;; Debug controls
  ;; Boolean, in association with chooser, turns debug LOG-TO-FILE on/off
  set gb-debug-flow-on 0 
  ;; Input variable to set a tick for stopping.
  set g-halt-at-tick -1  

  Set g-nrg-in-fruit ( sum [fruit] of patches )
  ;; ASSERT ( frb-nrg-accounts-are-all-valid ) 
  ;;   ( "Do-set: Nrg accounts invalid." ) -1
  LOG-TO-FILE "  Do-set: procedure completed" 

  ;; end of to-setup
end

;;-----------------------------------------------------------------------------|
;; Set the scenario number using the input from the chooser.
to f-set-scenario-number
  ;; This routine is to be executed by the observer.

  set g-scenario-number ge-scenario-0-beliefs  ;; default

  if( gs-scenario = "Society With 0 Beliefs" )
    [ set g-scenario-number ge-scenario-0-beliefs ]

  if( gs-scenario = "Society With 8 Beliefs" )
    [ set g-scenario-number ge-scenario-8-beliefs ]

  if( gs-scenario = "Society With Wisdom Sharing" )
    [ set g-scenario-number ge-scenario-8-shares ]

  if( gs-scenario = "Society With Tribal Bonds" )
    [ set g-scenario-number ge-scenario-n-tribes ]

  ;; End f-set-scenario-number
end

;;-----------------------------------------------------------------------------|
;; Initialize a population of seekers.
to f-initialize-seekers
  ;; This routine is to be executed by the observer.

  ;; Load nrg into the sunshine, for setup.
  set g-nrg-in-sunshine g-sun-nrg-per-tick
  set g-total-nrg-in-system ( g-total-nrg-in-system + g-nrg-in-sunshine )
  
  ;; Record maximum possible Sunshine takeup rate data.
  let empty-cells ( count patches with [fruit = 0] )
  set g-sun-takeup-maximum ( empty-cells * g-nrg-per-deposit )
  ;; Record expected Sunshine takeup rate data.
  set g-sun-takeup-expected ( g-sun-takeup-maximum * g-prob-of-deposit )

  ;; I want a population of each type of preferred gene.
  ;; Although I have allocated an amount of energy for this, I do not stop
  ;;   initializing new agents if/when it runs out.  I simply record the
  ;;   over-draw as a negative in the appropriate sink
  
  if( g-scenario-number = ge-scenario-0-beliefs )
    [ f-initialize-0-beliefs-society ]
  if( g-scenario-number = ge-scenario-8-beliefs )
    [ f-initialize-8-beliefs-society ]
  if( g-scenario-number = ge-scenario-8-shares )
    [ f-initialize-wisdom-sharing ] 
  if( g-scenario-number = ge-scenario-n-tribes )
    [ f-initialize-tribal-sharing ]

  ;; Place more energy into patches.
  ask patches
  [
    set fruit g-nrg-per-deposit
    set g-total-nrg-in-system ( g-total-nrg-in-system + g-nrg-per-deposit )
    set pcolor green
  ]

  ;; Record the actual Sunshine takeup rate data.
  set g-sun-takeup-actual 0
  
  ;; Put the rest of the Sunshine into the sink.
  set g-total-nrg-in-system ( g-total-nrg-in-system - g-sun-nrg-per-tick )
  f-store-data-in-sink ge-sinktype-discard g-nrg-in-sunshine
  
  ;; End of f-initialize-seekers
end

;;-----------------------------------------------------------------------------|
;; Initialize a society characterized by 0 belief systems.
to f-initialize-0-beliefs-society
  ;; This routine is to be executed by the observer.

  ;; In this scenario, we create 8 populations of seekers which are all
  ;;   identical except for location and heading.  None are affected 
  ;;   by any belief affiliation.  This is identical to the 8-beliefs society
  ;;   initialization, except the slider g-c1-belief-value is ignored.
  
  foreach gl-index-list  ;; 8 genes.
  [
    create-seekers g-no-of-tribes 
    [ 
      f-initialize-new-seeker 
    
      ;; In this scenario, seekers have a belief affiliation but it 
      ;;   is de-activated due to the value of the belief being set 
      ;;   to zero.
      
      ;; Identify the belief affiliation.
      set belief-affiliation 0
      ;; This is set to zero, but it not used.  All seekers have a
      ;;   preferred gene of zero, but the associated belief value is 
      ;;   also zero so it has no effect.
      
      ;; Set the heading as one of the 8 allowed headings.
      set heading ( item ( random 8 ) gl-heading-list )

      set color ( item belief-affiliation gl-colour-list )
      set age ( RAT / 2 ) ;; Age is halfway to mature.
      set nrg ( EPA ) ;; Nrg is at max.
      ;; All start with same age and nrg level to avoid bias.
      
      ;; Update the energy statistics.
      let nrg-was ( item belief-affiliation gl-nrg-by-belief )
      let nrg-is-now ( nrg-was + nrg )
      set gl-nrg-by-belief 
        ( replace-item belief-affiliation gl-nrg-by-belief nrg-is-now )
      LOG-TO-FILE ( word "  Do-set: gl-nrg-by-belief - " gl-nrg-by-belief )
      set g-nrg-in-sunshine ( g-nrg-in-sunshine - nrg )
      
      ;; Genetic and belief characters.
      set c1-bases [ 2 2 2 2 2 2 2 2 ]  ;; 8 unbiased genes
      set c1-genes [ 0 0 0 0 0 0 0 0 ]  ;; 8 unbiased genes
      set c1-learn [ 0 0 0 0 0 0 0 0 ]  ;; 8 unbiased beliefs
      
      ;; IGNORE THE g-c1-belief-value SLIDER.  Leave the beliefs unbiased.
      ;; Agent has an enhanced preference to exercise gene ?.
      ;; set c1-learn 
      ;;   ( replace-item ? c1-learn g-c1-belief-value ) ;; Biased beliefs
      
      ;; Calculate the strengths and phenotypic characters.
      f-find-strens-n-phenos
      
      LOG-TO-FILE ( word "  Do-set: C1-bases - " c1-bases )
      LOG-TO-FILE ( word "  Do-set: C1-genes - " c1-genes )
      LOG-TO-FILE ( word "  Do-set: c1-learn - " c1-learn )
      LOG-TO-FILE ( word "  Do-set: c1-stren - " c1-stren )
      LOG-TO-FILE ( word "  Do-set: c1-pheno - " c1-pheno )

      ;; Move each agent to a random point.
      setxy random-xcor random-ycor

    ]  ;; End of create.
  ]  ;; End of foreach

  ;; End of f-initialize-0-beliefs-society
end

;;-----------------------------------------------------------------------------|
;; Calculate the strengths and phenotypic values.
to f-find-strens-n-phenos
  ;; This routine is to be executed by a seeker.

  ;; It uses the second example of the map feature.
  ;; Examples of the map feature.
  ;; show (map + [1 2 3] [2 4 6])
  ;; => [3 6 9]
  ;; show (map [?1 + ?2 = ?3] 
  ;;           [1 2 3] 
  ;;           [2 4 6] 
  ;;           [3 5 9])
  ;; => [true false true]
    
  ;; Compute the strength as S=B^(G+L).
  set c1-stren ( map [?1 ^ ( ?2 + ?3 )] 
                     c1-bases 
                     c1-genes 
                     c1-learn )
  ;; Compute the phenotypic character as Pi=100*(Si/sum(Si)).
  set c1-pheno 
    ( map [?1 * ?2 / ?3] 
          ( n-values 8 [100] ) 
          c1-stren 
          ( n-values 8 [sum c1-stren] ) )
  ;; End of f-find-strens-n-phenos
end

;;-----------------------------------------------------------------------------|
;; Initialize a society characterized by 8 beliefs.
to f-initialize-8-beliefs-society
  ;; This routine is to be executed by the observer.

  ;; In this scenario, we create 8 populations of seekers which are all
  ;;   identical except for (a) location and heading; and (b) belief 
  ;;   affiliation.  There are 8 different belief affiliations, one for each
  ;;   compound gene in chromosome #1 (C1).  Each has their belief value
  ;;   set as per the slider g-c1-belief-value.
  
  foreach gl-index-list  ;; 8 genes.
  [
    create-seekers g-no-of-tribes 
    [ 
      f-initialize-new-seeker 
    
      ;; Identify the preferred heading-delta.
      set belief-affiliation ?
      
      ;; Set the heading as one of the 8 allowed headings.
      set heading ( item ( random 8 ) gl-heading-list )

      set color ( item ? gl-colour-list )
      set age ( RAT / 2 ) ;; Age is halfway to mature.
      set nrg ( EPA ) ;; Nrg is at max.
      ;; All start with same age and nrg level to avoid bias.
      
      ;; Update the energy statistics.
      let nrg-was ( item ? gl-nrg-by-belief )
      let nrg-is-now ( nrg-was + nrg )
      set gl-nrg-by-belief ( replace-item ? gl-nrg-by-belief nrg-is-now )
      LOG-TO-FILE ( word "  Do-set: gl-nrg-by-belief - " gl-nrg-by-belief )
      set g-nrg-in-sunshine ( g-nrg-in-sunshine - nrg )
      
      ;; Genetic and belief characters.
      set c1-bases [ 2 2 2 2 2 2 2 2 ]  ;; 8 unbiased genes
      set c1-genes [ 0 0 0 0 0 0 0 0 ]  ;; 8 unbiased genes
      set c1-learn [ 0 0 0 0 0 0 0 0 ]  ;; 8 unbiased beliefs
      ;; Agent has an enhanced preference to exercise gene ?.
      set c1-learn 
      ( replace-item ? c1-learn g-c1-belief-value )  ;; Biased beliefs

      ;; Calculate the strengths and phenotypic characters.
      f-find-strens-n-phenos
      
      LOG-TO-FILE ( word "  Do-set: C1-bases - " c1-bases )
      LOG-TO-FILE ( word "  Do-set: C1-genes - " c1-genes )
      LOG-TO-FILE ( word "  Do-set: c1-learn - " c1-learn )
      LOG-TO-FILE ( word "  Do-set: c1-stren - " c1-stren )
      LOG-TO-FILE ( word "  Do-set: c1-pheno - " c1-pheno )

      ;; Move each agent to a random point.
      setxy random-xcor random-ycor

    ]  ;; End of create.
  ]  ;; End of foreach
  
  ;; End of f-initialize-8-beliefs-society
end

;;-----------------------------------------------------------------------------|
;; Initialize a society characterized by 8 beliefs with wisdom sharing.
to f-initialize-wisdom-sharing
  ;; This routine is to be executed by the observer.

  ;; This will be an 8-beliefs society, similar to the previous scenario,
  ;;   but with the ability of junior seekers to get wisdom from their
  ;;   elders who share the same belief system.  
  ;; When two seekers having the same belief affiliation 
  ;;   occupy the same cell, a seeker may have wisdom about
  ;;   that belief transferred from the eldest such seeker, called
  ;;   the local guru.
  ;; Wisdom is transferred during the 'do-feed' step.
  
  ;; The seekers are no different from those in an 8-beliefs society.
  f-initialize-8-beliefs-society
  
  ;; End of f-initialize-wisdom-sharing
end

;;-----------------------------------------------------------------------------|
;; Initialize a society characterized by 8 belief systems.
to f-initialize-tribal-sharing
  ;; This routine is to be executed by the observer.

  ;; In this scenario, we create a number of tribes of seekers.  Each
  ;;   tribe consists of eight seekers having the full range of belief
  ;;   affiliations.
  
  let no-of-belieftypes 8
  let tribe-no 0
  let belief-no  0
  
  while [ tribe-no < g-no-of-tribes ]
  [
    LOG-TO-FILE ( WORD "  Do-set: Tribal affiliation - " tribe-no )
    set belief-no 0
    while [ belief-no < no-of-belieftypes ]
    [
      LOG-TO-FILE
        ( WORD "  Do-set: (Tribe, Belief)   - (" tribe-no ", " belief-no ")" )
      
      create-seekers 1
      [
        f-initialize-new-seeker
        
        ;; Identify the belief affiliation.
        set belief-affiliation belief-no
        ;; Set the heading as one of the 8 allowed headings.
        set heading ( item ( random 8 ) gl-heading-list )
        ;; Set the tribe number.
        set tribal-affiliation tribe-no

        set color ( item belief-no gl-colour-list )
        set age ( RAT / 2 ) ;; Age is halfway to mature.
        set nrg ( EPA ) ;; Nrg is at max.
        ;; All start with same age and nrg level to avoid bias.
      
        ;; Update the energy statistics.
        let nrg-was ( item belief-no gl-nrg-by-belief )
        let nrg-is-now ( nrg-was + nrg )
        set gl-nrg-by-belief 
          ( replace-item belief-no gl-nrg-by-belief nrg-is-now )
        LOG-TO-FILE ( word "  Do-set: gl-nrg-by-belief - " gl-nrg-by-belief )
        set g-nrg-in-sunshine ( g-nrg-in-sunshine - nrg )
      
        ;; Genetic and belief characters.
        set c1-bases [ 2 2 2 2 2 2 2 2 ]  ;; 8 unbiased genes
        set c1-genes [ 0 0 0 0 0 0 0 0 ]  ;; 8 unbiased genes
        set c1-learn [ 0 0 0 0 0 0 0 0 ]  ;; 8 unbiased beliefs
        ;; Agent has an enhanced preference to exercise gene belief-no.
        set c1-learn 
          ( replace-item belief-no c1-learn g-c1-belief-value ) ;; Biased beliefs

        ;; Calculate the strengths and phenotypic characters.
        f-find-strens-n-phenos
      
        LOG-TO-FILE ( word "  Do-set: C1-bases - " c1-bases )
        LOG-TO-FILE ( word "  Do-set: C1-genes - " c1-genes )
        LOG-TO-FILE ( word "  Do-set: c1-learn - " c1-learn )
        LOG-TO-FILE ( word "  Do-set: c1-stren - " c1-stren )
        LOG-TO-FILE ( word "  Do-set: c1-pheno - " c1-pheno )

        ;; Move each agent to a random point.
        setxy random-xcor random-ycor
      ]
      set belief-no ( belief-no + 1 )
    ]
    set tribe-no ( tribe-no + 1 )
  ]
  
  ;; End of f-initialize-tribal-sharing
end

;;-----------------------------------------------------------------------------|
;; Initialize a society characterized by 8 belief systems.
to f-impose-belief-affiliation
  ;; This routine is to be executed by the observer.

  ;; This routine will impose a belief affiliation on roughly 1/2 of the 
  ;;   current population of seekers.  The gene affected by this imposition 
  ;;   is determined by the slider g-belief-to-adjust.  The magnitude of the
  ;;   belief is determined by the slider g-c1-belief-value.
  
  ;; It can be called by the "Impose This Belief Affiliation" button at 
  ;;   any time during any scenario.
  
  let selector 0
  ask seekers
  [
    ifelse ( selector = 0 )
    [
      ;; This seeker has been selected to have its belief affiliation adjusted.
      ;; The gene adjusted is determined by g-gene-to-adjust
      let adjustment g-c1-belief-value
      let target g-gene-to-adjust
      set c1-learn ( replace-item target c1-learn adjustment )
      set belief-affiliation g-gene-to-adjust
      
      set selector 1
    ]  ;; End if selector = 0
    ;; Else selector = 1
    [
      ;; This seeker will not have its belief affiliation adjusted.
      ;; No action required, other than resetting the selector.
      
      set selector 0
    ]  ;; End else if selector = 1
  ]
  f-update-aggregates
  
  ;; End of f-impose-belief-affiliation
end

;;-----------------------------------------------------------------------------|
;; Initialize a single seeker.
to f-initialize-new-seeker
  ;; This routine is to be executed by a seeker.

  ;; BUILT-IN ATTRIBUTES
  ;; who         ;; set automatically
  ;; heading     ;; direction of motion
  ;; xcor        ;; min-pxcor <= xcor < max-pxcor
  ;; ycor        ;; min-pxcor <= xcor < max-pxcor
  ;; pen-mode    ;; "up" or "down"
  ;; pen-size    ;; in pixels
  ;; size        ;; size relative to a patch, default is 1

  ;; USER-DETERMINED ATTRIBUTES
  ;; These two are re-initialized specifically for some scenarios.
  set belief-affiliation 0
  set tribal-affiliation 0
  
  ;; The biophysical body function genes are static in this model.
  ;; Load chromosome 2 with the parameters from sliders.
  set DAT g-c2-dat-parm
  set DET g-c2-det-parm
  set RAT g-c2-rat-parm
  set RET g-c2-ret-parm
  set EPM g-c2-epm-parm
  set EPA g-c2-epa-parm

  ;; Associated with seeker dynamics.
  set mas-who -1                 ;; serial number of parent seeker.
  ;; age and nrg are set in group initialization routine.
  set cause-of-death ge-cod-none ;; for statistical purposes.

  ;; Set the logic trigger flags.
  set b-is-ready-to-move      1 ;; i.e. true
  set b-is-ready-to-reproduce 0 ;; i.e. false
  set b-is-ready-to-die       0 ;; i.e. false
  
  ;; Variables for calculating individual EROI and ETA.
  set nrg-returned g-dt-for-eroi ;; Numerator of EROI - an aggregate
  set nrg-invested g-dt-for-eroi ;; Denominator of EROI - an aggregate
  set nrg-income ( nrg-returned + nrg-invested )
  set eroi              1.0 ;; Nrg returned on nrg invested = (B/C)
  set eta               0.5 ;; individual efficiency        = (B/I)
  set l-er ( n-values g-dt-for-eroi [1] ) ;; Delta ERs = B of B/C
  set l-ei ( n-values g-dt-for-eroi [1] ) ;; Delta EIs = C of B/C
  set counter-eroi g-dt-for-eroi ;; For tracking time up to delta T.

  ;; end f-initialize-new-seeker
end

;;-----------------------------------------------------------------------------|
;; SECTION D – GO OR MAIN-LOOP PROCEDURE( S )
;;-----------------------------------------------------------------------------|

;;-----------------------------------------------------------------------------|
;; The go button
to go
  ;; This routine is to be executed by the observer.

  ;; Stop codes:
  ;; All stop decisions must be here in the 'go' procedure, as it causes an
  ;;   exit from the current procdure only.

  if( g-halt-at-tick = ticks  ) 
  [
    set g-halt-at-tick -1
    stop
  ]
  
  let b-should-stop-now false
  if( count turtles <= 0 ) [ set b-should-stop-now true ]
  if( b-should-stop-now = true )
  [
    stop
  ]

  ;; MANUAL CHANGE FOR DEBUG
  ;; If needed, each check for validity can be enabled between steps.
  ;; They have been suppressed (turned into comments) for the sake 
  ;;   of speed of execution, but can be re-enabled if a bug has 
  ;;   somehow been re-introduced.
  ;; A single call to the validity check has been left active inside of the
  ;;   Do-Post-Tick step.  If it flags a problem, re-activate these to
  ;;   narrow down where the problem starts.
  
  ;; Major steps or functions, done once per tick, in order of execution.
  do-pre-tick
  ;; if( frb-agents-are-all-valid = false ) 
  ;;   [ LOG-TO-FILE ( word "Agents failed validity test: Do-pre-tick." ) ]

  do-energize
  ;; if( frb-agents-are-all-valid = false ) 
  ;;   [ LOG-TO-FILE ( word "Agents failed validity test: Do-energize." ) ]

  do-move
  ;; if( frb-agents-are-all-valid = false ) 
  ;;   [ LOG-TO-FILE ( word "Agents failed validity test: Do-move." ) ]

  do-feed
  ;; if( frb-agents-are-all-valid = false ) 
  ;;   [ LOG-TO-FILE ( word "Agents failed validity test: do-feed." ) ]

  do-reproduce
  ;; if( frb-agents-are-all-valid = false ) 
  ;;   [ LOG-TO-FILE ( word "Agents failed validity test: Do-reproduce." ) ]

  do-die
  ;; if( frb-agents-are-all-valid = false ) 
  ;;   [ LOG-TO-FILE ( word "Agents failed validity test: Do-die." ) ]

  do-post-tick
  ;; if( frb-agents-are-all-valid = false ) 
  ;;   [ LOG-TO-FILE ( word "Agents failed validity test: Do-post-tick." ) ]

end

;;-----------------------------------------------------------------------------|
;; D1 - do-pre-tick procedure( s )
;;-----------------------------------------------------------------------------|
to do-pre-tick
  ;; This routine is to be executed by the observer.
  
  if( gb-debug-on = 1 )
  [
    ifelse( ( gs-debug-step-chooser = "all" ) or ( gs-debug-step-chooser = "pre-tick" ) )
    [ set gb-debug-flow-on 1 LOG-TO-FILE "" LOG-TO-FILE word "Do-pre-tick: Debug on.; tick = " ticks ]
    [ set gb-debug-flow-on 0 ]
  ]
  
  ;; Enter all commands that need to be done before a tick begins.
  ;; Supressed. f-update-aggregates
  
  ;; Advance the tick counter by 1 tick.
  ifelse( gb-plot-data = true )
  [
    ;; Advance the ticks by one and update the plots.
    tick
    ;; 'tick' is exactly the same as 'update-plots' except that the tick counter 
    ;;   is incremented before the plot commands are executed.
    
  ]
  ;; else
  [
    ;; Advance ticks by one but do not update the plots.
    tick-advance 1
  ]
  ;; End else

  ;; Once the data is plotted, the per-tick counts can be cleared.
  ;; List to hold counts of cause of death.
  set gl-causes-of-death-per-tick   ( n-values 6 [0] )

  ;; Global EROI system-wide calculations.
  ifelse( length gl-sys-nrg-returned <= g-dt-for-eroi )
  [
    ;; Append a zero for new data.
    set gl-sys-nrg-returned ( lput 0 gl-sys-nrg-returned ) ;; List of changes.
    set gl-sys-nrg-invested ( lput 0 gl-sys-nrg-invested ) ;; List of changes.
  ]
  ;; Else
  [
    ;; Remove old data
    set gl-sys-nrg-returned ( butfirst gl-sys-nrg-returned ) ;; List of changes.
    set gl-sys-nrg-invested ( butfirst gl-sys-nrg-invested ) ;; List of changes.
    ;; Append a zero for new data.
    set gl-sys-nrg-returned ( lput 0 gl-sys-nrg-returned ) ;; List of changes.
    set gl-sys-nrg-invested ( lput 0 gl-sys-nrg-invested ) ;; List of changes.
  ]

  ;; Reset the scenario number, in case the chooser has been changed.
  f-set-scenario-number
  
  ;; Clear the per-tick data for energy sinks.
  ;; This call must happen before the seeker population is stabilized.
  set gl-sinks-per-tick ( n-values 4 [0] )
  
  ask seekers [ set age ( age + 1 ) ]
  LOG-TO-FILE ( word "  Do-pre-tick: Seekers aged." )

  LOG-TO-FILE ( word "  Do-pre-tick: Halt at tick - " g-halt-at-tick  ) 
  LOG-TO-FILE ( word "  Do-pre-tick: Current tick - " ticks ) 

  LOG-TO-FILE "  Do-pre-tick: Routine completed."
end


;;-----------------------------------------------------------------------------|
;; D2 – do-energize procedure(s)
;;-----------------------------------------------------------------------------|
to do-energize
  ;; This routine is to be executed by the observer.
  
  if( gb-debug-on = 1 )
  [
    ifelse( ( gs-debug-step-chooser = "all" ) or ( gs-debug-step-chooser = "energize" ) )
    [ set gb-debug-flow-on 1 LOG-TO-FILE "" LOG-TO-FILE word "Do-energize: Debug on; tick = " ticks ]
    [ set gb-debug-flow-on 0 ]
  ]
  
  ;; The Sun radiates a fixed amount of energy per tick into the system.
  set g-nrg-in-sunshine ( g-nrg-in-sunshine + g-sun-nrg-per-tick )
  set g-total-nrg-in-system ( g-total-nrg-in-system + g-sun-nrg-per-tick )
  
  ;; Record maximum possible Sunshine takeup rate data.
  let empty-cells ( count patches with [fruit = 0] )
  set g-sun-takeup-maximum ( empty-cells * g-nrg-per-deposit )
  ;; Record expected Sunshine takeup rate data.
  set g-sun-takeup-expected ( g-sun-takeup-maximum * g-prob-of-deposit )

  ;; Make a list of the patches that are without fruit (without nrg)
  let empty-patch-list ( patches with [fruit = 0] )

  ;; Use the probability of deposit to determine if fruit is added.
  ask empty-patch-list
  [
    let random-number ( random 100000 )
    let threshold ( 100000 * g-prob-of-deposit )
    if ( ( random-number <= threshold ) and 
         ( g-nrg-in-sunshine > g-nrg-per-deposit ) )
    [
      set fruit ( fruit + g-nrg-per-deposit )
      set pcolor green
      ;; Record in system nrg accounts
      set g-nrg-in-fruit  ( g-nrg-in-fruit  + g-nrg-per-deposit )
      set g-nrg-in-sunshine ( g-nrg-in-sunshine - g-nrg-per-deposit )
    ]
  ]
  
  ;; Record actual Sunshine takeup rate data.
  set g-sun-takeup-actual ( g-sun-nrg-per-tick - g-nrg-in-sunshine )

  ;; Discard any remaining energy that has not been absorbed by plants.
  f-store-data-in-sink ge-sinktype-discard g-nrg-in-sunshine
  set g-total-nrg-in-system ( g-total-nrg-in-system - g-nrg-in-sunshine )
  set g-nrg-in-sunshine 0
      
  ;; Supressed. f-update-aggregates

  LOG-TO-FILE "  Do-energize: procedure completed"
end

;;-----------------------------------------------------------------------------|
;; D3 – do-move procedure(s)
;;-----------------------------------------------------------------------------|
to do-move
  ;; This routine is to be executed by the observer.
  
  if( gb-debug-on = 1 )
  [
    ifelse( ( gs-debug-step-chooser = "all" ) or ( gs-debug-step-chooser = "move" ) )
    [ set gb-debug-flow-on 1 LOG-TO-FILE "" LOG-TO-FILE word "Do-move: Debug on; tick = " ticks ]
    [ set gb-debug-flow-on 0 ]
  ]
  
  ;; The seekers move.
  ask seekers
  [
    if( b-is-ready-to-move = 1 )
    [
      f-seeker-moves
    ]  ;; End if( b-is-ready-to-move = 1 )
  ]  ;; End ask seekers

  ;; Supressed. f-update-aggregates

  LOG-TO-FILE "  Do-move: procedure completed"
end

;;-----------------------------------------------------------------------------|
;; A seeker moves according to genes and heuristics.
to f-seeker-moves
  ;; This routine is to be executed by a seeker.

  ;; When a seeker moves it expends energy out of its pool of nrg.
  ;; Determine if this seeker has sufficient nrg to move.
  ifelse ( nrg >= EPM )
  [
    ;; Establish a heading.
    f-seeker-sets-heading
    
    ;; Step forward
    forward 1
    
    ;; Expend the nrg to the sink.
    f-store-data-in-sink ge-sinktype-move-EPM EPM
    set nrg ( nrg - EPM )
    
    ;; Record the expenditure in the stats.
    let sum-was ( item belief-affiliation gl-nrg-by-belief )
    let sum-is-now ( sum-was - EPM )
    set gl-nrg-by-belief 
      ( replace-item belief-affiliation gl-nrg-by-belief sum-is-now )
    set g-total-nrg-in-system ( g-total-nrg-in-system - EPM )

    ;; Record the expenditure in the EROI variables
    f-record-ei-for-eroi EPM
    
    set b-is-ready-to-move 1
    ;; set b-is-ready-to-die 0
  ] 
  ;; Else
  [
    ;; The seeker is marked for death, and nrg is removed.
    ;; It will die and be removed when do-die is executed.    
    f-store-data-in-sink ge-sinktype-move-EPM nrg
    let sum-was ( item belief-affiliation gl-nrg-by-belief )
    let sum-is-now ( sum-was - nrg )
    set gl-nrg-by-belief 
      ( replace-item belief-affiliation gl-nrg-by-belief sum-is-now )
    set g-total-nrg-in-system ( g-total-nrg-in-system - nrg )

    ;; Record the expenditure in the EROI variables
    f-record-ei-for-eroi nrg
    
    set nrg 0 
    set cause-of-death ge-cod-hunger
    set b-is-ready-to-move 0
    set b-is-ready-to-die 1
  ]
  ;; End else

  LOG-TO-FILE ( word "  Do-move: S(heading,nrg,move-flag,die-flag) - (" 
    heading "," 
    floor nrg "," 
    b-is-ready-to-move"," 
    b-is-ready-to-die ")" )
  
  ;; End of f-seeker-moves  
end

;;-----------------------------------------------------------------------------|
;; A seeker sets a heading using c1-pheno.
to f-seeker-sets-heading
  ;; This routine is to be executed by a seeker.

    ;; The agent will consult its phenotypic characters, to determine the
    ;;   best direction for the next heading.  It is blind, and cannot sense
    ;;   other agents, or patches of fruit.  So the only guidance it has is
    ;;   the genetic information received from its mother, or from the social
    ;;   wisdom learned via its belief and tribal affiliations.
    
    ;; Take note of the current heading.
    let old-heading heading
    
    ;; DECIDE HOW MUCH TO TURN AS DELTA-HEADING.
    ;; Add up the indicators.
    let sum-of-phenos ( round ( sum c1-pheno ) )
    LOG-TO-FILE ( word "  Do-move: Phenos          - " c1-pheno )
    LOG-TO-FILE ( word "  Do-move: Summed Phenos   - " sum-of-phenos )
    ;; Each pheno is derived from genes plus learnings.  The size of
    ;;   the pheno creates a proportional target interval in the sum.  
    ;;   Choose an interval (i.e. choose a pheno) by getting a random number.
    let random-number ( random-float sum-of-phenos )
    LOG-TO-FILE ( word "  Do-move: Random Number   - " random-number )
    ;; The random-number must fall between two sequential pheno aggregates.
    ;; E.g if the random-number is 50%, and the first two pheno numbers 
    ;;   are 13% and 25% (+=38%) then that sum is less than 50%.  If the 
    ;;   next pheno is 20%, then the aggregate is 58%, which is bigger 
    ;;   than 50%.  So it falls between the 2nd and 3rd aggregate of the 
    ;;   phenos.  The third interval then is selected "randomly" with 
    ;;   probability determined by its relative size, among all phenos.
    ;;   Due to zero-based indexing, the correct index is 3-1 = 2.

    let counter 0
    let good-index -1
    let this-pheno 0
    let this-sum 0
    let next-sum 0
    while [ counter < 8 ]  ;; Do not overshoot
    [
      LOG-TO-FILE ( word "  Do-move: Counter         - " counter )
      set this-pheno ( item counter c1-pheno )
      LOG-TO-FILE ( word "  Do-move: This sum        - " this-sum )
      LOG-TO-FILE ( word "  Do-move: Pheno           - " this-pheno )
      set next-sum ( this-pheno + this-sum )
      LOG-TO-FILE ( word "  Do-move: Next sum        - " next-sum )
      if ( ( random-number >= this-sum ) and 
           ( random-number < next-sum  ) )
      [
        set good-index counter
        LOG-TO-FILE ( word "  Do-move: Selected gene   - " good-index )
      ]
      set this-sum next-sum
      set counter ( counter + 1 )
    ]
    let heading-delta ( item good-index gl-heading-list )
    LOG-TO-FILE ( word "  Do-move: Old heading     - " heading )

    ;; Set the new heading
    set heading ( heading + heading-delta )
    LOG-TO-FILE ( word "  Do-move: New heading     - " heading )
  
  ;; End of f-seeker-sets-heading  
end

;;-----------------------------------------------------------------------------|
;; Store data in the lists of sinks.
to f-store-data-in-sink [ sinktype value ]
  ;; This routine is to be executed by anyone.
  
  ;; Record it in the per-tick list.
  let old-value ( item sinktype gl-sinks-per-tick )
  let new-value ( old-value + value )
  set gl-sinks-per-tick ( replace-item sinktype gl-sinks-per-tick new-value )

  ;; Record it in the cumulative list.
  set old-value ( item sinktype gl-sinks-cumulative )
  set new-value ( old-value + value )
  set gl-sinks-cumulative 
    ( replace-item sinktype gl-sinks-cumulative new-value )

  ;; end of f-store-data-in-sink
end

;;-----------------------------------------------------------------------------|
;; Increment the count in the lists of causes of death.
to f-increment-cod-list [ breedtype codtype ]
  ;; This routine is to be executed by anyone.
  
  ;; Record it in the per-tick list.
  let old-count ( item codtype gl-causes-of-death-per-tick )
  let new-count ( old-count + 1 )
  set gl-causes-of-death-per-tick 
    ( replace-item codtype gl-causes-of-death-per-tick new-count )
    
  ;; Record it in the cumulative list.
  set old-count ( item codtype gl-causes-of-death-cumulative )
  set new-count ( old-count + 1 )
  set gl-causes-of-death-cumulative 
    ( replace-item codtype gl-causes-of-death-cumulative new-count )

  ;; End of f-increment-cod-list
end

;;-----------------------------------------------------------------------------|
;; Record the 'energy invested' component of EROI calculation.
to f-record-ei-for-eroi [eroi-ei]
  ;; This routine is to be executed by a seeker.
  
  ;; NOTE: EROI is ER/EI, that is Benefits over Costs, or (B/C).
  ;;       ETA is ER/(ER+EI), that is Benefits over Income, or (B/I).
  
  ;; LOG-TO-FILE ( word "  Do-move: Counter-eroi - " counter-eroi )
  ;; LOG-TO-FILE ( word "  Do-move: Nrg-returned - " nrg-returned )
  ;; LOG-TO-FILE ( word "  Do-move: Nrg-invested - " nrg-invested )
  ;; LOG-TO-FILE ( word "  Do-move: Nrg-income   - " nrg-income )
  ;; LOG-TO-FILE ( word "  Do-move: L-er         - " l-er )
  ;; LOG-TO-FILE ( word "  Do-move: L-ei         - " l-ei )
  
  LOG-TO-FILE ( word "  Do-move: g-sys-ei was    - " gl-sys-nrg-invested )
  ;; First, record it in the system-wide data.
  let last-index ( ( length gl-sys-nrg-invested ) - 1 )
  let old-value ( last gl-sys-nrg-invested )
  let new-value ( old-value + eroi-ei )
  set gl-sys-nrg-invested 
    ( replace-item last-index gl-sys-nrg-invested new-value )
  LOG-TO-FILE ( word "  Do-move: g-sys-ei is now - " gl-sys-nrg-invested )
  
  ;; Check to determine whether we are only appending data to the list,
  ;;   or we are dropping old data and appending new data.
  ifelse ( counter-eroi < g-dt-for-eroi )
  [
    ;; Case of appending new data only.
    ;; Increment the counter - done only in move-related function.
    set counter-eroi ( counter-eroi + 1 )
    ;; Append new entry to last of l-ei.
    set l-ei ( lput eroi-ei l-ei )
    ;; Append a place-holder zero to l-er.
    set l-er ( lput 0 l-er )
  ]
  ;; Else
  [
    ;; Case of dropping/appending data
    ;; Remove oldest entry.
    set l-ei ( butfirst l-ei )
    ;; Append new entry to last of l-ei
    set l-ei ( lput eroi-ei l-ei )
    
    ;; Adjust l-er, removing oldest and appending a place-holder zero.
    ;; Remove oldest entry.
    set l-er ( butfirst l-er )
    ;; Append zero to last of l-er
    set l-er ( lput 0 l-er )
  ]  ;; End else dropping/appending

  ;; Re-calculate the seekers stats.
  set nrg-invested ( sum l-ei )
  set nrg-income ( nrg-returned + nrg-invested )
  set eroi 0
  if (nrg-invested > 0) [ set eroi ( nrg-returned / nrg-invested ) ]
  set eta 0
  if (nrg-income > 0) [ set eta ( nrg-returned / nrg-income ) ]
  
  ;; LOG-TO-FILE ( word "  Do-move: Counter-eroi - " counter-eroi )
  ;; LOG-TO-FILE ( word "  Do-move: Nrg-returned - " nrg-returned )
  ;; LOG-TO-FILE ( word "  Do-move: Nrg-invested - " nrg-invested )
  ;; LOG-TO-FILE ( word "  Do-move: Nrg-income   - " nrg-income )
  ;; LOG-TO-FILE ( word "  Do-move: L-er         - " l-er )
  ;; LOG-TO-FILE ( word "  Do-move: L-ei         - " l-ei )
  
  ;; End of f-record-ei-for-eroi
end

;;-----------------------------------------------------------------------------|
;; D4 – do-feed procedure(s)
;;-----------------------------------------------------------------------------|
to do-feed
  ;; This routine is to be executed by the observer.
  
  if( gb-debug-on = 1 )
  [
    ifelse( ( gs-debug-step-chooser = "all" ) or ( gs-debug-step-chooser = "feed" ) )
    [ set gb-debug-flow-on 1 LOG-TO-FILE "" LOG-TO-FILE word "Do-feed: Debug on; tick = " ticks ]
    [ set gb-debug-flow-on 0 ]
  ]
  
  ;; Agents feed on fruit found in patches.
  ask seekers
  [
    let this-patch patch-here  ;; handle to the patch under the seeker.
    let nrg-available ( [fruit] of this-patch )
    if ( ( nrg < ( EPA - g-nrg-per-deposit ) ) and 
         ( nrg-available > 0 ) )
    [
      ;; Case of there is food to eat.
      ;; Seeker eats.
      set nrg ( nrg + nrg-available )
      let sum-was ( item belief-affiliation gl-nrg-by-belief )
      let sum-is-now ( sum-was + nrg-available )
      set gl-nrg-by-belief 
        ( replace-item belief-affiliation gl-nrg-by-belief sum-is-now )
      f-record-er-for-eroi nrg-available
      ask this-patch
      [
        set fruit 0
        set g-nrg-in-fruit ( g-nrg-in-fruit - nrg-available )
        set pcolor brown
      ]
    ]
    
    ;; A seeker checks for a local guru with similar belief affiliation.
    f-seek-guru-wisdom
    
    ;; A seeker checks for a local tribal elder of the same tribal affiliation.
    f-seek-tribal-wisdom
  ]  ;; End of ask seekers.
  
  ;; Supressed. f-update-aggregates

  LOG-TO-FILE "  Do-feed: procedure completed"
  
  ;; End Do-feed procedure.
end

;;-----------------------------------------------------------------------------|
;; Check if a guru exists locally, and seek a wisdom transfer.
to f-seek-guru-wisdom
  ;; This routine is to be executed by a seeker.
  
  ;; Do this routine for scenarios 2 and 3 only.
  if( g-scenario-number > 1 )
  [
    ;; Case of scenarios 2 and 3.
    ;; Unpack and rename needed characters
    let mywho   who
    let mypatch patch-here
    let myage   age
    let mybelief  belief-affiliation
      
    ;; Select a local guru.
    let belief-elders ( seekers with 
      [ ( who != mywho ) and
        ( patch-here = mypatch ) and
        ( belief-affiliation = mybelief ) and
        ( age >= myage ) ] )

    if( ( count belief-elders ) > 0 )
    [ 
      ;; Case of guru exists.
      LOG-TO-FILE ( word "  Do-feed: wXFer - SEEKING GURU'S WISDOM" ) 
      LOG-TO-FILE ( word "  Do-feed: wXFer - Elders             - " 
        ( [who] of belief-elders ) )
      let maxage ( max [age] of belief-elders )
      let guru ( one-of ( belief-elders with [age = maxage] ) )
      LOG-TO-FILE ( word "  Do-feed: wXFer - Guru found         - " guru )
      LOG-TO-FILE ( word "  Do-feed: wXFer - Belief affiliation - " mybelief ) 
      ;; I estimate that, at steady state, only 1 in ten seekers will 
      ;;   ever experience enlightenment by this routine.
    
      ;; Perform the transfer of wisdom.
      ;;   It is assumed that, since the guru has lived longer, his belief
      ;; has been tested more.  Since the guru is still alive
      ;; that level of belief is to be emulated.
      let mylevel ( item mybelief c1-learn )
      let gurulevel 0
      ask guru [ set gurulevel ( item mybelief c1-learn ) ]
      let level-delta ( ( gurulevel - mylevel ) / 2 )
      let mynewlevel ( mylevel + level-delta )
      set c1-learn ( replace-item mybelief c1-learn mynewlevel )
      LOG-TO-FILE ( word "  Do-feed: wXFer - MyOldLevel - " mylevel 
                                  ", GuruLevel - " gurulevel 
                                  ", LevelDelta - " level-delta 
                                  ", MyNewLevel - " mynewlevel )
    ]  ;; End if guru exists
  ]  ;; End if scenario > 1
  
  ;; End of f-seek-guru-wisdom
end

;;-----------------------------------------------------------------------------|
;; Check if a tribal elder exists locally, and seek a wisdom transfer.
to f-seek-tribal-wisdom
  ;; This routine is to be executed by a seeker.
  
  ;; Do this routine for scenario 3 only.
  if( g-scenario-number > 2 )
  [
    ;; Case of scenario 3.
    ;; Unpack and rename needed characters
    let mywho   who
    let mypatch patch-here
    let myage   age
    let mytribe tribal-affiliation
      
    ;; Select a local tribal elder.
    let tribal-elders ( seekers with 
      [ ( who != mywho ) and
        ( patch-here = mypatch ) and
        ( tribal-affiliation = mytribe ) and
        ( age >= myage ) ] )
    
    if( ( count tribal-elders ) > 0 )
    [ 
      ;; Case of tribal elder exists.
      LOG-TO-FILE ( word "  Do-feed: wXFer - SEEKING TRIBAL WISDOM" ) 
      LOG-TO-FILE ( word "  Do-feed: wXFer - Tribal elders      - " 
        ( [who] of tribal-elders ) )
      let maxage ( max [age] of tribal-elders )
      let tribal-elder ( one-of ( tribal-elders with [age = maxage] ) )
      LOG-TO-FILE ( word "  Do-feed: wXFer - Teacher found      - " tribal-elder )
        
      ;; Transfer from the tribal elder to the seeker.
      let target-gene ( [belief-affiliation] of tribal-elder )
      LOG-TO-FILE ( word "  Do-feed: wXFer - Belief affiliation - " target-gene )
      let mylevel ( item target-gene c1-learn )
      let elderlevel 0
      ask tribal-elder [ set elderlevel ( item target-gene c1-learn ) ]
      let level-delta ( ( elderlevel - mylevel ) / 2 )
      let mynewlevel ( mylevel + level-delta )
      set c1-learn ( replace-item target-gene c1-learn mynewlevel )
      LOG-TO-FILE ( word "  Do-feed: wXFer - MyOldLevel - " mylevel 
                                  ", ElderLevel - " elderlevel 
                                  ", LevelDelta - " level-delta 
                                  ", MyNewLevel - " mynewlevel )
    ]  ;; End case of elder exists
  ]  ;; End Scenario 3 special processing for tribal connections.
  
  ;; End of f-seek-Tribal-wisdom
end

;;-----------------------------------------------------------------------------|
;; Record the 'energy returned' component of EROI calculation.
to f-record-er-for-eroi [eroi-er]
  ;; This routine is to be executed by a seeker.
  
  ;; NOTE: EROI is ER/EI, that is Benefits over Costs, or (B/C).
  ;;       ETA is ER/(ER+EI), that is Benefits over Income, or (B/I).
  
  ;; LOG-TO-FILE ( word "  Do-feed: Counter-eroi - " counter-eroi )
  ;; LOG-TO-FILE ( word "  Do-feed: Nrg-returned - " nrg-returned )
  ;; LOG-TO-FILE ( word "  Do-feed: Nrg-invested - " nrg-invested )
  ;; LOG-TO-FILE ( word "  Do-feed: Nrg-income   - " nrg-income )
  ;; LOG-TO-FILE ( word "  Do-feed: L-er         - " l-er )
  ;; LOG-TO-FILE ( word "  Do-feed: L-ei         - " l-ei )
  
  ;; LOG-TO-FILE ( word "  Do-move: g-sys-er was    - " gl-sys-nrg-returned )
  ;; First, record it in the system-wide data.
  let last-index ( ( length gl-sys-nrg-returned ) - 1 )
  let old-value ( last gl-sys-nrg-returned )
  let new-value ( old-value + eroi-er )
  set gl-sys-nrg-returned 
    ( replace-item last-index gl-sys-nrg-returned new-value )
  ;; LOG-TO-FILE ( word "  Do-move: g-sys-er is now - " gl-sys-nrg-returned )
  
  ;; This routine is called by Do-feed.
  ;; The associated routine f-record-er-for-eroi is called by Do-move.
  ;; The two routines work together to calculate EROI.  
  ;; DO NOT increment the counter here - done only in move-related function.
  ;; set counter-eroi ( counter-eroi + 1 )
  ;; Remove the place-holder zero, put there in the move-related routine.
  set l-er ( butlast l-er )
  ;; Append new entry to last of l-er.
  set l-er ( lput eroi-er l-er )

  ;; Re-calculate the seekers stats.
  set nrg-returned ( sum l-er )
  set nrg-income ( nrg-returned + nrg-invested )
  set eroi 0
  if (nrg-invested > 0) [ set eroi ( nrg-returned / nrg-invested ) ]
  set eta 0
  if (nrg-income > 0) [ set eta ( nrg-returned / nrg-income ) ]
  
  ;; LOG-TO-FILE ( word "  Do-feed: Counter-eroi - " counter-eroi )
  ;; LOG-TO-FILE ( word "  Do-feed: Nrg-returned - " nrg-returned )
  ;; LOG-TO-FILE ( word "  Do-feed: Nrg-invested - " nrg-invested )
  ;; LOG-TO-FILE ( word "  Do-feed: Nrg-income   - " nrg-income )
  ;; LOG-TO-FILE ( word "  Do-feed: L-er         - " l-er )
  ;; LOG-TO-FILE ( word "  Do-feed: L-ei         - " l-ei )
  
  ;; End of f-record-er-for-eroi
end

;;-----------------------------------------------------------------------------|
;; D5 – do-reproduce procedure(s)
;;-----------------------------------------------------------------------------|
to do-reproduce
  ;; This routine is to be executed by the observer.
  
  if( gb-debug-on = 1 )
  [
    ifelse( ( gs-debug-step-chooser = "all" ) or ( gs-debug-step-chooser = "reproduce" ) )
    [ set gb-debug-flow-on 1 LOG-TO-FILE "" LOG-TO-FILE word "Do-rep: Debug on; tick = " ticks ]
    [ set gb-debug-flow-on 0 ]
  ]
  
  ask seekers
  [
    f-set-seeker-repro-flag
    f-reproduce-seeker-by-fission
  ]
  
  ;; Supressed. f-update-aggregates

  LOG-TO-FILE "  Do-rep: procedure completed"
end

;;-----------------------------------------------------------------------------|
;; f-set-seeker-repro-flag
to f-set-seeker-repro-flag
  ;; This routine is to be executed by a seeker.
  
  set b-is-ready-to-reproduce 1        ;; i.e. true is the default.
  if( nrg < RET )
    [ set b-is-ready-to-reproduce 0 ]  ;; i.e. false due to lack of health.
  
  if( age < RAT )
    [ set b-is-ready-to-reproduce 0 ]  ;; i.e. false due to lack of maturity.
  
  if( b-is-ready-to-reproduce = 1 )
  [
    LOG-TO-FILE 
      ( word "  Do-rep: S(age,nrg,rep-flag) - (" 
        age "," floor nrg  "," b-is-ready-to-reproduce ")" )
  ]
    
  ;; End f-set-seeker-repro-flag
end

;;-----------------------------------------------------------------------------|
;; A seeker reproduces via fission, one mother having two daughters.
to f-reproduce-seeker-by-fission
  ;; This routine is to be executed by a seeker.
  
  if( b-is-ready-to-reproduce = 1 )   ;; 1 = true
  [
    LOG-TO-FILE ( word "  Do-rep: seeker Ma - " who )

    let mother self
    let mothers-who who
    let mothers-patch patch-here

    let first-share-of-nrg floor( nrg / 2 )
    let second-share-of-nrg ( nrg - first-share-of-nrg )
  
    let daughter-count 0
    ask mothers-patch
    [
      sprout-seekers 2
      [
        set daughter-count ( daughter-count + 1 )
        LOG-TO-FILE ( word "  Do-rep: seeker D" daughter-count " - " who )
        f-initialize-new-seeker
        set color ( [color] of mother )
        
        ;; Copy the C1 genetic/learned material.
        set c1-bases ( [c1-bases] of mother )
        set c1-genes ( [c1-genes] of mother )
        set c1-learn ( [c1-learn] of mother )
        set c1-stren ( [c1-stren] of mother )
        set c1-pheno ( [c1-pheno] of mother )
        set belief-affiliation ( [belief-affiliation] of mother )
        set tribal-affiliation ( [tribal-affiliation] of mother )
        

        ;; C2 genes are static.
        set DAT ( [DAT] of mother )
        set DET ( [DET] of mother )
        set RAT ( [RAT] of mother )
        set RET ( [RET] of mother )
        set EPM ( [EPM] of mother )
        set EPA ( [EPA] of mother )
        
        ;; Note the mother of this daughter.
        set mas-who ( [who] of mother )
        set age 0

        ifelse ( daughter-count = 1 )
        [ set nrg first-share-of-nrg ]
        [ set nrg second-share-of-nrg ]
        
        set cause-of-death          0
        set b-is-ready-to-move      1
        set b-is-ready-to-reproduce 0
        set b-is-ready-to-die       0
        
        ;; Variables for calculating individual EROI and ETA.
        ;; All inherited from mother.
        set nrg-returned ( [nrg-returned] of mother )
        set nrg-invested ( [nrg-invested] of mother )
        set nrg-income   ( [nrg-income] of mother )
        set eroi         ( [eroi] of mother )
        set eta          ( [eta]  of mother )
        set l-er         ( [l-er] of mother )
        set l-ei         ( [l-ei] of mother )
        set counter-eroi ( [counter-eroi] of mother )

        f-mutate-new-seeker
      ]
    ]
    
    ;; Set the cause of death for the mother.
    set cause-of-death ge-cod-fission
    ;; The mother disappears after fission, leaving two daughters.
    ;; Death actually occurs in the Do-die step.
    
  ]

  ;; End f-reproduce-seeker-by-fission
end

;;-----------------------------------------------------------------------------|
;; A new seeker mutates, changing the genetic basis of strategies.
to f-mutate-new-seeker
  ;; This routine is to be executed by a seeker.
  
  ;; Decide if a genetic mutation is to happen.
  let random-number ( random-float 1 )
  let threshold ( g-prob-of-mutation )
  LOG-TO-FILE 
    ( word "  Do-rep: PreMut (RN, TH)    - (" random-number ", " threshold ")" )
  if ( random-number <= threshold )
  [
    ;; Case of mutation to be done.
    LOG-TO-FILE ( word "  Do-rep: PreMut c1-bases    - " c1-bases )
    LOG-TO-FILE ( word "  Do-rep: PreMut c1-genes    - " c1-genes )
    LOG-TO-FILE ( word "  Do-rep: PreMut c1-learn    - " c1-learn )
    LOG-TO-FILE ( word "  Do-rep: PreMut c1-stren    - " c1-stren )
    LOG-TO-FILE ( word "  Do-rep: PreMut c1-pheno    - " c1-pheno )

    ;; Select the gene to be mutated.  These genes control 
    ;;   the search strategy.
    let gene-to-be-mutated ( random 8 )
    LOG-TO-FILE ( word "  Do-rep: Target gene #      - " gene-to-be-mutated )
    
    ;; Mutate the gene base.
    let oldbase ( item gene-to-be-mutated c1-bases )
    LOG-TO-FILE ( word "  Do-rep: Old base value     - " oldbase )
    ;; Choose a factor for the base.
    let base-factor ( item (random 8) gl-base-factors )
    ;; Mutate it
    let newbase ( oldbase * base-factor )
    LOG-TO-FILE ( word "  Do-rep: Factor             - " base-factor )
    set c1-bases ( replace-item gene-to-be-mutated c1-bases newbase )
    LOG-TO-FILE ( word "  Do-rep: New base value     - " newbase )
    
    ;; Mutate the gene.  The gene value is an integer of either sign.
    let oldgene ( item gene-to-be-mutated c1-genes )
    LOG-TO-FILE ( word "  Do-rep: Old gene value     - " oldgene )
    ;; Decide whether it will increase or decrease in value.
    let delta ( -1 + ( 2 * ( random 2 ) ) )  ;; Either a -1 or a 1.
    LOG-TO-FILE ( word "  Do-rep: Delta              - " delta )
    let newgene ( oldgene + delta )
    set c1-genes ( replace-item gene-to-be-mutated c1-genes newgene )
    LOG-TO-FILE ( word "  Do-rep: New gene value     - " newgene )

    ;; The belief values do not mutate here.

    ;; Calculate the strengths and phenotypic characters.
    f-find-strens-n-phenos
      
    LOG-TO-FILE ( word "  Do-rep: AftMut c1-bases    - " c1-bases )
    LOG-TO-FILE ( word "  Do-rep: AftMut c1-genes    - " c1-genes )
    LOG-TO-FILE ( word "  Do-rep: AftMut c1-learn    - " c1-learn )
    LOG-TO-FILE ( word "  Do-rep: AftMut c1-stren    - " c1-stren )
    LOG-TO-FILE ( word "  Do-rep: AftMut c1-pheno    - " c1-pheno )
  ]
  
  ;; Belief values do not alter in scenarios 0 and 1.
  if( g-scenario-number > 1 )
  [ 
    ;; Decide if a shift of belief value is to happen.
    set random-number ( random-float 1 )
    set threshold ( g-prob-of-belief-drift )
    LOG-TO-FILE 
      ( word "  Do-rep: PreAlt (RN, TH)    - (" random-number ", " threshold ")" )
    if ( random-number <= threshold )
    [
      ;; Case of belief drift to be done.
      LOG-TO-FILE ( word "  Do-rep: PreAlt c1-bases    - " c1-bases )
      LOG-TO-FILE ( word "  Do-rep: PreAlt c1-genes    - " c1-genes )
      LOG-TO-FILE ( word "  Do-rep: PreAlt c1-learn    - " c1-learn )
      LOG-TO-FILE ( word "  Do-rep: PreAlt c1-stren    - " c1-stren )
      LOG-TO-FILE ( word "  Do-rep: PreAlt c1-pheno    - " c1-pheno )

      ;; Alter the belief.  The belief value is an integer 
      ;;   of either sign.
      let myoldbelief ( item belief-affiliation c1-learn )
      LOG-TO-FILE ( word "  Do-rep: Belief Affiliation - " belief-affiliation )
      LOG-TO-FILE ( word "  Do-rep: Old belief value   - " myoldbelief )
      ;; Decide whether it will increase or decrease in value.
      let delta ( -1 + ( 2 * ( random 2 ) ) )  ;; Either -1 or 1.
      LOG-TO-FILE ( word "  Do-rep: Delta              - " delta )
      let mynewbelief ( myoldbelief + delta )
      LOG-TO-FILE ( word "  Do-rep: New belief value   - " mynewbelief )
      set c1-learn ( replace-item belief-affiliation c1-learn mynewbelief )

      ;; Calculate the strengths and phenotypic characters.
      f-find-strens-n-phenos
      
      LOG-TO-FILE ( word "  Do-rep: AftAlt c1-bases    - " c1-bases )
      LOG-TO-FILE ( word "  Do-rep: AftAlt c1-genes    - " c1-genes )
      LOG-TO-FILE ( word "  Do-rep: AftAlt c1-learn    - " c1-learn )
      LOG-TO-FILE ( word "  Do-rep: AftAlt c1-stren    - " c1-stren )
      LOG-TO-FILE ( word "  Do-rep: AftAlt c1-pheno    - " c1-pheno )
    ]
  ]
  
  ;; End of f-mutate-new-seeker
end
  
;;-----------------------------------------------------------------------------|
;; D6 – do-die procedure(s)
;;-----------------------------------------------------------------------------|
to do-die
  ;; This routine is to be executed by the observer.
  
  if( gb-debug-on = 1 )
  [
    ifelse( ( gs-debug-step-chooser = "all" ) or ( gs-debug-step-chooser = "die" ) )
    [ set gb-debug-flow-on 1 LOG-TO-FILE "" LOG-TO-FILE word "Do-die: Debug on; tick = " ticks ]
    [ set gb-debug-flow-on 0 ]
  ]
  
  if( ( count seekers ) > 0 )
  [
    ask seekers
    [
      f-set-seeker-death-flag
      f-seeker-dies
    ]
  ]
  
  ;; Supressed. f-update-aggregates

  LOG-TO-FILE "  Do-die: procedure completed"
end

;;-----------------------------------------------------------------------------|
;; f-set-seeker-death-flag
to f-set-seeker-death-flag
  ;; This routine is to be executed by a seeker.
  
  set b-is-ready-to-die 0        ;; i.e. false, default.

  ;; If a cause of death has already been noted, it dies.
  ifelse( cause-of-death > ge-cod-none )
  [
    ;; A cause of death has been previously flagged.
    ;; This is either due to hunger (in do-move) or fission (in do-repro).
    ;; In both cases nrg has been stripped out already.
    ;; In the cases of DET and DAT, the flag is not yet set, 
    ;;   and the nrg remains.
    set b-is-ready-to-die 1 
  ]
  ;; Else
  [
    ;; No cause of death has been set yet.  Check basic vital signs.
    if( nrg <= DET )  ;; Death Energy Threshold.
    [ 
      set b-is-ready-to-die 1 
      set cause-of-death ge-cod-hunger 
      f-store-data-in-sink ge-sinktype-die-det nrg
      let sum-was ( item belief-affiliation gl-nrg-by-belief )
      let sum-is-now ( sum-was - nrg )
      set gl-nrg-by-belief 
        ( replace-item belief-affiliation gl-nrg-by-belief sum-is-now )
      set g-total-nrg-in-system ( g-total-nrg-in-system - nrg )
      set nrg 0
    ]
    
    if( age > DAT )   ;; Death Age Threshold.
    [ 
      set b-is-ready-to-die 1 
      set cause-of-death ge-cod-oldage 
      f-store-data-in-sink ge-sinktype-die-dat nrg
      let sum-was ( item belief-affiliation gl-nrg-by-belief )
      let sum-is-now ( sum-was - nrg )
      set gl-nrg-by-belief 
        ( replace-item belief-affiliation gl-nrg-by-belief sum-is-now )
      set g-total-nrg-in-system ( g-total-nrg-in-system - nrg )
      set nrg 0
    ]
  ]

  if( b-is-ready-to-die = 1 )
  [
    LOG-TO-FILE ( WORD "  Do-die: S(age,nrg,cod) - ("
      age "," 
      nrg "," 
      cause-of-death ")" )
  ]
  
  ;; End f-set-seeker-death-flag
end

;;-----------------------------------------------------------------------------|
;; f-seeker-dies
to f-seeker-dies
  ;; This routine is to be executed by a seeker.
  
  if( b-is-ready-to-die = 1 )
  [
    ;; Nrg was stripped out in do-move step.
    ;; However, nrg may exist for those who die of old age.
    f-increment-cod-list breed cause-of-death
    die  ;; The seeker disappears from the system.    
  ]

  ;; End f-seeker-dies
end

;;-----------------------------------------------------------------------------|
;; D7 - do-post-tick procedure(s)
;;-----------------------------------------------------------------------------|
to do-post-tick
  ;; This routine is to be executed by the observer.
   
  if( gb-debug-on = 1 )
  [
    ifelse( ( gs-debug-step-chooser = "all" ) or ( gs-debug-step-chooser = "post-tick" ) )
    [ set gb-debug-flow-on 1  LOG-TO-FILE "" LOG-TO-FILE word "Do-Post-tick: Debug on; tick = " ticks ]
    [ set gb-debug-flow-on 0 ]
  ]
  
  ;; MANUAL CHANGE FOR DEBUG.
  ;; This is a call to a debug routine which could be suppressed if all is okay.
  ;; This is one of a group of such calls, most of which are between steps in 
  ;;   the 'Go' routine.  They are suppressed there, but can be enabled again.
  ;; I have decided to leave this one active, for now.
  ;; It checks all agents, every tick, to ensure that all values are greater than
  ;;   or equal to zero.  
  if( frb-agents-are-all-valid = false ) 
    [ LOG-TO-FILE ( word "  Do-post-tick: Agents failed validity test." ) ]
  
  ;; Global EROI system-wide calculations.
  set g-sys-nrg-returned 
    ( sum gl-sys-nrg-returned ) ;; Total nrg returned within delta T.
  set g-sys-nrg-invested 
    ( sum gl-sys-nrg-invested ) ;; Total nrg invested within delta T.
  set g-sys-nrg-income ( g-sys-nrg-returned + g-sys-nrg-invested )
  set g-sys-eroi 0 ;; System-wide EROI, per tick.
  if( g-sys-nrg-invested != 0 ) 
    [ set g-sys-eroi ( g-sys-nrg-returned / g-sys-nrg-invested ) ]
  set g-sys-eta 0 ;; System-wide ETA, per tick.
  if( g-sys-nrg-income != 0 ) 
    [ set g-sys-eta ( g-sys-nrg-returned / g-sys-nrg-income ) ]

  ;; Update the aggregates for display in the monitors.
  f-update-aggregates

  display

  LOG-TO-FILE "  Do-post-tick: procedure completed."
end

;;-----------------------------------------------------------------------------|
;; SECTION E – DRAWING AND MAINTENANCE PROCEDURE(S)
;;-----------------------------------------------------------------------------|

;;-----------------------------------------------------------------------------|
;; Update the values of global aggregate numbers.
to f-update-aggregates
  ;; This routine is to be executed by the observer.

  ;; Although this is a display-only routine, it may implicitly call the 
  ;;   PRNG and so may have an effect on the trajectory of the model.  In a 
  ;;   standard 'go' run it is called only once per tick, before graphs are 
  ;;   updated.  If you use the one-step debug buttons, it is called once 
  ;;   after each step, so debug runs that use those buttons will not 
  ;;   replicate a real run.
  
  ;;---------------------------------------------------------------------------|
  ;; The following agent sets, counts and averages are for data collection 
  ;;   and display in monitors and plots.
  
  ;; Counts
  set g-no-of-patches ( count patches )
  foreach gl-index-list
  [
    set gl-no-of-seekers 
    ( replace-item ? gl-no-of-seekers 
      ( count seekers with [belief-affiliation = ?] ) ) 
  ]
  set g-no-of-seekers ( sum gl-no-of-seekers )
  ;; For the 8-beliefs society, track those with targeted belief affiliation.
  set g-no-with-this-belief 
    ( count seekers with [belief-affiliation = g-gene-to-adjust] )
  
  ;; Averages - seekers
  ifelse( 0 = ( count seekers ) )
  [
    set g-ave-age      0        ;; age of seekers
    set g-ave-nrg      0        ;; nrg of seekers
    set g-ind-min-eroi 1        ;; eroi of seekers
    set g-ind-ave-eroi 1        ;; eroi of seekers
    set g-ind-max-eroi 1        ;; eroi of seekers
    set g-ind-min-eta  0.5      ;; eta of seekers
    set g-ind-ave-eta  0.5      ;; eta of seekers
    set g-ind-max-eta  0.5      ;; eta of seekers

    set g-ave-C1-b0   0         ;; c1, base character, gene-0
    set g-ave-C1-b1   0         ;; c1, base character, gene-1
    set g-ave-C1-b2   0         ;; c1, base character, gene-2
    set g-ave-C1-b3   0         ;; c1, base character, gene-3
    set g-ave-C1-b4   0         ;; c1, base character, gene-4
    set g-ave-C1-b5   0         ;; c1, base character, gene-5
    set g-ave-C1-b6   0         ;; c1, base character, gene-6
    set g-ave-C1-b7   0         ;; c1, base character, gene-7
 
    set g-ave-C1-g0   0         ;; c1, genotypic character, gene-0
    set g-ave-C1-g1   0         ;; c1, genotypic character, gene-1
    set g-ave-C1-g2   0         ;; c1, genotypic character, gene-2
    set g-ave-C1-g3   0         ;; c1, genotypic character, gene-3
    set g-ave-C1-g4   0         ;; c1, genotypic character, gene-4
    set g-ave-C1-g5   0         ;; c1, genotypic character, gene-5
    set g-ave-C1-g6   0         ;; c1, genotypic character, gene-6
    set g-ave-C1-g7   0         ;; c1, genotypic character, gene-7
 
    set g-ave-C1-l0   0         ;; c1, learned character, gene-0
    set g-ave-C1-l1   0         ;; c1, learned character, gene-1
    set g-ave-C1-l2   0         ;; c1, learned character, gene-2
    set g-ave-C1-l3   0         ;; c1, learned character, gene-3
    set g-ave-C1-l4   0         ;; c1, learned character, gene-4
    set g-ave-C1-l5   0         ;; c1, learned character, gene-5
    set g-ave-C1-l6   0         ;; c1, learned character, gene-6
    set g-ave-C1-l7   0         ;; c1, learned character, gene-7

    set g-ave-C1-s0   0         ;; c1, strength character, gene-0
    set g-ave-C1-s1   0         ;; c1, strength character, gene-1
    set g-ave-C1-s2   0         ;; c1, strength character, gene-2
    set g-ave-C1-s3   0         ;; c1, strength character, gene-3
    set g-ave-C1-s4   0         ;; c1, strength character, gene-4
    set g-ave-C1-s5   0         ;; c1, strength character, gene-5
    set g-ave-C1-s6   0         ;; c1, strength character, gene-6
    set g-ave-C1-s7   0         ;; c1, strength character, gene-7

    set g-ave-C1-p0   0         ;; c1, phenotypic character, gene-0
    set g-ave-C1-p1   0         ;; c1, phenotypic character, gene-1
    set g-ave-C1-p2   0         ;; c1, phenotypic character, gene-2
    set g-ave-C1-p3   0         ;; c1, phenotypic character, gene-3
    set g-ave-C1-p4   0         ;; c1, phenotypic character, gene-4
    set g-ave-C1-p5   0         ;; c1, phenotypic character, gene-5
    set g-ave-C1-p6   0         ;; c1, phenotypic character, gene-6
    set g-ave-C1-p7   0         ;; c1, phenotypic character, gene-7
    
    set gl-nrg-by-tribe [ 0 0 0 0 0 0 0 0 0 0 ] ;; Max of 10 tribes
  ]
  ;; Else
  [
    set g-ave-age   ( sum [age] of seekers ) / g-no-of-seekers
    set g-ave-nrg   ( sum [nrg] of seekers ) / g-no-of-seekers
    set g-ind-min-eroi  ( min [eroi] of seekers )
    set g-ind-ave-eroi  ( sum [eroi] of seekers ) / g-no-of-seekers
    set g-ind-max-eroi  ( max [eroi] of seekers )
    set g-ind-min-eta  ( min [eta] of seekers )
    set g-ind-ave-eta  ( sum [eta] of seekers ) / g-no-of-seekers
    set g-ind-max-eta  ( max [eta] of seekers )

    set g-ave-C1-b0 ( f-compute-C1-bases-average 0 )
    set g-ave-C1-b1 ( f-compute-C1-bases-average 1 )
    set g-ave-C1-b2 ( f-compute-C1-bases-average 2 )
    set g-ave-C1-b3 ( f-compute-C1-bases-average 3 )
    set g-ave-C1-b4 ( f-compute-C1-bases-average 4 )
    set g-ave-C1-b5 ( f-compute-C1-bases-average 5 )
    set g-ave-C1-b6 ( f-compute-C1-bases-average 6 )
    set g-ave-C1-b7 ( f-compute-C1-bases-average 7 )

    set g-ave-C1-g0 ( f-compute-C1-genes-average 0 )
    set g-ave-C1-g1 ( f-compute-C1-genes-average 1 )
    set g-ave-C1-g2 ( f-compute-C1-genes-average 2 )
    set g-ave-C1-g3 ( f-compute-C1-genes-average 3 )
    set g-ave-C1-g4 ( f-compute-C1-genes-average 4 )
    set g-ave-C1-g5 ( f-compute-C1-genes-average 5 )
    set g-ave-C1-g6 ( f-compute-C1-genes-average 6 )
    set g-ave-C1-g7 ( f-compute-C1-genes-average 7 )

    set g-ave-C1-l0 ( f-compute-c1-learn-average 0 )
    set g-ave-C1-l1 ( f-compute-c1-learn-average 1 )
    set g-ave-C1-l2 ( f-compute-c1-learn-average 2 )
    set g-ave-C1-l3 ( f-compute-c1-learn-average 3 )
    set g-ave-C1-l4 ( f-compute-c1-learn-average 4 )
    set g-ave-C1-l5 ( f-compute-c1-learn-average 5 )
    set g-ave-C1-l6 ( f-compute-c1-learn-average 6 )
    set g-ave-C1-l7 ( f-compute-c1-learn-average 7 )

    set g-ave-C1-s0 ( f-compute-c1-stren-average 0 )
    set g-ave-C1-s1 ( f-compute-c1-stren-average 1 )
    set g-ave-C1-s2 ( f-compute-c1-stren-average 2 )
    set g-ave-C1-s3 ( f-compute-c1-stren-average 3 )
    set g-ave-C1-s4 ( f-compute-c1-stren-average 4 )
    set g-ave-C1-s5 ( f-compute-c1-stren-average 5 )
    set g-ave-C1-s6 ( f-compute-c1-stren-average 6 )
    set g-ave-C1-s7 ( f-compute-c1-stren-average 7 )

    set g-ave-C1-p0 ( f-compute-c1-pheno-average 0 )
    set g-ave-C1-p1 ( f-compute-c1-pheno-average 1 )
    set g-ave-C1-p2 ( f-compute-c1-pheno-average 2 )
    set g-ave-C1-p3 ( f-compute-c1-pheno-average 3 )
    set g-ave-C1-p4 ( f-compute-c1-pheno-average 4 )
    set g-ave-C1-p5 ( f-compute-c1-pheno-average 5 )
    set g-ave-C1-p6 ( f-compute-c1-pheno-average 6 )
    set g-ave-C1-p7 ( f-compute-c1-pheno-average 7 )
    
    ;; Make a list of existing tribal-affiliation numbers.,
    let tribe-list ( sort remove-duplicates ( [tribal-affiliation] of seekers ) )
    
    set gl-nrg-by-tribe [ 0 0 0 0 0 0 0 0 0 0 ] ;; Max of 10 tribes
    foreach tribe-list
    [
      let total-nrg ( sum [nrg] of seekers with [tribal-affiliation = ?] )
      set gl-nrg-by-tribe 
        ( replace-item ? gl-nrg-by-tribe total-nrg )
    ]
  ]
  ;; End else

  
;;-----------------------------------------------------------------------------|
  ;; To ensure that the PRNG is called whether or not plots are displayed, the
  ;;   calculations needed for any histogram plots which invoke the PRNG
  ;;   implicitly should be carried out here where they will happen every tick.

;;-----------------------------------------------------------------------------|
  ;; Setup for Plot "AAAAAA"
  
  ;; This log entry may come from any step during debug operations.  
  LOG-TO-FILE "  Do-update: All aggregates updated."  

  ;; End of f-update-aggregates
end

;;-----------------------------------------------------------------------------|
;; Compute an average for C1-bases, by preferred gene type.
to-report f-compute-C1-bases-average [ gene-to-check ]
  ;; This routine is to be executed by the observer.
  let count-of-seekers ( count seekers )
  let appropriate-sum ( sum ( [item gene-to-check c1-bases] of seekers ) )
  let answer 0
  if ( count-of-seekers > 0 ) 
    [ set answer ( appropriate-sum / count-of-seekers ) ]
  ;; LOG-TO-FILE ( word "  Do-update: g# - " gene-to-check ", ave [B] - " answer ) 
  
  report answer
  
  ;; End of f-compute-C1-bases-average
end

;;-----------------------------------------------------------------------------|
;; Compute an average for C1-genes, by preferred gene type.
to-report f-compute-C1-genes-average [ gene-to-check ]
  ;; This routine is to be executed by the observer.
  let count-of-seekers ( count seekers )
  let appropriate-sum ( sum ( [item gene-to-check c1-genes] of seekers ) )
  let answer 0
  if ( count-of-seekers > 0 ) 
    [ set answer ( appropriate-sum / count-of-seekers ) ]
  ;; LOG-TO-FILE ( word "  Do-update: g# - " gene-to-check ", ave [G] - " answer ) 
  
  report answer
  
  ;; End of f-compute-C1-genes-average
end

;;-----------------------------------------------------------------------------|
;; Compute an average for c1-learn, by preferred gene type.
to-report f-compute-c1-learn-average [ gene-to-check ]
  ;; This routine is to be executed by the observer.
  let count-of-seekers ( count seekers )
  let appropriate-sum ( sum ( [item gene-to-check c1-learn] of seekers ) )
  let answer 0
  if ( count-of-seekers > 0 ) 
    [ set answer ( appropriate-sum / count-of-seekers ) ]
  ;; LOG-TO-FILE ( word "  Do-update: g# - " gene-to-check ", ave [M] - " answer ) 
  
  report answer
  
  ;; End of f-compute-c1-learn-average
end

;;-----------------------------------------------------------------------------|
;; Compute an average for c1-stren, by preferred gene type.
to-report f-compute-c1-stren-average [ gene-to-check ]
  ;; This routine is to be executed by the observer.
  let count-of-seekers ( count seekers )
  let appropriate-sum ( sum ( [item gene-to-check c1-stren] of seekers ) )
  let answer 0
  if ( count-of-seekers > 0 ) 
    [ set answer ( appropriate-sum / count-of-seekers ) ]
  ;; LOG-TO-FILE ( word "  Do-update: g# - " gene-to-check ", ave [S] - " answer ) 
  
  report answer
  
  ;; End of f-compute-c1-stren-average
end

;;-----------------------------------------------------------------------------|
;; Compute an average for c1-pheno, by preferred gene type.
to-report f-compute-c1-pheno-average [ gene-to-check ]
  ;; This routine is to be executed by the observer.
  let count-of-seekers ( count seekers )
  let appropriate-sum ( sum ( [item gene-to-check c1-pheno] of seekers ) )
  let answer 0
  if ( count-of-seekers > 0 ) 
    [ set answer ( appropriate-sum / count-of-seekers ) ]
  ;; LOG-TO-FILE ( word "  Do-update: g# - " gene-to-check ", ave [P] - " answer ) 
  
  report answer
  
  ;; End of f-compute-c1-pheno-average
end

;;--------------------------
;; DATA CAPTURE TO CSV FILES
;;--------------------------

;;-----------------------------------------------------------------------------|
;; Record the data is several selected plots to CSV files
to f-record-selected-plots
  ;; This routine is to be executed by the observer.
  
  ;; The template for the export command is:
  ;; export-plot plotname filename

  ;; Get a common timestamp for all plots.
  let timestamp fr-get-time-stamp
  
  ;; Plot 01
  let plotname "Nrg - By Belief Affiliation"
  let plot-filename 
    ( word timestamp 
           "_Sc" g-scenario-number 
           "_Se" g-use-this-seed 
           "_Pl01_NBBA.CSV" )
  export-plot plotname plot-filename
  
  ;; Plot 02
  set plotname "Nrg - By Tribal Affiliation"
  set plot-filename 
    ( word timestamp 
           "_Sc" g-scenario-number 
           "_Se" g-use-this-seed 
           "_Pl02_NBTA.CSV" )
  export-plot plotname plot-filename
  
  ;; Plot 03
  set plotname "[B]ase Values By Gene #"
  set plot-filename 
    ( word timestamp 
           "_Sc" g-scenario-number 
           "_Se" g-use-this-seed 
           "_Pl03_BVBG.CSV" )
  export-plot plotname plot-filename
  
  ;; Plot 04
  set plotname "[G]ene Values By Gene #"
  set plot-filename 
    ( word timestamp 
           "_Sc" g-scenario-number 
           "_Se" g-use-this-seed 
           "_Pl04_GVBG.CSV" )
  export-plot plotname plot-filename
  
  ;; Plot 05
  set plotname "[L]earned Beliefs By Gene #"
  set plot-filename 
    ( word timestamp 
           "_Sc" g-scenario-number 
           "_Se" g-use-this-seed 
           "_Pl05_LVBG.CSV" )
  export-plot plotname plot-filename
  
  ;; Plot 06
  set plotname "[S]trengths By Gene #"
  set plot-filename 
    ( word timestamp 
           "_Sc" g-scenario-number 
           "_Se" g-use-this-seed 
           "_Pl06_SVBG.CSV" )
  export-plot plotname plot-filename
  
  ;; Plot 06
  set plotname "[P]henotype Values By Gene #"
  set plot-filename 
    ( word timestamp 
           "_Sc" g-scenario-number 
           "_Se" g-use-this-seed 
           "_Pl07_PVBG.CSV" )
  export-plot plotname plot-filename
  
  ;; End f-record-selected-plots
end

;;-----------------------------------------------------------------------------|
;; Construct a time stamp for a file name for data in CSV format.
to-report fr-get-time-stamp
  ;; This routine is to be executed by the observer.
  ;;
  ;; Date-string format "01:19:36.685 PM 19-Sep-2002"
  let date-string date-and-time
  let time-stamp ""
  ;; Append the year as yy.
  set time-stamp word time-stamp ( substring date-string 25 27 )
  ;; Append the month as Mmm.
  set time-stamp word time-stamp fr-convert-mmm-mm ( substring date-string 19 22 )
  ;; Append the day as dd.
  set time-stamp word time-stamp ( substring date-string 16 18 )
  ;; Append a dash.
  set time-stamp word time-stamp "_"

  ;; Append the hour as hh.
  set time-stamp word time-stamp fr-convert1224 ( substring date-string 0 2 ) ( substring date-string 13 15 )
  ;; Append the minute as mm.
  set time-stamp word time-stamp ( substring date-string 3 5 )
  ;; Append the second as ss.
  set time-stamp word time-stamp ( substring date-string 6 8 )

  report time-stamp
  ;; End fr-get-time-stamp
end

;;-----------------------------------------------------------------------------|
;; DEBUG AND DEBUG LOG FILE MANAGEMENT FUNCTIONS
;;-----------------------------------------------------------------------------|

;;-----------------------------------------------------------------------------|
;; Open a log file for debug output.
to f-open-log-file
  ;; This routine is to be executed by the observer.
  
  ;; Ensure previous log file is closed.
  if ( is-string? gs-log-file-name )
  [
    if ( file-exists? gs-log-file-name )
    [
      file-close-all
    ]
  ]
  
  ;; Date-string format "01:19:36.685 PM 19-Sep-2002"
  let date-string date-and-time
  set gs-log-file-name "TpLab_Log_"
  ;; Append the year as yy.
  set gs-log-file-name word gs-log-file-name ( substring date-string 25 27 )
  ;; Append the month as Mmm.
  set gs-log-file-name word gs-log-file-name fr-convert-mmm-mm ( substring date-string 19 22 )
  ;; Append the day as dd.
  set gs-log-file-name word gs-log-file-name ( substring date-string 16 18 )
  ;; Append a dash.
  set gs-log-file-name word gs-log-file-name "_"

  ;; Append the hour as hh.
  set gs-log-file-name word gs-log-file-name fr-convert1224 ( substring date-string 0 2 ) ( substring date-string 13 15 )
  ;; Append the minute as mm.
  set gs-log-file-name word gs-log-file-name ( substring date-string 3 5 )
  ;; Append the second as ss.
  set gs-log-file-name word gs-log-file-name ( substring date-string 6 8 )
  ;; Append the .txt extension.
  set gs-log-file-name word gs-log-file-name ".txt"

  file-open gs-log-file-name
  file-show "Log File for a TpLab (NetLogo) Model."
  file-show word "File Name: " gs-log-file-name
  file-show word "File opened at:" date-and-time
  file-show ""
  
  ;; Send a message directly to the command centre.
  ifelse ( file-exists? gs-log-file-name )
  [
    show word gs-log-file-name " opened."
  ]
  [
    show word gs-log-file-name " not opened."
  ]
end

;;-----------------------------------------------------------------------------|
;; Convert month in text form to digital form.
to-report fr-convert-mmm-mm [ mmm ]
  ;; This routine is to be executed by the observer.
  ;; It converts a string in the form mmm ( alpha text ) to the form mm ( digit-text ).
  
  let mm "00"
  if( mmm = "Jan" ) [ set mm "01" ]
  if( mmm = "Feb" ) [ set mm "02" ]
  if( mmm = "Mar" ) [ set mm "03" ]
  if( mmm = "Apr" ) [ set mm "04" ]
  if( mmm = "May" ) [ set mm "05" ]
  if( mmm = "Jun" ) [ set mm "06" ]
  if( mmm = "Jul" ) [ set mm "07" ]
  if( mmm = "Aug" ) [ set mm "08" ]
  if( mmm = "SeP" ) [ set mm "09" ]
  if( mmm = "Oct" ) [ set mm "10" ]
  if( mmm = "Nov" ) [ set mm "11" ]
  if( mmm = "Dec" ) [ set mm "12" ]
  report mm
end

;;-----------------------------------------------------------------------------|
;; Convert hour in 12 format to 24 hour format.
to-report fr-convert1224 [ hh ampm ]
  ;; This routine is to be executed by the observer.
  ;; It converts a string in 12 hour format to 24 hour format.
  
  let hour read-from-string hh
  if( ampm = "PM" ) [ set hour ( hour + 12 ) ]
  
  let dd ( word "00" hour )
  let d2 last dd
  set dd but-last dd
  let d1 last dd
  set dd ( word d1 d2 )
  report dd
end

;;-----------------------------------------------------------------------------|
;; Close a log file for debug output.
to f-close-log-file
  ;; This routine is to be executed by the observer.
  
  let b-filename-exists 0
  if ( is-string? gs-log-file-name ) 
  [
    if ( file-exists? gs-log-file-name )
    [
      set b-filename-exists 1
    ]
  ] 

  ifelse( b-filename-exists = 1 )
  [
    ;; Ensure the file is selected.
    file-open gs-log-file-name
      
    ;; Stanp it.
    LOG-TO-FILE word "File closed at: " date-and-time
      
    ;; Flush the buffers.
    file-flush 
      
    ;; Close it.
    file-close-all
      
    ;; Note sent to command centre.
    show word gs-log-file-name " closed."
    
    ;; Revert to dummy name.
    set gs-log-file-name "dummyname"
  ]
  [
    if( gs-log-file-name = "dummyname" )
      [ show "No log file is open.  Cannot close it." ]
  ]
end

;;-----------------------------------------------------------------------------|
;; Select an already opened log file.
to f-select-log-file
  ;; This routine is to be executed by the observer.
  
  ifelse ( file-exists? gs-log-file-name )
  [
    ;; Ensure the file is selected.
    file-open gs-log-file-name
    
    ;; Ensure it is open for writing.
    LOG-TO-FILE ""
    LOG-TO-FILE "SELECTED"    
  ]
  [
    show word gs-log-file-name " is not open.  Cannot select it."
  ]
end

;;-----------------------------------------------------------------------------|
;; Change the debug mode from on to off, or vice versa.
to f-toggle-debug
  ;; This routine is to be executed by the observer, and is activated by a 
  ;;   button.
  
  ifelse( gb-debug-on = 1 )
  [
    ;; Debug is On, turn it Off.
    ;; Close the file before turning debug logging off.
    f-close-log-file
    set gs-debug-status "0 (Off)"  ;; This appears in the monitor.
    set gb-debug-on 0              ;; But this controls the debug feature.
  ]
  [
    ;; Debug is Off, turn it On.
    set gs-debug-status "1 (On)"   ;; This appears in the monitor.
    set gb-debug-on 1              ;; But this controls the debug feature.
    ;; The switches, if needed, are reset manually by the user.
    ;; Open the log file after turning debug logging on.
    f-open-log-file
  ]
end

;;-----------------------------------------------------------------------------|
;; 'Show' a string in a debug log.
to LOG-TO-FILE [ log-this-string ]
  ;; This routine may be executed by observer or seeker.
  ;; It should be invoked as a debug routine only, and would not be used for 
  ;;    normal output.  It sends output to the debug log file, or, optionally,
  ;;    also to the command centre.
  
  ;; gb-debug-on is a global Boolean and has value 1 (true) or 0 (false).
  if( gb-debug-on = 1 )
  [
    ;; gb-debug-flow-on is declared as a global Boolean variable, and its value 
    ;;   is 0 ( false ) or 1 ( true ) and is set on or off at the beginning of each 
    ;;   function ( each do-step ).  It is controlled by the chooser that selects 'all' 
    ;;   or a specific do-function.
    ;; 
    ;; When it is 'on' you can assume the debug log file exists and is open for
    ;;   write.
    
    if( gb-debug-flow-on = 1 )
    [
      file-show log-this-string
      show log-this-string
    ] 
  ]
end

;;-----------------------------------------------------------------------------|
;; This replicates the effect of an 'ASSERTION' in C++
to ASSERT [ error-test error-string error-who ]
;; This routine can be run by observer or seeker.

if( error-test = false )
[
  show ( word error-test " " error-string " " error-who )
  ;; Cause a run-time error and display a message.
  error ( word "Agent: " error-who " - " error-string )
]

end


;;-----------------------------------------------------------------------------|
;; Check whether the nrg accounts are all valid.
to-report frb-nrg-accounts-are-all-valid 
;; This routine can be run by the observer.

  let b-accounts-are-all-valid 1
  
  if( gb-debug-on = 1 )
  [
    ;; Do the check only if debug is on.
    let temp-nrg-in-seekers  ( sum [nrg] of seekers )
    let temp-nrg-in-fruit    ( sum [fruit] of patches )
    let temp-nrg-in-sunshine g-nrg-in-sunshine
    
    let temp-total-nrg ( temp-nrg-in-seekers +
      temp-nrg-in-fruit +
      temp-nrg-in-sunshine )
    
    if (temp-nrg-in-seekers != ( sum gl-nrg-by-belief ) )
    [ 
      set b-accounts-are-all-valid 0 
      LOG-TO-FILE ( word 
        "S-nrg-check: SB:" temp-nrg-in-seekers 
        ", IS:" ( sum gl-nrg-by-belief ) )    
    ]
    
    if (temp-nrg-in-fruit != g-nrg-in-fruit )
    [ 
      set b-accounts-are-all-valid 0 
      LOG-TO-FILE ( word 
        "F-nrg-check: SB:" temp-nrg-in-fruit 
        ", IS:" g-nrg-in-fruit )    
    ]
    
    if (temp-total-nrg != g-total-nrg-in-system )
    [ 
      set b-accounts-are-all-valid 0 
      LOG-TO-FILE ( word 
        "T-nrg-check: SB:" temp-total-nrg 
        ", IS:" g-total-nrg-in-system )    
    ]
  ]
  
  report b-accounts-are-all-valid
  
  ;; End of frb-nrg-accounts-are-all-valid
end

;;-----------------------------------------------------------------------------|
;; Check whether the agents are all valid.
to-report frb-agents-are-all-valid 
;; This routine can be run by the observer.

  let b-agents-are-all-valid true
  
  if( gb-debug-on = 1 )
  [
    ;; Do the check only if debug is on.
    
    ;; Check the seekers.
    ask seekers
    [
      if( frb-seeker-is-valid = false ) [ set b-agents-are-all-valid false ]
    ]
  ]
  
  report b-agents-are-all-valid
  
  ;; End of frb-agents-are-all-valid
end

;;-----------------------------------------------------------------------------|
;; Check whether a seeker is valid.
to-report frb-seeker-is-valid 
;; This routine can be run by a seeker.

  let b-seeker-is-valid true
  
  report b-seeker-is-valid
  
  ;; End of frb-seeker-is-valid
end
@#$#@#$#@
GRAPHICS-WINDOW
245
13
1165
454
45
20
10.0
1
10
1
1
1
0
0
0
1
-45
45
-20
20
1
1
1
Ecological Arena - Ticks
30.0

BUTTON
19
396
193
430
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
19
359
194
392
g-use-this-seed
g-use-this-seed
1
100
7
1
1
NIL
HORIZONTAL

BUTTON
325
493
393
526
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
325
459
393
492
One Tick
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
75
11
160
48
TpLab
28
15.0
1

TEXTBOX
7227
13
7807
40
AVERAGE VALUES - C1 GENES - BY COMPONENT, BY GENE #.
20
55.0
1

TEXTBOX
8
45
237
93
'TELEOLOGICAL PRUNING'\n         LABORATORY
18
0.0
1

TEXTBOX
18
155
231
282
QUICK START INSTRUCTIONS\n1. Select a scenario.\n2. Select a PRNG seed.\n3. Optionally, set the advanced switches\n     found on PANEL 02.\n4. Push the \"Setup\" button.\n5. Adjust the speed.\n6. Push the \"Go\" button.\n
11
0.0
1

TEXTBOX
136
478
214
510
OPERATIONS CONTROLS
11
0.0
1

BUTTON
2521
360
2576
393
Move
let temp1 gs-debug-step-chooser\nset gs-debug-step-chooser \"move\"\ndo-move\nf-update-aggregates\nset gs-debug-step-chooser temp1\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
2757
360
2812
393
Post-Tick
let temp1 gs-debug-step-chooser\nset gs-debug-step-chooser \"post-tick\"\ndo-post-tick\n;; f-update-aggregates ;; Redundant.\nset gs-debug-step-chooser temp1\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
2411
294
2466
327
Setup
let temp1 gs-debug-step-chooser\nset gs-debug-step-chooser \"setup\"\nsetup\nset gs-debug-step-chooser temp1\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
2410
360
2465
393
Pre-Tick
let temp1 gs-debug-step-chooser\nset gs-debug-step-chooser \"pre-tick\"\ndo-pre-tick\nf-update-aggregates\nset gs-debug-step-chooser temp1\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
396
459
466
525
g-halt-at-tick
-1
1
0
Number

CHOOSER
19
311
193
356
gs-scenario
gs-scenario
"Society With 0 Beliefs" "Society With 8 Beliefs" "Society With Wisdom Sharing" "Society With Tribal Bonds"
2

BUTTON
2467
294
2544
327
One Tick
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
2544
294
2607
327
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
2412
134
2560
180
Toggle Debug
f-toggle-debug
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
2562
135
2619
180
Debug
gs-debug-status
0
1
11

CHOOSER
2412
204
2566
249
gs-debug-step-chooser
gs-debug-step-chooser
"all" "setup" "pre-tick" "energize" "move" "feed" "reproduce" "die" "post-tick"
1

TEXTBOX
1250
359
1483
384
BEHAVIOUR SPACE USAGE:
18
0.0
1

TEXTBOX
2411
46
2561
72
DEBUG TOOLS\n
18
0.0
1

TEXTBOX
61
125
163
144
(NetLogo Platform)
11
0.0
1

BUTTON
9799
486
9862
519
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
472
465
560
527
<<---   To halt the run at a pre-specified tick, enter it here.
11
15.0
1

TEXTBOX
9558
10
10173
32
NRG & AGE HISTOGRAMs, AND CAUSES OF DEATH
20
55.0
1

BUTTON
7244
470
7308
503
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
606
505
1160
531
2-Sliders; 3-Debug/CSV; 4-Clock; 5/6/7/8/9-Graphs --->>>
18
15.0
1

TEXTBOX
2412
75
2562
97
Intructions:
18
0.0
1

TEXTBOX
2412
100
2847
130
To toggle the debug feature on or off use this button.  This opens a log file and enables debug options.  If debug is toggled off, the other options do nothing.
11
0.0
1

TEXTBOX
2415
185
2850
206
Then choose a single step, or 'all', in the chooser;\n
11
0.0
1

TEXTBOX
2414
276
2619
295
STANDARD OPERATION BUTTONS:
11
0.0
1

TEXTBOX
2410
341
2790
360
SINGLE-STEP WITHIN A TICK - should be pushed in order to debug a tick.
11
0.0
1

TEXTBOX
1316
9
1733
34
ADVANCED SYSTEM SETUP PARAMETERS
20
55.0
1

BUTTON
2206
299
2288
333
Reset Defaults
f-reset-default-parameters
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
1251
95
1501
336
- Adjust these prior to pressing the 'Setup' buttons.\n- The 'Reset Defaults' Button recovers a valid set of Parameters.\n- There are 8 coloured populations established, one for each belief affiliation.\n- The 'belief-value' parm adjusts the strength of the belief that tries to over-ride the genes.\n- The actual nrg delivered per tick is no-of-cells x Sun-nrg-per-tick x prob-of-deposit x nrg-per-deposit.\n- On reproduction, one gene may suffer a mutation, affecting both base and power.\n- The 'delta time' associated with calculation of EROI is used for moving averages.\n
11
0.0
1

TEXTBOX
1250
64
1400
86
NORMAL USAGE:
18
0.0
1

TEXTBOX
1250
387
1400
407
Instructions:
16
0.0
1

TEXTBOX
1251
413
1497
505
     Use the 'BehaviorSpace' option under the 'Tools' menu to design your set of runs using the sliders found here, on this panel, or using the 'random seed' slider on the main panel.\n     The operations switches cannot be controlled by BS, as far as I know.
11
0.0
1

TEXTBOX
3194
474
3507
503
4-Clock; 5/6/7/8/9 - Graphs --->>>
18
15.0
1

TEXTBOX
5397
491
5881
518
6-Nrg; 7-Genes; 8-Power/EROI; 9-Nrg/Age/COD --->>>
18
15.0
1

TEXTBOX
7320
474
7480
503
Be sure other 'Go' buttons are off before you turn this on.
11
15.0
1

TEXTBOX
9870
491
10020
519
Be sure other 'Go' buttons are off before you turn this on.
11
15.0
1

TEXTBOX
2615
298
2780
327
Be sure other 'Go' buttons are off before you turn this on.
11
0.0
1

TEXTBOX
235
462
319
518
Be sure other 'Go' buttons are off before you turn this on.
11
15.0
1

TEXTBOX
2922
45
3217
69
CSV DATA COLLECTION TOOLS
18
0.0
1

BUTTON
7104
470
7168
503
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
7166
470
7245
503
One Tick
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
9649
486
9721
519
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
9721
486
9800
519
One Tick
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
12
102
112
128
PANEL 01:
20
0.0
1

TEXTBOX
2372
46
2407
80
03a
20
0.0
1

TEXTBOX
2871
45
2910
70
03b
20
0.0
1

TEXTBOX
1205
64
1240
89
02a
20
0.0
1

TEXTBOX
1206
358
1244
383
02b
20
0.0
1

TEXTBOX
7124
12
7218
36
PANEL 07:
20
0.0
1

TEXTBOX
9455
10
9548
35
PANEL 09:
20
0.0
1

SWITCH
2922
141
3096
174
gb-plot-data
gb-plot-data
0
1
-1000

TEXTBOX
10617
157
10659
357
E\nN\nD
50
0.0
1

TEXTBOX
7890
485
8227
515
8-Power/EROI; 9-Nrg/Age/COD --->>>
18
15.0
1

TEXTBOX
1517
303
1995
331
[ ANY CHANGES TO THESE SLIDERS MID-RUN MAY CAUSE THE PROGRAM TO BOMB, OR IT MAY PROVE INTERESTING.  MANY COMBINATIONS HAVE NOT BEEN TESTED. ]
11
15.0
1

BUTTON
2575
360
2630
393
Feed
let temp1 gs-debug-step-chooser\nset gs-debug-step-chooser \"feed\"\ndo-feed\nf-update-aggregates\nset gs-debug-step-chooser temp1\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
2631
360
2694
393
Reproduce
let temp1 gs-debug-step-chooser\nset gs-debug-step-chooser \"reproduce\"\ndo-reproduce\nf-update-aggregates\nset gs-debug-step-chooser temp1\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
2695
360
2758
393
Die
let temp1 gs-debug-step-chooser\nset gs-debug-step-chooser \"die\"\ndo-die\nf-update-aggregates\nset gs-debug-step-chooser temp1\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
1519
64
1743
97
g-no-of-tribes
g-no-of-tribes
1
10
10
1
1
Tribes
HORIZONTAL

MONITOR
5944
63
6017
108
Ave Nrg
g-ave-nrg
1
1
11

MONITOR
3529
210
3606
255
B3
g-ave-c1-b3
5
1
11

PLOT
10014
52
10574
467
Causes of Death - Seekers
Ticks
Deaths
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Old Age" 1.0 0 -13345367 true "" "plot ( item 3 gl-causes-of-death-per-tick )"
"Fission/100" 1.0 0 -13840069 true "" "plot ( ( item 2 gl-causes-of-death-per-tick ) / 100 )"
"Hunger" 1.0 0 -2674135 true "" "plot ( item 1 gl-causes-of-death-per-tick )"

TEXTBOX
4833
8
5106
34
POPULATION GRAPHICS
20
54.0
1

MONITOR
3529
303
3606
348
B5
g-ave-c1-b5
5
1
11

MONITOR
3529
70
3606
115
B0
g-ave-c1-b0
5
1
11

TEXTBOX
1215
10
1308
36
PANEL 02:
20
0.0
1

TEXTBOX
115
103
227
128
THE MODEL
20
55.0
1

TEXTBOX
2374
9
2469
34
PANEL 03:
20
0.0
1

TEXTBOX
2471
9
2861
41
DEBUG AND DATA COLLECTION TOOLS
20
55.0
1

PLOT
4917
51
5155
421
Pops - By Belief Affiliation
Ticks
Count
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"P0" 1.0 0 -13840069 true "" "plot ( item 0 gl-no-of-seekers )"
"P1" 1.0 0 -8732573 true "" "if( g-scenario-number > 0 ) [ plot ( item 1 gl-no-of-seekers ) ]"
"P2" 1.0 0 -11033397 true "" "if( g-scenario-number > 0 ) [ plot ( item 2 gl-no-of-seekers ) ]"
"P3" 1.0 0 -817084 true "" "if( g-scenario-number > 0 ) [ plot ( item 3 gl-no-of-seekers ) ]"
"P4" 1.0 0 -2674135 true "" "if( g-scenario-number > 0 ) [ plot ( item 4 gl-no-of-seekers ) ]"
"P5" 1.0 0 -3844592 true "" "if( g-scenario-number > 0 ) [ plot ( item 5 gl-no-of-seekers ) ]"
"P6" 1.0 0 -12345184 true "" "if( g-scenario-number > 0 ) [ plot ( item 6 gl-no-of-seekers ) ]"
"P7" 1.0 0 -12087248 true "" "if( g-scenario-number > 0 ) [ plot ( item 7 gl-no-of-seekers ) ]"

TEXTBOX
10492
482
10601
512
END ---|||\n
18
15.0
1

MONITOR
7740
464
7797
509
Ticks
ticks
17
1
11

TEXTBOX
3539
9
3635
34
PANEL 04:
20
0.0
1

TEXTBOX
1887
488
2353
516
3-Debug/CSV; 4-Clock; 5/6/7/8/9 - Graphs --->>>
18
15.0
1

BUTTON
3228
143
3334
176
Clear All Plots
clear-all-plots
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
3762
475
3841
508
One Tick
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
3841
475
3920
508
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
3924
471
4045
514
Be sure other 'Go' buttons are off before you turn this on.
11
15.0
1

TEXTBOX
19
297
150
316
SETUP CONTROLS
11
0.0
1

TEXTBOX
8285
8
8382
30
PANEL 08:
20
0.0
1

TEXTBOX
8388
8
8914
38
TIME LINES - POWER AND EROI (MOVING AVERAGES)
20
55.0
1

BUTTON
8352
472
8431
505
One Tick
Go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
8288
472
8352
505
Setup
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
8431
472
8494
505
Go
Go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
8497
476
8653
505
Be sure other 'Go' buttons are off before you turn this on.
11
15.0
1

TEXTBOX
9067
494
9398
523
9-Nrg & Age Histograms & COD --->>>
18
15.0
1

BUTTON
4747
482
4811
515
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
4810
482
4889
515
One Tick
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
4889
482
4952
515
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
4959
482
5109
510
Be sure other 'Go' buttons are off before you turn this on.
11
15.0
1

TEXTBOX
3635
10
4085
34
CLOCK FACE - AVERAGES, 8 COMPOUND GENES
20
0.0
1

MONITOR
3975
123
4041
168
P0 - F
g-ave-c1-p0
5
1
11

MONITOR
3677
71
3745
116
L0
g-ave-c1-l0
5
1
11

MONITOR
4050
148
4114
193
P1 - FR
g-ave-c1-p1
5
1
11

MONITOR
3677
117
3745
162
L1
g-ave-c1-L1
5
1
11

MONITOR
3901
286
3965
331
P5 - BL
g-ave-c1-p5
5
1
11

MONITOR
3677
303
3745
348
L5
g-ave-c1-L5
5
1
11

MONITOR
4071
216
4135
261
P2 - R
g-ave-c1-p2
5
1
11

MONITOR
3677
163
3746
208
L2
g-ave-c1-L2
5
1
11

MONITOR
4065
284
4129
329
P3 - BR
g-ave-c1-p3
5
1
11

MONITOR
3677
210
3746
255
L3
g-ave-c1-L3
5
1
11

MONITOR
3977
310
4044
355
P4 - B
g-ave-c1-p4
5
1
11

MONITOR
3677
257
3745
302
L4
g-ave-c1-L4
5
1
11

MONITOR
3875
219
3939
264
P6 - L
g-ave-c1-p6
5
1
11

MONITOR
3677
350
3744
395
L6
g-ave-c1-L6
5
1
11

MONITOR
3890
149
3954
194
P7 - FL
g-ave-c1-p7
5
1
11

MONITOR
3677
396
3746
441
L7
g-ave-c1-L7
5
1
11

BUTTON
3699
475
3763
508
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
1175
12
1190
516
|   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   |   
11
0.0
1

MONITOR
8867
474
8924
519
Ticks
ticks
0
1
11

SLIDER
2010
61
2182
94
g-c2-dat-parm
g-c2-dat-parm
100
3200
1600
10
1
Ticks
HORIZONTAL

SLIDER
2010
94
2182
127
g-c2-det-parm
g-c2-det-parm
4
40
4
4
1
Units
HORIZONTAL

SLIDER
2010
127
2182
160
g-c2-rat-parm
g-c2-rat-parm
50
3200
800
10
1
Ticks
HORIZONTAL

SLIDER
2010
159
2182
192
g-c2-ret-parm
g-c2-ret-parm
200
1600
1000
1
1
Units
HORIZONTAL

SLIDER
2010
192
2182
225
g-c2-epm-parm
g-c2-epm-parm
1
40
4
1
1
Units
HORIZONTAL

SLIDER
2010
225
2182
258
g-c2-epa-parm
g-c2-epa-parm
800
3000
1600
100
1
Units
HORIZONTAL

MONITOR
3529
117
3606
162
B1
g-ave-c1-b1
5
1
11

MONITOR
3529
257
3606
302
B4
g-ave-c1-b4
5
1
11

MONITOR
3529
163
3606
208
B2
g-ave-c1-b2
5
1
11

MONITOR
4744
97
4816
142
Ave Age
g-ave-age
1
1
11

SLIDER
1519
97
1743
130
g-sun-nrg-per-tick
g-sun-nrg-per-tick
1000
12000
6000
40
1
Units
HORIZONTAL

SLIDER
1519
130
1743
163
g-nrg-per-deposit
g-nrg-per-deposit
1
100
40
1
1
Units
HORIZONTAL

SLIDER
1519
164
1743
197
g-prob-of-deposit
g-prob-of-deposit
0
1
0.1
.001
1
NIL
HORIZONTAL

SLIDER
1519
197
1743
230
g-prob-of-mutation
g-prob-of-mutation
0
1
0.5
.001
1
NIL
HORIZONTAL

BUTTON
2466
360
2521
393
Energize
let temp1 gs-debug-step-chooser\nset gs-debug-step-chooser \"energize\"\ndo-energize\nf-update-aggregates\nset gs-debug-step-chooser temp1
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
6027
63
6265
396
Nrg Levels
Ticks
Nrg Units
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"F" 1.0 0 -13840069 true "" "plot g-nrg-in-fruit"
"S" 1.0 0 -2674135 true "" "plot ( sum gl-nrg-by-belief )"

PLOT
8016
59
8243
378
[P]henotype Values By Gene #
Ticks
Phenotypic Units
0.0
10.0
5.0
15.0
true
true
"" ""
PENS
"P0" 1.0 0 -13840069 true "" "plot g-ave-C1-p0"
"P1" 1.0 0 -8732573 true "" "plot g-ave-C1-p1"
"P2" 1.0 0 -11033397 true "" "plot g-ave-C1-p2"
"P3" 1.0 0 -817084 true "" "plot g-ave-C1-p3"
"P4" 1.0 0 -2674135 true "" "plot g-ave-C1-p4"
"P5" 1.0 0 -3844592 true "" "plot g-ave-C1-p5"
"P6" 1.0 0 -12345184 true "" "plot g-ave-C1-p6"
"P7" 1.0 0 -12087248 true "" "plot g-ave-C1-p7"

PLOT
7557
59
7785
378
[L]earned Beliefs By Gene #
Ticks
Moral Units
0.0
10.0
0.5
1.5
true
true
"" ""
PENS
"L0" 1.0 0 -13840069 true "" "plot g-ave-C1-L0"
"L1" 1.0 0 -8732573 true "" "plot g-ave-C1-L1"
"L2" 1.0 0 -11033397 true "" "plot g-ave-C1-L2"
"L3" 1.0 0 -817084 true "" "plot g-ave-C1-L3"
"L4" 1.0 0 -2674135 true "" "plot g-ave-C1-L4"
"L5" 1.0 0 -3844592 true "" "plot g-ave-C1-L5"
"L6" 1.0 0 -12345184 true "" "plot g-ave-C1-L6"
"L7" 1.0 0 -12087248 true "" "plot g-ave-C1-L7"

PLOT
7327
59
7556
378
[G]ene Values By Gene #
Ticks
Gene Units
0.0
10.0
-0.5
0.5
true
true
"" ""
PENS
"G0" 1.0 0 -13840069 true "" "plot g-ave-C1-g0"
"G1" 1.0 0 -8732573 true "" "plot g-ave-C1-g1"
"G2" 1.0 0 -11033397 true "" "plot g-ave-C1-g2"
"G3" 1.0 0 -817084 true "" "plot g-ave-C1-g3"
"G4" 1.0 0 -2674135 true "" "plot g-ave-C1-g4"
"G5" 1.0 0 -3844592 true "" "plot g-ave-C1-g5"
"G6" 1.0 0 -12345184 true "" "plot g-ave-C1-g6"
"G7" 1.0 0 -12087248 true "" "plot g-ave-C1-g7"

PLOT
6573
63
6811
395
Nrg - By Belief Affiliation
Ticks
Nrg Units
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"P0" 1.0 0 -13840069 true "" "plot ( item 0 gl-nrg-by-belief )"
"P1" 1.0 0 -8732573 true "" "plot ( item 1 gl-nrg-by-belief )"
"P2" 1.0 0 -11033397 true "" "plot ( item 2 gl-nrg-by-belief )"
"P3" 1.0 0 -817084 true "" "plot ( item 3 gl-nrg-by-belief )"
"P4" 1.0 0 -2674135 true "" "plot ( item 4 gl-nrg-by-belief )"
"P5" 1.0 0 -3844592 true "" "plot ( item 5 gl-nrg-by-belief )"
"P6" 1.0 0 -12345184 true "" "plot ( item 6 gl-nrg-by-belief )"
"P7" 1.0 0 -12087248 true "" "plot ( item 7 gl-nrg-by-belief )"

MONITOR
3610
70
3675
115
G0
g-ave-c1-g0
5
1
11

MONITOR
3609
117
3675
162
G1
g-ave-c1-G1
5
1
11

MONITOR
3608
163
3675
208
G2
g-ave-c1-G2
5
1
11

MONITOR
3609
211
3675
256
G3
g-ave-c1-G3
5
1
11

MONITOR
3610
257
3676
302
G4
g-ave-c1-G4
5
1
11

MONITOR
3609
303
3676
348
G5
g-ave-c1-G5
5
1
11

MONITOR
3608
350
3675
395
G6
g-ave-c1-G6
5
1
11

MONITOR
3609
397
3675
442
G7
g-ave-c1-G7
5
1
11

TEXTBOX
3819
42
4115
70
The strength is calculated as Si=Bi^(Gi+Li).\nThe phenotypic attribute is calculated as Pi=Si/sum(all Sj).
11
15.0
1

TEXTBOX
3530
40
3595
68
[B]ase Values
11
0.0
1

SLIDER
1519
412
1741
445
g-c1-belief-value
g-c1-belief-value
0
3
2
.1
1
NIL
HORIZONTAL

TEXTBOX
2349
9
2364
527
|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|
11
0.0
1

TEXTBOX
4705
12
4721
525
|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|
11
0.0
1

SLIDER
1518
264
1742
297
g-dt-for-eroi
g-dt-for-eroi
40
400
40
40
1
Ticks
HORIZONTAL

PLOT
8288
48
9059
253
EROI - INDIVIDUAL (MAX, AVE, MIN) AND SYSTEM-WIDE
Ticks
EROI = [ ER / EI ]
0.0
10.0
0.0
2.0
true
true
"" ""
PENS
"Max" 1.0 0 -2674135 true "" "plot g-ind-max-eroi"
"Ave" 1.0 0 -13840069 true "" "plot g-ind-ave-eroi"
"Min" 1.0 0 -13345367 true "" "plot g-ind-min-eroi"
"SysWide" 1.0 0 -5825686 true "" "plot g-sys-eroi"
"1.00" 1.0 0 -16777216 true "" "plot 1"

PLOT
9061
48
9413
466
POWER OF NRG EXPENDED, TOTAL POPULATION, AND NRG IN FRUIT (SCALED BY 1/100)
Ticks
Nrg Units/Counts
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Power" 1.0 0 -2674135 true "" "plot ( item 1 gl-sinks-per-tick )"
"Population" 1.0 0 -16777216 true "" "plot ( g-no-of-seekers )"
"F/50" 1.0 0 -13840069 true "" "plot  ( g-nrg-in-fruit / 50 )"

MONITOR
4833
51
4906
96
Pop L0
count seekers with [belief-affiliation = 0]
0
1
11

MONITOR
4833
98
4906
143
Pop L1
count seekers with [belief-affiliation = 1]
17
1
11

MONITOR
4833
144
4906
189
Pop L2
count seekers with [belief-affiliation = 2]
17
1
11

MONITOR
4833
191
4906
236
Pop L3
count seekers with [belief-affiliation = 3]
17
1
11

MONITOR
4833
237
4906
282
Pop L4
count seekers with [belief-affiliation = 4]
17
1
11

MONITOR
4833
282
4905
327
Pop L5
count seekers with [belief-affiliation = 5]
17
1
11

MONITOR
4833
328
4905
373
Pop L6
count seekers with [belief-affiliation = 6]
17
1
11

MONITOR
4833
374
4905
419
Pop L7
count seekers with [belief-affiliation = 7]
17
1
11

MONITOR
4744
51
4817
96
Total Pop
count seekers
17
1
11

TEXTBOX
4002
185
4017
207
0
20
0.0
1

TEXTBOX
4031
198
4046
220
1
20
0.0
1

TEXTBOX
4050
225
4066
247
2
20
0.0
1

TEXTBOX
4034
255
4051
277
3
20
0.0
1

TEXTBOX
4003
268
4018
290
4
20
0.0
1

TEXTBOX
3974
252
3989
274
5
20
0.0
1

TEXTBOX
3957
225
3972
247
6
20
0.0
1

TEXTBOX
3972
198
3987
220
7
20
0.0
1

TEXTBOX
4001
225
4017
248
O
20
0.0
1

TEXTBOX
5899
21
5917
531
|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n
11
0.0
1

TEXTBOX
8262
11
8277
536
|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|
11
0.0
1

TEXTBOX
9432
11
9448
529
|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|
11
0.0
1

TEXTBOX
10602
11
10618
506
|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n
11
0.0
1

TEXTBOX
3503
9
3518
519
|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n
11
0.0
1

TEXTBOX
2012
44
2183
63
C2 Controls - Biological Functions
11
0.0
1

TEXTBOX
2186
59
2348
102
-----------------------------------\nDeath Age Threshold\n-----------------------------------
11
0.0
1

TEXTBOX
2189
235
2349
265
Maximum Energy Per Agent\n-----------------------------------
11
0.0
1

TEXTBOX
2187
106
2348
135
Death Energy Threshold\n-----------------------------------
11
0.0
1

TEXTBOX
2186
137
2348
168
Reproductive Age Threshold\n-----------------------------------
11
0.0
1

TEXTBOX
2186
169
2348
204
Reproductive Age Threshold\n-----------------------------------
11
0.0
1

TEXTBOX
2187
202
2347
232
Energy Per Move\n-----------------------------------
11
0.0
1

TEXTBOX
2021
262
2258
281
Derived From \"Simulated Evolution\" by Michael Palmiter.
8
0.0
1

TEXTBOX
1749
92
2002
120
-----------------------------------------------------------\nMaximum Energy From Sun Per Tick
11
0.0
1

TEXTBOX
1522
40
1679
59
Various Advanced Controls
11
0.0
1

TEXTBOX
1749
127
2002
157
-----------------------------------------------------------\nEnergy Deposited, In Fruit, Per Empty Cell
11
0.0
1

TEXTBOX
1750
157
2002
202
-----------------------------------------------------------\nProbability Of Deposit Per Empty Cell Per Tick
11
0.0
1

TEXTBOX
1750
188
2000
218
-----------------------------------------------------------\nProbability of Gene Mutation On Reproduction
11
0.0
1

TEXTBOX
1747
58
2002
88
-----------------------------------------------------------\nMultiply by 8 for total population at start.
11
0.0
1

TEXTBOX
1745
406
1999
436
-----------------------------------------------------------\nStrength of the Belief To Impose
11
0.0
1

TEXTBOX
1747
257
2003
287
-----------------------------------------------------------\nDelta Time for Moving Averages for EROI
11
0.0
1

TEXTBOX
1747
288
2005
308
-----------------------------------------------------------
11
0.0
1

TEXTBOX
4159
25
4689
485
DESCRIPTION OF GENES:\n     These 'seekers' are searching for patches of fruit, and there is not enough for everybody.  Each seeker has a set of eight compound genes (base and exponent) and a set of learned beliefs that combine to control its behaviour.  But the seekers are all virtually blind, so the only advantage one seeker can have over another is a more effective search pattern.  During a move, one gene of the eight is expressed, and the phenotype of that gene determines a change of heading as indicated in the 'clock face' diagram to the left.  For example, gene 0 causes the seeker to step forward.  Gene 1 causes a turn of 45 degrees to the right, then a step forward.  Gene 2 causes a turn of 90 degrees to the right, then a step forward.  The phenotypic attribute of each gene determines its relative probability of being expressed in a turn.  These compound genes encode an implicit heuristic search pattern.  All seekers start with a very ineffective search pattern, and evolve a more effective search pattern.\n     Each compound gene has three components: [B]ase, [G]enetic exponent, [L]earned belief.  [B] and [G] encode the instinctive behaviour of the seeker, and [L] codes cognitive influences on behaviour.  These three values can be used to calculate [S]trength and [P]henotypic attribute for each compound gene.  The [P] values code the effective search pattern.  \n     If the 'g-c1-belief-value' slider is set to zero then the [L] factor is null and there is no cognitive influence.  Try it!  But when that slider is very large, each seeker has a 'belief' that one gene has more strategic value than another gene, and the behaviour is strongly biased towards a particular search pattern, for good or bad effect.  There are eight different affiliations of seekers, one for each type of learned belief, which all compete with one another.  See the information tab for more details.\n     The C2 adjustable genes are constant during a run.
14
0.0
1

TEXTBOX
4130
492
4700
519
5-Pop; 6-Nrg; 7-Genes; 8-Power/EROI; 9-Nrg/Age Hist/COD   --->>>
18
15.0
1

TEXTBOX
4727
8
4833
29
PANEL 05:
20
0.0
1

MONITOR
3529
350
3605
395
B6
g-ave-c1-b6
5
1
11

MONITOR
3528
397
3604
442
B7
g-ave-c1-b7
5
1
11

TEXTBOX
5165
46
5353
416
<<<----\nThe population of seekers is broken out by belief affiliation.  For example, Pop L0 agents have a 'belief' that the compound gene 0 is a 'better' choice and so are biased to use it more than experience would advise. In scenarios 0-2 agents within a belief affiliation are not able to hold any other beliefs, but in scenario 2, they can at least advise one another on the best emphasis for that belief.  Initially, there is a population endowed with one each of eight beliefs affiliations.  The belief affiliation radically affects the outcome. Typically, one population will eventually radiate and dominate the ecosystem.
13
0.0
1

TEXTBOX
4835
432
5348
451
|<------ Watch this area for Scenarios 2 through 4 when the eight belief affiliations are active. ------>|
11
0.0
1

TEXTBOX
6030
408
6261
438
Nrg in fruit (green patches) and total nrg in all the seekers (agents represented as arrows).
11
0.0
1

TEXTBOX
6582
404
7040
463
Each population holds nrg.  Here it it shown broken out by belief affiliation or by tribal affiliation.  Usually one or two types of affiliation radiate and dominate the ecosystem.  One belief affiliation almost always comes to dominate scenario 3.  But, in scenario 4, any tribal affiliation may come to dominate.
11
0.0
1

PLOT
7097
59
7327
378
[B]ase Values By Gene #
Ticks
Base Units
0.0
10.0
1.5
2.5
true
true
"" ""
PENS
"B0" 1.0 0 -13840069 true "" "plot g-ave-C1-b0"
"B1" 1.0 0 -8732573 true "" "plot g-ave-C1-B1"
"B2" 1.0 0 -11033397 true "" "plot g-ave-C1-B2"
"B3" 1.0 0 -817084 true "" "plot g-ave-C1-B3"
"B4" 1.0 0 -2674135 true "" "plot g-ave-C1-B4"
"B5" 1.0 0 -3844592 true "" "plot g-ave-C1-B5"
"B6" 1.0 0 -12345184 true "" "plot g-ave-C1-B6"
"B7" 1.0 0 -12087248 true "" "plot g-ave-C1-B7"

PLOT
9456
52
9741
467
Nrg of Seekers
Nrg Units
Count of Seekers
0.0
10.0
0.0
10.0
true
false
"set-current-plot \"Nrg of Seekers\"\nset-histogram-num-bars 135\nset-current-plot-pen \"Nrg\"\nset-plot-pen-mode 1      ;; 0=line mode; 1=bar mode; 2=point mode" "set-current-plot \"Nrg of Seekers\"\nCLEAR-PLOT\nset-plot-x-range 0 ( g-C2-epa-parm )\nset-plot-pen-mode 1      ;; 0=line mode; 1=bar mode; 2=point mode\nset-histogram-num-bars 135"
PENS
"Nrg" 1.0 0 -2674135 true "" "histogram [nrg] of seekers"

TEXTBOX
7107
384
7322
444
These are the [B]ases of the eight compound genes.  All start at a value of 2.  Evolutionary pressure is mild.
11
0.0
1

TEXTBOX
7327
379
7542
439
These are the exponents of the eight compound [G]enes.  All start at a value of 0.  These play a significant role in any effective search pattern.
11
0.0
1

TEXTBOX
7557
377
7781
447
These values encode the [L]earned beliefs of the seekers as to which swivelling behaviour is best.  False beliefs impervious to change from experience may be adaptive, or may not be.
11
0.0
1

TEXTBOX
8017
380
8237
468
These [P]henotypic characters control the search pattern of seekers.  Those seekers with the most effective search pattern usually increase their numbers, while those with less effective search patterns usually decrease in number.
11
0.0
1

TEXTBOX
7424
39
7973
58
All values in this panel are averaged instantaneously across all currently living seekers.
14
15.0
1

TEXTBOX
8040
9
8220
48
[Si]=[Bi]^([Gi]+[Li])\n[Pi]=[Si]/sum[Sj]
15
65.0
1

MONITOR
7800
464
7850
509
Seed
g-use-this-seed
0
1
11

SLIDER
1519
378
1742
411
g-gene-to-adjust
g-gene-to-adjust
0
7
0
1
1
Gene Type
HORIZONTAL

TEXTBOX
1747
375
2002
415
-----------------------------------------------------------\nBelief Affiliation
11
0.0
1

TEXTBOX
1742
435
1999
455
-----------------------------------------------------------
11
0.0
1

TEXTBOX
1521
449
1984
481
[ THESE TWO SLIDERS ARE USED TO IMPOSE A MORAL ALIGNMENT ON A RANDOM 1/2 OF THE CURRENT POPULATION OF SEEKERS, MID-RUN, IN ANY SCENARIO. ]
11
15.0
1

BUTTON
1828
377
1994
417
<- Impose This Belief Affiliation
f-impose-belief-affiliation
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1994
327
2168
372
Active Scenario
gs-scenario
0
1
11

MONITOR
1994
418
2168
463
Current Population
g-no-of-seekers
0
1
11

MONITOR
1994
374
2168
419
Number Sharing This Belief
g-no-with-this-belief
17
1
11

MONITOR
3540
465
3597
510
Ticks
Ticks
0
1
11

MONITOR
4744
143
4817
188
Ticks
Ticks
0
1
11

MONITOR
10162
477
10219
522
Ticks
Ticks
0
1
11

PLOT
6267
63
6573
396
Sun Takeup Rates
Ticks
Nrg Units
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Max Possible" 1.0 0 -16777216 true "" "plot g-sun-takeup-maximum"
"Expected" 1.0 0 -7500403 true "" "plot g-sun-takeup-expected"
"Limit" 1.0 0 -2674135 true "" "plot g-sun-nrg-per-tick"
"Actual" 1.0 0 -10899396 true "" "plot g-sun-takeup-actual"

PLOT
9741
52
10014
467
Age of Seekers
Age
Count of Seekers
0.0
10.0
0.0
10.0
true
false
"set-current-plot \"Age of Seekers\"\nset-histogram-num-bars 135\nset-current-plot-pen \"Age\"\nset-plot-pen-mode 1      ;; 0=line mode; 1=bar mode; 2=point mode" "set-current-plot \"Age of Seekers\"\nCLEAR-PLOT\nset-plot-x-range 0 ( g-C2-dat-parm )\nset-plot-pen-mode 1      ;; 0=line mode; 1=bar mode; 2=point mode\nset-histogram-num-bars 135"
PENS
"age" 1.0 0 -13840069 true "" "histogram [age] of seekers"

SLIDER
1519
230
1743
263
g-prob-of-belief-drift
g-prob-of-belief-drift
0
1
0.5
.001
1
NIL
HORIZONTAL

TEXTBOX
1747
224
2002
253
-----------------------------------------------------------\nProbability of Beleif Drift On Reproduction
11
0.0
1

MONITOR
2636
420
2693
465
Ticks
Ticks
0
1
11

PLOT
6810
63
7048
395
Nrg - By Tribal Affiliation
Ticks
Nrg Units
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"T0" 1.0 0 -2674135 true "" "plot (item 0 gl-nrg-by-tribe )"
"T1" 1.0 0 -955883 true "" "plot (item 1 gl-nrg-by-tribe )"
"T2" 1.0 0 -6459832 true "" "plot (item 2 gl-nrg-by-tribe )"
"T3" 1.0 0 -1184463 true "" "plot (item 3 gl-nrg-by-tribe )"
"T4" 1.0 0 -10899396 true "" "plot (item 4 gl-nrg-by-tribe )"
"T5" 1.0 0 -13840069 true "" "plot (item 5 gl-nrg-by-tribe )"
"T6" 1.0 0 -14835848 true "" "plot (item 6 gl-nrg-by-tribe )"
"T7" 1.0 0 -11221820 true "" "plot (item 7 gl-nrg-by-tribe )"
"T8" 1.0 0 -13345367 true "" "plot (item 8 gl-nrg-by-tribe )"
"T9" 1.0 0 -8630108 true "" "plot (item 9 gl-nrg-by-tribe )"

MONITOR
3747
70
3815
115
S0
g-ave-c1-s0
3
1
11

MONITOR
3747
117
3815
162
S1
g-ave-c1-s1
3
1
11

MONITOR
3747
163
3814
208
S2
g-ave-c1-s2
3
1
11

MONITOR
3747
210
3814
255
S3
g-ave-c1-s3
3
1
11

MONITOR
3747
257
3814
302
S4
g-ave-c1-s4
3
1
11

MONITOR
3747
303
3814
348
S5
g-ave-c1-s5
3
1
11

MONITOR
3747
350
3814
395
S6
g-ave-c1-s6
3
1
11

MONITOR
3747
395
3814
440
S7
g-ave-c1-s7
3
1
11

TEXTBOX
3610
40
3672
69
[G]enetic Values
11
0.0
1

TEXTBOX
3677
40
3747
73
[L]earned Belief Values
11
0.0
1

TEXTBOX
3746
40
3805
73
Gene [S]trengths
11
0.0
1

TEXTBOX
3930
95
4097
115
Gene [P]henotypic Attributes
11
0.0
1

PLOT
7786
59
8016
378
[S]trengths By Gene #
Ticks
Strength Units
0.0
10.0
0.0
2.0
true
true
"" ""
PENS
"S0" 1.0 0 -13840069 true "" "plot g-ave-c1-s0"
"S1" 1.0 0 -8732573 true "" "plot g-ave-c1-s1"
"S2" 1.0 0 -11033397 true "" "plot g-ave-c1-s2"
"S3" 1.0 0 -817084 true "" "plot g-ave-c1-s3"
"S4" 1.0 0 -2674135 true "" "plot g-ave-c1-s4"
"S5" 1.0 0 -3844592 true "" "plot g-ave-c1-s5"
"S6" 1.0 0 -12345184 true "" "plot g-ave-c1-s6"
"S7" 1.0 0 -12087248 true "" "plot g-ave-c1-s7"

TEXTBOX
7786
379
8008
449
These values are calculated as [S]=([B]^([G]+[L])).  The strengths of the eight compound genes determine the heuristic search patterns of the blind seekers.
11
0.0
1

BUTTON
2920
392
3093
426
Record Selected Plots
f-record-selected-plots
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
3100
395
3270
425
Dumps data from seven plots - may take a minute or two.
11
105.0
1

MONITOR
7557
464
7739
509
Scenario
gs-scenario
17
1
11

TEXTBOX
2923
71
3408
128
The entire contents of any run can be saved to a CSV file (Comma Separated Values file) at any time.  Simply open the control panel (small carots at the bottom left) and enter the command 'export-world filename.csv'.  Such 'whole-world' files can then be re-imported into NetLogo TpLab later, and the run continued.  Or, they can be loaded into MS Excel for data manipulation.\n\n
11
0.0
1

TEXTBOX
1520
354
1670
372
Special controls
11
0.0
1

TEXTBOX
2923
266
3410
380
OR, the button below can be used to activate a brief routine that dumps the data from seven selected graphs:\n\nPANEL 5:\n-  Nrg - By Belief Affiliation\n-  Nrg - By Tribal Affiliation\nPANEL 6:\n-  All five graphs.
11
0.0
1

TEXTBOX
2923
185
3410
246
OR:\nEach graph in NetLogo maintains a small database of plotted points.  These points can be saved to a CSV (comma separated values) file which can then be loaded into MS Excel for data manipulation.  At any time, simply right-click on the graph and select 'export'.\n
11
0.0
1

TEXTBOX
3102
137
3220
185
Use this switch to turn data collection off or on for all graphs.
11
15.0
1

TEXTBOX
3338
146
3451
174
Use this button to discard all data.
11
15.0
1

TEXTBOX
7066
19
7081
514
|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n|\n
11
0.0
1

TEXTBOX
5944
21
6094
46
PANEL 06:
20
0.0
1

TEXTBOX
6053
20
6253
42
ENERGY GRAPHICS
20
65.0
1

BUTTON
5948
470
6012
503
Setup
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
6014
470
6093
503
One Tick
Go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
6095
470
6158
503
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

TEXTBOX
6165
474
6315
502
Be sure other 'Go' buttons are off before you turn this on.
11
15.0
1

MONITOR
5364
8
5437
53
Pop T0
count seekers with [tribal-affiliation = 0]
0
1
11

MONITOR
5364
51
5437
96
Pop T1
count seekers with [tribal-affiliation\n = 1]
0
1
11

MONITOR
5364
96
5437
141
Pop T2
count seekers with [tribal-affiliation = 2]
0
1
11

MONITOR
5364
140
5438
185
Pop T3
count seekers with [tribal-affiliation = 3]
0
1
11

MONITOR
5364
186
5438
231
Pop T4
count seekers with [tribal-affiliation = 4]
0
1
11

MONITOR
5364
231
5436
276
Pop T5
count seekers with [tribal-affiliation = 5]
0
1
11

MONITOR
5364
277
5436
322
Pop T6
count seekers with [tribal-affiliation = 6]
0
1
11

MONITOR
5364
321
5436
366
Pop T7
count seekers with [tribal-affiliation = 7]
0
1
11

MONITOR
5364
367
5436
412
Pop T8
count seekers with [tribal-affiliation = 8]
0
1
11

MONITOR
5364
412
5436
457
Pop T9
count seekers with [tribal-affiliation = 2]
0
1
11

PLOT
5438
8
5708
458
Pops - By Tribal Affiliation
Ticks
Count
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"T0" 1.0 0 -2674135 true "" "plot count seekers with [tribal-affiliation = 0]"
"T1" 1.0 0 -817084 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 1] ]"
"T2" 1.0 0 -6459832 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 2] ]"
"T3" 1.0 0 -1184463 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 3] ]"
"T4" 1.0 0 -10899396 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 4] ]"
"T5" 1.0 0 -13840069 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 5] ]"
"T6" 1.0 0 -14835848 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 5] ]"
"T7" 1.0 0 -11221820 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 7] ]"
"T8" 1.0 0 -13345367 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 8] ]"
"T9" 1.0 0 -8630108 true "" "if (g-scenario-number > 2) \n  [ plot count seekers with [tribal-affiliation = 9] ]"

TEXTBOX
5723
37
5883
327
<<<---\nHere, the population of seekers is broken out by tribal affiliation.  For example, Pop T0 agents share wisdom and beliefs of all types among one another.  All tribes start equally endowed with such wisdom, and must evolve to maximize their population using that wisdom.  Some tribal affiliations share wisdom effectively and gain advantage, while others do not.
13
0.0
1

TEXTBOX
5365
462
5886
481
|<---------------- Watch this area for Scenario 4 when the ten tribal affiliations are active. ---------------->|
11
0.0
1

TEXTBOX
6672
482
7061
505
7-Genes; 8-Power/EROI; 9-Nrg/Age/COD --->>>
18
15.0
1

TEXTBOX
6274
407
6569
435
Max Possible comes from the Sun.  Limit is determined by ability of patches to absorb it.
11
0.0
1

MONITOR
5164
474
5313
519
Scenario
gs-scenario
17
1
11

MONITOR
5114
474
5164
519
Scen #
g-scenario-number
0
1
11

MONITOR
6356
464
6406
509
Scen #
g-scenario-number
0
1
11

MONITOR
6408
465
6572
510
Scenario
gs-scenario
0
1
11

MONITOR
7507
464
7557
509
Scen #
g-scenario-number
17
1
11

PLOT
8288
252
9059
466
ETA - INDIVIDUAL (MAX, AVE MIN) AND SYSTEM-WIDE
Ticks
ETA = [ ER / (ER + EI ) ]
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Max" 1.0 0 -2674135 true "" "plot g-ind-max-eta"
"Ave" 1.0 0 -13840069 true "" "plot g-ind-ave-eta"
"Min" 1.0 0 -13345367 true "" "plot g-ind-min-eta"
"SysWide" 1.0 0 -5298144 true "" "plot g-sys-eta"
"1.00" 1.0 0 -16777216 true "" "plot 1.00"
"0.50" 1.0 0 -16777216 true "" "plot 0.50"

@#$#@#$#@
## WHAT IS IT?

**TpLab** - This is the third of a series of three planned models in a study of the maximum power principle (MPP) - the other two models being **OamLab** and **MppLab**.  In **TpLab** energy arrives in a steady stream from the Sun and is captured in plants that produce fruit, which appears randomly in dales in the forest.  Blind nomadic seekers roaming the forest, each according to their own heuristic search strategy, seek the dales currently having food, and harvest it.  The strategies start as ineffective bland heuristics, and evolve to be much more sophisticated. In the most simple scenario, all actions are purely instinctive.  But an ability to have beliefs influence behaviour is built into the genes, and belief systems evolve within the societies of seekers.  These teleological belief systems can over-rule instinctual behaviours.  What happens to seekers of fruit that hold beliefs too tightly?  What happens to belief systems that are inconsistent with reality?  Under what circumstances can such erroneous belief systems persist?  What happens in societies that share a dialectic across multiple belief systems?  Can the seekers of fruit become seekers of truth and wisdom?

**TpLab**, short for **Teleological Pruning Laboratory**, demonstrates evolutionary processes that shape the belief systems of a society to align with the effects of the Lotka/Odum Maximum Power Principle (MPP).  That is, the belief systems are shaped to produce an overall system that functions at the highest rate of consumption of matter and energy that is consistent with the available resources.  

**MAXIMUM POWER PRINCIPLE (MPP)** - It is clear that, when isolated and left alone, all systems die, run down, erode, decay and/or dissipate their matter and energy.  This process is closely associated with the phenomenon of entropy production, and the 2nd law of thermodynamics (sometimes referred to as the maximum entropy principle, or MEP) easily describes how this happens.  Any such energetically isolated system automatically alters and reconfigures itself, moving through a series of ever more probable states a trajectory of ever increasing entropy, until a most probable configuration or state of maximal entropy is achieved.  This state of maximal entropy is usually characterized by an excess of sameness, a lack of structures, shapes or spatial variations, and by characteristic distributions of energies among the parts.  Once having achieved such a configuration, the isolated system then remains in a state of maximal entropy (a most probable state) forever after.  However, when we look around ourselves, virtually everything we see is characterized by remarkable improbable variety, a plethora of structures and shapes, and turbulent distributions of energy.  Clearly, when a system is not isolated and left alone, there is another dynamic able to overpower the 2nd law and undo its nasty work.  Such is the nature of the proposed 4th law of thermodynamics.  A.J. Lotka, H.T. Odum and R.C. Pinkerton called this the Maximum Power Principle (MPP).  H.T. Odum (1924-2002) went on, over four decades, to argue that the MPP is the best candidate for the fourth law of thermodynamics, having explanatory value for such things as ecosystems, economies, and other self-organizing systems.  However, in spite of the fact that there is plenty of anecdotal evidence in support of the concept, it remains little understood, and little studied it seems.  The MPP says (my words) that any self-organizing system that is open with respect to a flow of energy will configure itself to store/consume/use energy at the maximum rate possible.  This model is a "laboratory" in which I can study the nature of the MPP.

This model is the third "laboratory" produced in my personal study of the existence and nature of the MPP.  In that study, it has become apparent to me that, through the processes of natural selection, the MPP not only shapes our genes to enhance global consumption, but it also aligns our belief systems to the same end.  Those beliefs and practices that stand in the way of increased consumptive power of our emerging global society are pruned away, and those that enhance the consumptive power of global society spread and come to dominate society.  So, in like manner to the way our genes are pruned, like the branches of a wild apple tree, our belief systems are also pruned.  I refer to this process as Teleological Pruning (or just TP).  TP, being a sub-dynamic of the MPP, is sometimes herein referred to as the MPP/TP.

**TELEOLOGICAL PRUNING (MPP/TP)** - The purpose of this model is to take the study of the MPP into the realm of social effects.  It is a desktop laboratory for the study of the MPP/TP.  In its broadest interpretation, the MPP says (my words) that any self-organizing system that is open with respect to a flow of matter and energy will continuously reconfigure itself to store/consume/use energy at the maximum rate possible, consistent with available sources of matter and energy.  In a variety of diary notes that I have written in an attempt to understand the MPP, I have developed an opinion about just how this affects socially.  Not only are our genes shaped to effect this end, but our national political and social structures, our technologies, and our belief systems, plans and practices are also shaped to enhance the overall flow and consumption of matter and energy by the global economy.  This even shapes the political and economic organs of global governance, and the political and economic theories that inspire them.  It is highly recommended that interested people read my diary notes (referenced below) prior to studying this model.

**Conceptual Development** - **TpLab** is being written as part of a personal study of the dynamics of sustainable economics.  To understand where and how **TpLab** fits in, I need to recount some past history.  My earliest model economy was a purely biophysical ecosystem model called **PSoup** (short for Primordial Soup) in which the dynamics of evolution could be demonstrated (Ref D).  That was based on a model called **Simulated Evolution (SE)** by Dr Michael Palmiter (Ref E).  In that PSoup model, blind agents compete for limited amounts of energy, and they evolve a nuanced 8-parameter heuristic search pattern that is ever more effective in locating unseen resources.  The 8-parameter search pattern can be drawn on the eight points of the compass in a diagram I call a **clock face**.  In a PSoup arena, the MPP, via natural selection, shapes the population to function at remarkably high rates of consumption.  I have been using that ecological model as the foundational basis of several agent-based models over the past 15 years, and know it well.  **SE** is the basis of **PSoup**, **ModEco**, **MppLab** (Refs D, F & I below) - and now **TpLab**.

For **TpLab** I decided to take the compound genes which encode the agent's heuristic search pattern, and incorporate within them a parallel belief system which modifies the phenotype of each agent in way that the belief system directly enhances or reduces the effects of each gene.  The belief system has the same 8-parameter structure as the genetic material, and is super-imposed over the genetic material, but identifiable as a distinct component within the compound gene.  It can be manipulated independently of the genetic material, but, at the same time is subject to natural selection through the shared phenotype.  But, there is a difference: the belief systems can be modified by learning experiences over the lifetime of the agent, or by social sharing, whereas the genomes are cast at birth and unchanging throughout life.  By this means, the basic ecological model of **PSoup** is converted into a social model in **TpLab**.

**The BIG Question** - Why consider the phenomenon of teleological pruning?  This is driven by the question that many scientists have considered in recent years.  **In this modern world, why is economic theory so clearly unscientific, and how does it manage to remain resistant to all attempts to make it science-based?**  For example, in a world that is clearly depleted of most of its great treasures and on the brink of ecological collapse, to cure these global ills political and economic theory still recommends - indeed demands - an endless pattern of consumptive growth.  This appears to be sheer madness, but there are few asking why this emperor has no clothes.

**My FINAL answer** - All those who believe that something must be done to avert the coming catastrophe are participating in a belief system that is not aligned with the effects of the MPP.  That is to say, their belief system does not support the ever increasing rate of consumption of matter and energy until the resources are totally depleted.  That belief system, itself, is subject to evolutionary pressures, and is non-adaptive in an evolutionary sense, and is extirpated by natural selection.  Any economic or political theory that incorporates such obstructive beliefs is also extirpated by natural selection.  Modern political and economic theory is then impervious to all attempts to put science into it.  That is MY theory that I call "Teleological Pruning" (i.e. pruning of our theories, purposes and plans by natural selection).

**TpLab** is a hybrid system, having two intimately linked sub-systems - a biophysical sub-system encoded in a genome, and a teleological sub-system encoded in a set of beliefs:

-  In the biophysical sub-system, genes have fixed structure at birth, behavior is encoded as heuristic patterns in genes, and learning happens as the genes and the populations that hold them evolve, generation after generation.  Agents are born, struggle for survival, die of hunger or old age, and reproduce only when old enough and healthy enough.  
-  In the teleological sub-system, agents hold beliefs which alter their genome-derived behaviour.  These alterations in behaviour make them more or less efficient at finding food than might be expected from the heuristic encoded in their genome.
-  Success is determined by behaviour derived from both the biophysical genome and from the parallel belief system.

**Research Questions** - Here are my research questions for **TpLab**:  

-  Can I effectively model a society which is a hybrid between a biophysical ecosystem and a teleological system?
-  Can I demonstrate teleological pruning convincingly?  Do the belief systems always align with the heuristic search patterns that evolution produces in the genomes?    
-  Can desired and potent belief systems that are contrary to the effects of the MPP (e.g. we should not do it that way) nevertheless continue to persist?

Unfortunately, I believe that the answers that come out of this version of **TpLab** are: "Yes!", "Yes!" and "It seems not!".

## HOW IT WORKS

There are several scenarios which all work in a similar fashion, except for key minor design differences.  So, I will cover the common design details first, then outline the differences by scenario.  Variables and features are enclosed in square [brackets].  Those that are under direct user control, via sliders, are also in **[bold]**.

**Nrg** - energy, referred to in the model as nrg, exists only in the form of fruit in the arena and nrg in the bodies of the agents.  A flux of nrg comes from the Sun at the start of each tick, is deposited into fruit, is eaten by agents, and is expended by agents.  There are two mechanisms that throttle the rate of inflow of nrg.  There is a hard maximum amount of **[Sun-nrg-per-tick]** of inflow.  However, the maximum amount of nrg that comes from the Sun may also be limited by the size and state of the arena.  The amount that arrives per tick can be estimated using the formula [Nrg Flux] = [Number of empty cells] X **[Prob-of-deposit]** X **[nrg-per-deposit]**.  The actual nrg flux per tick is the lesser of those two amounts, and that determines the carrying capacity of the arena.  That flux of nrg is available to be expended by the agents in an amount of **[c2-epm-parm]** units of nrg per agent per move.

**Seekers** - the only breed of agents in this model is called Seekers.  They are seekers of energy-rich green fruit that appears in the brown arena at random times and random places.  But, the seekers are blind and cannot sense the nearby existence of fruit unless it occupies the same cell in which they stand.  Their efficiency in finding fruit depends on the heuristic search pattern they undertake, in competition with their contemporaries.  Those with a more effective search pattern will have a survival advantage over the others.  Seekers are born, move, feed, and live or die.  Few live more than 800 ticks, and a typical run of 100,000 ticks spans the lives of over 100 generations of seekers.  All of the characteristics of the lives of these agents are encoded in two sets of genes called chromosomes #1 and #2.

**Chromosome #1 - c1** - This chromosome contains eight compound genes built in a fashion first used by Dr Michael Palmiter, the author of **Simulated Evolution** (Ref E). In Dr Palmiter's design, a compound gene had two fundamental components that I label [B] and [G], and store in each seeker in lists called **c1-bases** and **c1-genes**.  Interpretation of each of these compound genes requires two calculations.  The [S]trength of the gene is calculated as the base [B] raised to the power [G].  This can be written as [Si]=[Bi]^[Gi], where 0 <= i <= 7.  Then the [P]henotypic character is the relative strength, calculated as the strength [S] of the gene divided by the sum of all eight strengths of all eight genes, written as [Pi]=100x[Si]/SUM([Sj]).  Seekers reproduce via fission, so each mother has two daughters, and perishes in the act, splitting her heritage of nrg between the two.  The values of [B] and [G] might possibly mutate (change randomly) by a small amount during reproduction.  The probability of mutation is set by the slider **[prob-of-mutation]**.

In **TpLab** there is one additional fundamental component that has been added to each of the eight compound genes - and that is the [L]earned belief.  [L] is added to [G] before calculating the [S]trength, so [Si]=[Bi]^([Gi]+[Li]).  [Pi] is calculated the same way as before.  

The use and interpretation of these genes is described below.

**Chromosome #2 - c2** - This chromosome contains the six genes that control biological functions.  In this model, these genes are static and do not change, but to maintain similarity with Palmiter's **SE** model, I encode them into each agent.  The six genes are:

-  **[c2-dat-parm]** - Death Age Threshold - Seekers that reach this age (in ticks) die of old age.  Either DAT or DET cause death.
-  **[c2-det-parm]** - Death Energy Threshold - Seekers whose energy level falls below this number (in nrg units) die of hunger.  Either DAT or DET cause death.
-  **[c2-rat-parm]** - Reproductive Age Threshold - Seekers that reach this age are eligible to reproduce.  RAT and RET but both be appropriate for reproduction.
-  **[c2-ret-parm]** - Reproductive Energy Threshold - Seekers that reach this nrg level are eligible to reproduce.  RAT and RET but both be appropriate for reproduction.
-  **[c2-epm-parm]** - Energy Per Move - Every agent expends this many nrg units per move.
-  **[c2-epa-parm]** - Energy Per Agent - Seekers cannot harvest fruit that will cause the nrg level to exceed this amount.

**C1 Phenotype** - The search pattern of the agents is controlled by the C1 genes.  If you number the eight genes from 0 to 7 then they can be arranged in a pattern that I call the **clock face**.  With each tick of the model, each agent randomly activates one of its genes, turns to the right an amount equal to that gene number times 45 degrees, then steps forward to feed.  So, it always enters one of the eight cells that surround it.  When a seeker moves, it first determines which of the eight genes is expressed, it turns to the right by the appropriate angle designated by the expressed gene, and then steps forward and attempts to feed.  As shown in the following diagram, the ultimate effect of each of the genes is:

-  0 -   0 degree turn -  F - (step forward), 
-  1 -  45 degree turn - FR - (step forward to the right), 
-  2 -  90 degree turn -  R - (step to the right), 
-  3 - 135 degree turn - BR - (step back to the right), 
-  4 - 180 degree turn -  B - (step backward), 
-  5 - 225 degree turn - BL - (step back to the left), 
-  6 - 270 degree turn -  L - (step to the left), or 
-  7 - 315 degree turn - FL - (step forward to the left).


                         Gene #s       Gene Interpretations
                       ------------      ---------------
                       | 7  0  1  |      | FL   F   FR |
                       |          |      |             |
                       | 6     2  |      | L         R |
                       |          |      |             |
                       | 5  4  3  |      | BL   B   BR |
                       ------------      ---------------

The mechanism used to decide which gene is expressed is the product of Dr Palmiter's genius.  The [probability of expression of each gene 'i'] is calculated using the phenotypic values [Pi] with this formula: [Pi]=[Si]/SUM([Sj]) for 0 <= i <= 7.  As an example, suppose the eight [S]trengths are 1, 1, 1, 1, 2, 1, 1 and 1.  The 1s are calculated as 2^0, and the 2 is 2^1.  Then gene 4 is twice as strong as the others, and has a probability of expression [Pi]=100x(2/9), while the others have a probability of expression of [Pi]=100x(1/9).  Both are expressed as percentage probabilities.


        An almost bland C1 chromosome.
     Gene #               (i=) 0   1   2   3   4   5   6   7
     ------                   -------------------------------
     [Bi]                      2   2   2   2   2   2   2   2
     [Gi]                      0   0   0   0   1   0   0   0
     ------                   -------------------------------
     [Si]=[Bi]^[Gi]            1   1   1   1   2   1   1   1
     [Pi]=100x[Si]/SUM([Sj])  1/9 1/9 1/9 1/9 2/9 1/9 1/9 1/9  (all x100) 


All scenarios start with a bland set of C1 genes - i.e. [Bi] = 2; [Gi] = 0; and [Si] = 1 for all i.  All genes in a bland C1 chromosome have an equal probability of expression in each turn.  This results in a search pattern that amounts to a random walk.  Such a walk tends to deplete the local resources for each seeker and leads to death, unless all seekers are equally bland and there is lots of nrg flowing in to replenish locally depleted nrg sources.  

Not long after a run starts, the [Bi]s and [Gi]s mutate slightly, minor alterations in search patterns appear, and competitive advantages emerge.  Different values of [Bi] and [Gi] across a genome represent different heuristic search patterns.  Drawn from my experience with **PSoup** (Ref D), here are a few phenotypic search patterns that may appear:

                Notable Phenotypic Search Patterns
            Arrow      Jitterbug   Bent Arrow     Dodger
         -----------  -----------  -----------  -----------
         |    ^    |  |         |  | \\      |  | \     / |
         |    |    |  |         |  |  \\     |  |  \   /  |  
         |    |    |  |    |    |  |   ||    |  |   | |   |
         |         |  |    |    |  |         |  |         |
         |         |  |    V    |  |         |  |         |
         -----------  -----------  -----------  -----------


Those compound genes not shown are very small, having a negative value for [Gi], and are expressed very rarely.  Natural selection chooses the most effective search patterns.  

-  The **Arrow** has a strong gene 0, and charges almost straight forward relentlessly.  It dominates a wrapped arena (i.e. wrapped at the edges) but is gradually reduced in numbers in a unwrapped or boxed in arena.  
-  The **Jitterbug**, has a strong gene 4, jumps back and forth between two cells, and is quickly extirpated as soon as resources become scarce.  It persists only when there is energy to spare.
-  The **Bent Arrow** has a strong gene 1 or gene 7, and tends to be an unstable transitional form on the evolutionary path to an arrow or a dodger.  It travels in wide sweeping circles that perform a random walk in slow time, as lesser genes are occasionally expressed. 
-  The **Dodger** has two strong genes, genes 1 and 7.  It dodges forwards in a jagged path that wanders in a far-ranging brownian-like motion.  It eventually dominates in arenas in which there are a lot of impediments to movement, such as an unwrapped (boxed in) arena.

Environments can be created in which other patterns emerge as the dominant form in a steady-state environment.  In Scenario 0 of **TpLab** (discussed below) I expect we will see arrows and dodgers as the arena may be either wrapped or unwrapped. The belief systems inject a bias towards a different search pattern.  The research questions revolve around this: What heuristic search patterns, if any, will emerge and become dominant in scenarios 1 through 3?

**Belief Systems** - I have injected a belief system into Palmiter's genes as follows.  There are eight learned belief values [Li] superimposed upon and stored along with the gene values [Gi].  The strength of each compound gene 'i' is then calculated as [Si] = [Bi]^([Gi]+[Li]), and the [Pi]s are calculated as usual.  The [Pi] values are used in the same way to determine which gene is expressed for any given turn.  

The [Bi]s, [Gi]s and [Li]s mutate on reproduction, but the [Li]s also can change through social interaction.  In two scenarios the [L]earned belief values are inherited from the mother and established for life for that agent.  In two scenarios they are shared within the society, and can spring to dominance or wane to insignificance.


        An almost bland C1 chromosome.
     Gene #               (i=) 0   1   2   3   4   5   6   7
     ------                   -------------------------------
     [Bi]                      2   2   2   2   2   2   2   2
     [Gi]                      0   0   0   0   1   1   0   0
     [Li]                      0   0   0   0   1  -1   0   0
     ------                   -------------------------------
     [Si]=[Bi]^([Gi]+[Li])     1   1   1   1   4   1   1   1


The belief system works as follows.  The [L]earned belief values are interpreted as moral convictions, or beliefs, or preferences, purposes, plans or intentions.  The emulate free-will decisions of the seekers.  There is one ultimately dominant search pattern that can be encoded in the [Bi]s and [Gi]s.  Each learned belief [Li] encodes what the seeker believes should be done to improve the outcome of a search, and the contribution to the [S]trength modifies the seeker's behaviour - modifies the actual search pattern.  The learned belief might enhance the effects of a [G]ene, as shown for [G4] in the above diagram, or it might cancel its effects, as shown for [G5].  It can be read as "I, the seeker, have learned to believe that I will find more fruit if I turn 180 degrees more often, and turn 135 degrees less often. 

Those learned beliefs that align with the ultimately dominant search pattern are TRUE BELIEFS, while those that disagree with the dominant search pattern are FALSE BELIEFS. So, while the seekers are seeking fruit, they are also collectively seeking for the TRUTH - seeking wisdom.  According to the MPP, truth is defined by evolution and natural selection as that pattern that leads to the highest rate of consumption of nrg by the system.  All learned belief systems that are not aligned with that dominant consumptive pattern must evolve and conform to it, or be rendered impotent.

**User Interface** - The user interface is organized into several panels, as follows:

-  **PANEL 01 - The Model** - Contains the arena where the seekers live, and the basic tools to select a scenario, with setup and go buttons.
-  **PANEL 02 - Advanced System Setup Parameters** - Contains the sliders that can be used to manage or modify the scenarios.  Most key parameters are available for user experimentation, but default parameters can be retrieved.
-  **PANEL 03 - Debug and Data Collection Tools** - Contains a variety of tools used during development that may not be of interest to users.  For those experienced with NetLogo, and who wish to study or modify this model, they remain there for your use.
-  **PANEL 04 - Clock Face - Averages, 8 Compound Genes** - Contains a simulation of the clock face showing the averages for each of the five components of each of the eight compound genes.  
-  **PANEL 05 - Population Graphics** - Contains graphs that display population counts broken out by learned belief affiliations and by tribal affiliations (see discussion of scenarios below).
-  **PANEL 06 - Energy Graphics** - Contains graphs that display various aspects of the the nrg stocks and flows, with break out by learned belief affiliations and by tribal affiliations (see discussion of scenarios below).
-  **PANEL 07 - Average Values - C1 Genes - By Component, By Gene #** - Contains five graphs, one for each type of component.
-  **PANEL 08 - Time Lines - Power and EROI (Moving Averages)** - These graphs provide a peek into the response of EROI and power to the evolution of the society.
-  **PANEL 09 - Nrg & Age Histograms, and Causes of Death** - The nrg histogram displays the usual Maxwell-Boltzmann distribution of wealth (in this case - nrg), and other graphs are of mild interest.

**SCENARIOS** - There are four scenarios available:

-  **SCENARIO 0 - Society With 0 Beliefs** - In this society all [Li]s are set to zero, and stay at zero throughout a run.  This replicates the pure biophysical demonstration of evolution as first demonstrated by Dr Palmiter in **Simulated Evolution**, and as replicated in **PSoup**.  There are no learned belief affiliations, and no tribal affiliations.  It is every seeker for herself.  Arrows and bent arrows dominate.
-  **SCENARIO 1 - Society with 8 Beliefs** - This is the same as Scenario 0 except each seeker is initially endowed with a single learned belief (i.e. one Li != 0) which is unchanging throughout life, and is passed on to offspring.  These represent an unchanging belief that one gene should have more weight that all of the rest.  This scenario poses the question "If an affiliation of seekers holds fast to a learned belief that is irreducible in effect, and poorly aligned with the effects of evolution, can it survive?"
-  **SCENARIO 2 - Society With Wisdom Sharing** - In this scenario, which starts out similar to Scenario 1, the seekers are able to learn - to adjust their learned beliefs as life proceeds.  Each seeker is a member of an affiliation of seekers having the same learned belief - that we can call a belief affiliation.  A belief affiliation might be considered to be like a professional association.  When seekers of the same belief affiliation meet, the eldest (the guru) can pass wisdom to the others.  Essentially, the [Li] of the younger seekers is adjusted to be more like the [Li] of the guru. 
-  **SCENARIO 3 - Society With Tribal Bonds** - This is like Scenario 2, but each seeker belongs to two social affiliations.  All seekers belong to a belief affiliation - of which there are a maximum of eight.  In addition, each seeker belongs to a tribal affiliation.  Each tribe originally includes seekers from all eight belief affiliations, and so is collectively agnostic in outlook.  You might consider them to be communities of seekers having a common interest in survival, such as might exist in a village, or tribe, or group of lifelong friends.  Seekers can seek professional wisdom from gurus having a common learned belief, as in Scenario 2.  However, they can also seek wisdom from tribal elders of their own tribe, in connection with the elder's own belief affiliation.  So, in this scenario a seeker may eventually hold beliefs of various strengths outside of their own inherent belief affiliation, and pass that very varied belief system on to its offspring.

## HOW TO USE IT

You have heard, I suppose, of one-player games.  One of my tutored students calls programs like this a zero-player game.  You set the parameters, you start it, and you get yourself a glass of port, and sit and watch it develop.

There are four different scenarios, as described above, and each of those can be run with different random number seeds.

To perform a run of the model:

-  In PANEL 1 select a scenario options;
-  Select a random seed;
-  Enter a 'tick' at which you want the run to stop;
-  Optionally, go to PANEL 2 and adjust any of the advanced controls;
-  Back on PANEL 1, click on the **Setup** button to configure the system according to the scenario chosen;
-  At the top of PANEL 1, adjust the speed to maximum; and
-  Click on the **Go** button.

More adventurous users might scroll to the right and find the panel of "advanced" sliders which can be used to alter a number of parameters of the model.  This is not recommended until you understand the standard operation of the model, but there is a lot to learn there.  Many combinations of these sliders will be immediately fatal to the population.  I have not built in fail-safe code.  But if you adjust them a little, you will see things happen.  Their effect is explained in some detail in the user documentation (Ref K).  The 'reset defaults' button found there will bring you back to ground zero if need be.  

The following process will show the progressive effects of belief systems, and the social spread of wisdom:

-  Run scenario 0 (Society With 0 Beliefs) two or three times, to 15,000 ticks each, and watch the difference between the evolution of the forward-motion genes (green) and the backward-motion genes (red).  This can be seen best in the graph "Average [P]henotypic Values" in PANEL 6.  In this scenario belief systems have absolutely no effect.  This is the baseline scenario against which all others should be compared.
-  Predict what you think will happen when every seeker has a strong unalterable preference for turning a given distance.  Run scenario 1 (Society With 8 Beliefs) once, to 15,000 ticks.  Compare the results with those noted in scenario 0.
-  Predict what you think will happen when younger seekers can strengthen or reduce their learned belief by contact with older 'wiser' seekers holding the same views.  Run scenario 2 (Society With Wisdom Sharing) once, to 15,000 ticks.  Again, compare the results with those noted in previous scenarios.
-  Now, predict what you think will happen when seekers seek wisdom not only from those with the same belief affiliation, but also from other elders in their own tribe having very varied learned beliefs.  Then, run scenario 3 (Society With Tribal Bonds)                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          