breed [readers reader]
breed [booksellers bookseller]
breed [manuscripts manuscript]

readers-own [
  book-list
  number-of-books
  money
  curiosity
  my-seller
  ]

booksellers-own [
  my-manuscripts
  min-title-diversity
  preferred-print-run
  price-target
  career-span
  capital
  ]



manuscripts-own [
  my-publisher
  title
  editions
  appeal
  price
  depreciation-rate
  book-age-threshold
  earnings
  expenses
  copies-on-the-shelf
  shelf-time
  viable
  failed-solicitations
  ]



to setup
  ca
 
  ; Booksellers publish and distribute manuscripts. They have a capital of money, and if they slip
  ; into too much debt, they go bankrupt and die.
  set-default-shape booksellers "house"
  create-booksellers initial-booksellers [ 
    set color brown
    set size 3
    set career-span 0
    set min-title-diversity 0
    set preferred-print-run 200 - random 190
    set price-target 0.5 + random-gamma 1 2
    set my-manuscripts nobody
    set capital initial-capital
    setxy random-xcor random-ycor
    ]
   
  ; Readers' behavior is controlled by several variables: color, closed-mindedness, and money. All readers will
  ; seek out books of their own color. If they are curious (that is, not incurious) they will also
  ; look for books of different color. They gain money every tick and use money to buy books.
  set-default-shape readers "circle"
  create-readers number-of-readers [
    set color 55 + random-float 4
    set money 0.5 + random-gamma 1 2
    set book-list []
    set size 1.5
    set curiosity random 100
    set my-seller one-of booksellers
    setxy random-xcor random-ycor
  ]
  
  ; Manuscripts are owned by booksellers. They are "published" by hatching
  ; a number of books that share their variables.
  set-default-shape manuscripts "star"

  reset-ticks
end

; Go activates readers and booksellers.
to go
  tick
  ask readers [ determine-behavior ]
  ask booksellers [manage-inventory check-accounts]
  if not any? booksellers [ stop ]
  if mean [capital] of booksellers > initial-capital * 2 or mean [career-span] of booksellers > 500 [ create-booksellers 1 [
    set color brown
    set size 3
    set career-span 0
    set preferred-print-run 200 - random 190
    set price-target 0.5 + random-gamma 1 2
    set my-manuscripts nobody
    set capital initial-capital
    setxy random-xcor random-ycor ] 
  ]
  
end

to determine-behavior
  set curiosity curiosity + 1
  if my-seller = nobody [ set my-seller one-of booksellers ]
  ifelse curiosity > curiosity-threshold 
    [ buy-books ]
    [ move ]
end

; Readers move about randomly and carry their books with them.
to move                
  rt random 50
  lt random 50
  fd 0.5
end
   
; Reader look for books to buy. The decision to purchase a book is based on
; a simple randomization of the books' appeal. 
to buy-books
  if my-seller != nobody [ face my-seller ]
  fd 1
  
  if count booksellers in-radius seller-radius > 0 [
  let temp-list [book-list] of self
  let books-to-buy manuscripts in-radius seller-radius with [ ; creates agentset of nearby books   
    copies-on-the-shelf > 0 and                        ; must be owned by bookseller
    not member? title temp-list and
    (appeal + random-float 1 - random-float 1) >= price and  ; appeal (randomly adjusted) must be higher than price
    price <= [money] of myself                               ; reader must have enough money
    ]
  
  let chosen-book one-of books-to-buy                         ; reader chooses one of the eligible books
  ifelse chosen-book != nobody [
    let chosen-seller [my-publisher] of chosen-book
    set book-list fput [title] of chosen-book book-list
    set number-of-books length book-list
    set curiosity 0
    ask chosen-seller [                             
      set capital capital + [price] of chosen-book ]           ; booksellers account is increased
    ask chosen-book [ 
      set earnings earnings + price 
      set copies-on-the-shelf copies-on-the-shelf - 1 ]  
  ] 
  [ 
    set my-seller one-of booksellers 
    ]
  ]
end

; Booksellers check their inventory.
to manage-inventory
  set capital capital - overhead
  set career-span career-span + 1
  set min-title-diversity abs capital / 100
  set my-manuscripts manuscripts with [my-publisher = myself]
  ask my-manuscripts [update-mss]
  
  if count my-manuscripts with [viable = true] <= min-title-diversity [ acquire-copy ]
  
  if count my-manuscripts with [copies-on-the-shelf > 0] < min-title-diversity [ print-mss ]
  
end
  
; When sitting on the shelf, books lose appeal over time.
to update-mss
  set appeal appeal - depreciation-rate
  if earnings >= expenses [ 
    set viable true 
    set shelf-time 0 
    ]
  if expenses > earnings [
    set viable false
    set shelf-time shelf-time + 1 
    ]
  if shelf-time > book-age-threshold [ 
    ask my-publisher [ set capital capital + ([price] of myself * pulp-rate) ]
    die
    ]
  if failed-solicitations > 5 [ die ]
  if appeal < 0 [ die ]
end

; Bookseller creates a new title.
to acquire-copy
    hatch-manuscripts 1 [
    set size 1
    set color 25 + random-float 4
    set title random-float 100 ; The title create an identifier that can be inherited by copies
    set viable true
    set appeal 1.5 - random-float 1 
    set my-publisher myself
    set price [price-target] of myself - random-float .5
    set depreciation-rate 0.01 - random-float .009
    set book-age-threshold 700 - random 600
    ]
end

  
; Manuscripts "hatch" copies of themselves as books. When manuscripts hatch books
; those books retain the original's title, appeal, and price.
to print-mss
  let chosen-manuscript one-of my-manuscripts with [ viable = true ]
  if chosen-manuscript != nobody [
  set capital capital - (fixed-cost + ([price] of chosen-manuscript * (preferred-print-run * per-unit-cost)))
  ask chosen-manuscript 
    [
      set expenses expenses + (fixed-cost + ([price] of chosen-manuscript * ([preferred-print-run] of myself * per-unit-cost)))
      set copies-on-the-shelf copies-on-the-shelf + [preferred-print-run] of myself
      ]
  ]
end

; Booksellers check to see if they have gone bankrupt.
to check-accounts
  if capital < -200 [ 
    ask my-manuscripts [ die ]
    die 
    ]
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
870
691
32
32
10.0
1
10
1
1
1
0
0
0
1
-32
32
-32
32
0
0
1
ticks
30.0

BUTTON
7
16
73
49
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
92
18
155
51
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
12
250
184
283
per-unit-cost
per-unit-cost
0.1
.9
0.4
.1
1
NIL
HORIZONTAL

SLIDER
6
61
176
94
number-of-readers
number-of-readers
0
2000
1000
20
1
NIL
HORIZONTAL

SLIDER
8
115
180
148
initial-capital
initial-capital
10
200
50
10
1
NIL
HORIZONTAL

SLIDER
10
157
182
190
overhead
overhead
0
1
0.2
.02
1
NIL
HORIZONTAL

SLIDER
10
203
182
236
fixed-cost
fixed-cost
1
50
10
1
1
NIL
HORIZONTAL

SLIDER
12
294
184
327
pulp-rate
pulp-rate
.01
.2
0.05
.01
1
NIL
HORIZONTAL

SLIDER
14
462
186
495
seller-radius
seller-radius
1
10
3
1
1
NIL
HORIZONTAL

BUTTON
39
519
120
552
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
1087
14
1287
164
Number of Booksellers
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot count booksellers"

PLOT
1088
180
1636
485
Total Inventory
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Number of Books" 1.0 0 -16777216 true "" "plot sum [copies-on-the-shelf] of manuscripts"
"Total cash" 1.0 0 -2674135 true "" "plot sum [capital] of booksellers"

PLOT
1087
525
1287
675
Total Available Titles
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot count manuscripts with [copies-on-the-shelf > 0]"

SLIDER
17
574
189
607
initial-booksellers
initial-booksellers
1
10
1
1
1
NIL
HORIZONTAL

MONITOR
1338
527
1440
572
Avg Seller Age
mean [career-span] of booksellers
0
1
11

MONITOR
1338
582
1404
627
Old Hats
count booksellers with [career-span > 4000]
0
1
11

MONITOR
1341
639
1405
684
Newbies
count booksellers with [career-span < 500]
0
1
11

MONITOR
1411
583
1543
628
Num of Booksellers
count booksellers
0
1
11

MONITOR
1413
639
1517
684
Avg Print Runs
mean [preferred-print-run] of booksellers
0
1
11

SLIDER
11
336
185
369
curiosity-threshold
curiosity-threshold
0
100
50
1
1
NIL
HORIZONTAL

MONITOR
1475
527
1532
572
Avg $
mean [capital] of booksellers
0
1
11

MONITOR
1317
19
1396
64
Mean Price
mean [price-target] of booksellers
2
1
11

MONITOR
1417
21
1529
66
High End Sellers
count booksellers with [price-target > mean [price-target] of booksellers] / count booksellers
2
1
11

MONITOR
1323
79
1413
124
$ of Readers
mean [money] of readers
1
1
11

MONITOR
1439
84
1571
129
Oldest Hat
max-one-of booksellers [career-span]
0
1
11

MONITOR
1440
131
1571
176
Richest Seller
max-one-of booksellers [capital]
0
1
11

MONITOR
1578
84
1694
129
Old Hats' Pricing
mean [price-target] of booksellers with [career-span > 4000]
2
1
11

MONITOR
1582
137
1677
182
Old Hats' PRs
mean [preferred-print-run] of booksellers with [career-span > 4000]
2
1
11

@#$#@#$#@
# Bookshops

## Purpose

This simulation explores the causes and indirectly observed behaviors of seventeenth-century London booksellers. It examines when and how books run through multiple editions, when they are held for sale, and when they are reverted to pulp. Its main purpose is to test the relative importance of production and inventory costs to booksellers' success.

## Entities, state variables, and scales

The model contains the following entities: readers, booksellers, books, and manuscripts. Each reader is assigned a single, unchanging state variable, "money." Money for readers does not represent an account that rises and falls, but instead represents their general purchasing power. Readers maintain collections of books (kept as a list of titles). Booksellers are hatched in random positions but do not move. They maintain a money account ("capital") that increases and decreases as they sells and publish books. They produce and maintain collections of books and manuscripts. If they go too deeply into debt, they goes bankrupt and die. If all booksellers die the simulation stops.

Manuscripts have unique and unchanging variables: title, price, appeal, my-owner. Books are hatched by manuscripts and retain their parents' variables. As books sit in a bookseller's inventory waiting to be purchased, their appeal gradually decreases. As their appeal falls, they are less likely to be purchased, and when their appeal reaches zero, they die.

The model is greatly compressed in time, space, and number. Each tick is meant to represent approximately one day of real time. The size of the space and the number of readers, books, and booksellers (versus seventeenth-century London) is reduced by an adjustable factor of approximately 100 to 1.

Money is not meant to directly reflect actual monetary units. Various price values are meant to be tested in relation to each other. What percentage of printing costs are fixed? What is the proportion between a book's average price and the volatility of a booksellers' capital account? What is the ideal proportion of print runs to book-buyers? How do changing financial conditions change the kinds of bookselling businesses that are successful.

## Process overview and scheduling

Booksellers manage their inventory by looking through the agentset of their books and manuscripts. They ask their books to "age" and they look for opportunities to reprint existing manuscripts or publish new ones.

When first published, books are given an "appeal" score. Every tick, that score depreciates according to an adjustable rate. If the books lose all consumer appeal, the bookseller will sell them off at a cheap rate as paper-stock. 

If a bookseller's inventory falls below the adjustable "inventory-to-print-run" rate, they review their existing manuscripts. They then call their manuscripts to perform the procedure "review-for-reprint." When reviewed, manuscripts check themselves to see if they remain viable. If no copies of the manuscript exist (that is, if the book failed to sell and all copies were destroyed), the manuscript is destroyed. If the manuscript is still selling, it will be reprinted.

If a bookseller's stock of manuscripts falls below the minimum level, he will print new ones.

Each tick, the following sequence of procedures is called:
1. Readers move randomly and look for books to buy.
2. If readers move through the bookstore, they might purchase a book.
3. The bookseller examines inventory to see if it meets minimum quantity and diversity
4. Bookseller calls inventory, the agentset "my-books," to perform procedures
5. Books in inventory gradually decrease in appeal according to the "depreciation rate"
6. Books with appeal zero die, then ask parent manuscript to retire.
7. Bookseller publishes new books by asking existing manuscripts to hatch new books. 
8. The bookseller calculates his account and confirms that he has not gone bankrupt. 

## Design Concepts

<em>Adaptation.</em> The bookseller is constantly evaluating his inventory, discarding unsold books and attempting to maintain an inventory with maximum appeal.

<em>Fitness.</em> Fitness of a book is determined by how much appeal it has. If many copies sell, it will be reprinted and maintain its appeal. If it sits unsold, it will be pulped and its parent manuscript will be retired.

<em>Prediction.</em> The bookseller has no ability to predict the success of a newly published manuscript, but will reprint works that have sold well in the past. If a title fails to sell, it will not be reprinted.

<em>Interaction.</em> Readers choose books by weighing the books' appeal versus the books' price. (These interactions are adjusted randomly in each case.) The bookseller controls the books by managing his inventory.

## Initialization

At set up, the bookseller is placed at the center and readers are distributed randomly. Various factors are controlled using sliders. These are:

<b>1. Number of readers.</b> Ranges from 0 to 500 in increments of 20. 
<b>2. Initial capital.</b> Ranges from 100 to 1000 in increments of 20. This sets the booksellers' initial money account.
<b>3. Overhead.</b> Ranges from 0 to 1 in increments of .02. This decreases the bookseller's account each tick.
<b>4. Average price.</b> Ranges from 3 to 7 in increments of 1. This sets the average price of each book.
<b>5. Depreciation rate.</b> Ranges from 0 to .5 in increments of .1. This sets the rate at which books lose appeal each click.
<b>6. Compositor rate.</b> Ranges from 10 to 50 in increments of 5. This sets the fixed cost for any new edition.
<b>7. Printing cost.</b> Ranges from .1 to .9 in increments of .1. This sets the <em>percentage</em> of per-copy cost of new editions.
<b>8. Print run.</b> Ranges from 5 to 100 in increments of 5. This sets the average size of each print run.
<b>9. "Inventory to print run."</b> Ranges from .5 to 5 in increments of .5. This sets the ratio of inventory the bookseller keeps in relation to the average print run.
<b>10. Minimum title diversity.</b> Ranges from 1 to 10 in increments of 1. Sets the minimum number of book titles the bookseller will keep on hand at any moment.

## Submodels



## CREDITS AND REFERENCES

This model was created by Michael Gavin, University of South Carolina. 2013.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Printing Cost" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>sum [new-titles] of booksellers</metric>
    <metric>count manuscripts with [viable = false]</metric>
    <metric>sum [new-editions] of booksellers</metric>
    <metric>sum [capital] of booksellers</metric>
    <steppedValueSet variable="printing-cost" first="0.1" step="0.1" last="0.9"/>
  </experiment>
  <experiment name="Compositor Rate and Print Run" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <enumeratedValueSet variable="min-title-diversity">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-capital">
      <value value="200"/>
    </enumeratedValueSet>
    <steppedValueSet variable="number-of-readers" first="100" step="100" last="500"/>
    <enumeratedValueSet variable="printing-cost">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="compositor-rate">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="overhead">
      <value value="0.1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="print-run" first="10" step="10" last="100"/>
    <enumeratedValueSet variable="seller-radius">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="pulp-rate">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inventory-to-print-run">
      <value value="2.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Print-run to Population" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <enumeratedValueSet variable="pulp-rate">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seller-radius">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="print-run">
      <value value="20"/>
      <value value="30"/>
      <value value="40"/>
      <value value="50"/>
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="compositor-rate">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-readers">
      <value value="100"/>
      <value value="200"/>
      <value value="300"/>
      <value value="400"/>
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inventory-to-print-run">
      <value value="2.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="overhead">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-capital">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-title-diversity">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="printing-cost">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Initial-capital to Lifespan" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <enumeratedValueSet variable="pulp-rate">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seller-radius">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="print-run">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="compositor-rate">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-readers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inventory-to-print-run">
      <value value="2.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="overhead">
      <value value="0.1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="initial-capital" first="100" step="50" last="500"/>
    <enumeratedValueSet variable="min-title-diversity">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="printing-cost">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
