;;____________________________
;
;;   SETUP AND HOUSEKEEPING
;;____________________________

breed [ parties party ]

globals [
  total-votes
  max-voteshare            ; largest vote share on any patch
  mean-voterx              ; mean voter x-coordinate
  mean-votery              ; mean voter y-coordinate
  
  cycle
  election                 ; election number

  n-parties                ; total number of parties that competed in the previous election
  voter-misery             ; mean quadratic Euclidean voter distance from closest party
  mean-eccentricity        ; mean Euclidean distance of parties from (mean-voterx, mean-votery)
  enp                      ; effective number of parties = 1/(sum of squared party vote shares)
  
  rule-number              ; rule number
  rule-list                ; list of available decision rules
  rule-voteshare           ; list of vote shares won by the set of parties using each rule
  rule-pcount              ; number of parties using rule
  rule-eccent              ; mean eccentricity of parties using rule
]

parties-own [
  rule                      ; party's parameterized decision rule
  species                   ; party's decision rule species
  
                     ; decision rule parameters
  speed                     ; distance each party moves per tick
  comfort-kappa             ; comfort threshold - implement rule iff fitness below this
  neighborhood-eta          ; radius from party postion of local neighborhood
  
                     ; indicators
  mysize                    ; current party size
  old-size                  ; party's size previous tick
  old-x                     ; x-coordinate of my position at previous election
  old-y                     ; y-coordinate of my position at previous election
  age                       ; number of elections survived since birth
  fitness                   ; party's evolutionary fitness
  eccentricity              ; party's Euclidean distance from (mean-voterx, mean-votery)
  valence                   ; party's non-policy valence
  
                     ; rule-specific variables
  prey                      ; for predators:  party I am thinking of attacking
  best-x                    ; for explorers:  explored x-coordinate with highest vote share during campaign
  best-y                    ; for explorers:  explored y-coordinate with highest vote share during campaign
  best-size                 ; for explorers:  highest vote share of party during campaign
]

patches-own [
  votes                     ; number of voters on patch
  vote-share                ; proportion of total voters on patch
  closest-party             ; party with smallest Euclidean distance from patch
  best-party                ; party giving voter greatest utility (given both policy positions and valences)
  misery                    ; quadratic distance from closest party, weighted by patch's vote share
]

to setup
  clear-all
  file-close  
    
  if (endogenous-birth = false) [set misery-alpha 0 set misery-beta 0]  
  
  if (birth-death-file = true) [
    if (file-exists? bd-file-name) [file-delete bd-file-name]
    file-open bd-file-name
  ]

  set rule-list (list 
      "S1000a" "S2000a" "S3000a" "S4000a" "S5000a"
      "S1000b" "S2000b" "S3000b" "S4000b" "S5000b"
      "S1000c" "S2000c" "S3000c" "S4000c" "S5000c"
      "S1000d" "S2000d" "S3000d" "S4000d" "S5000d"
      "S1000e" "S2000e" "S3000e" "S4000e" "S5000e"
      "S1000f" "S2000f" "S3000f" "S4000f" "S5000f"
      "S1000g" "S2000g" "S3000g" "S4000g" "S5000g"
            
      "A1000a" "A2000a" "A3000a" "A4000a" "A5000a"
      "A1000b" "A2000b" "A3000b" "A4000b" "A5000b"
      "A1000c" "A2000c" "A3000c" "A4000c" "A5000c"
      "A1000d" "A2000d" "A3000d" "A4000d" "A5000d"
      "A1000e" "A2000e" "A3000e" "A4000e" "A5000e"
      "A1000f" "A2000f" "A3000f" "A4000f" "A5000f"
      "A1000g" "A2000g" "A3000g" "A4000g" "A5000g"

      "H1006" "H2006" "H3006" "H4006" "H5006"
      "H1011" "H2011" "H3011" "H4011" "H5011"
      "H1016" "H2016" "H3016" "H4016" "H5016"
      "H1021" "H2021" "H3021" "H4021" "H5021"
      "H1026" "H2026" "H3026" "H4026" "H5026"
      "H1031" "H2031" "H3031" "H4031" "H5031"
      "H1100" "H2100" "H3100" "H4100" "H5100"

      "P1006" "P2006" "P3006" "P4006" "P5006"
      "P1011" "P2011" "P3011" "P4011" "P5011"
      "P1016" "P2016" "P3016" "P4016" "P5016"
      "P1021" "P2021" "P3021" "P4021" "P5021"
      "P1026" "P2026" "P3026" "P4026" "P5026"
      "P1031" "P2031" "P3031" "P4031" "P5031"
      "P1100" "P2100" "P3100" "P4100" "P5100"

      "E1006" "E2006" "E3006" "E4006" "E5006"
      "E1011" "E2011" "E3011" "E4011" "E5011"
      "E1016" "E2016" "E3016" "E4016" "E5016"
      "E1021" "E2021" "E3021" "E4021" "E5021"
      "E1026" "E2026" "E3026" "E4026" "E5026"
      "E1031" "E2031" "E3031" "E4031" "E5031"
      "E1100" "E2100" "E3100" "E4100" "E5100"
  )    
      ;; the 5-character rule names store the parameterization of each rule, as follows:
      ;; character 1 = rule species; chars 2 = exogenous valence level; chars 3-5 = comfort kappa
      ;; for example, H2012 is a Hunter with exogenous valence level 2 and kappa = 0.12
      ;; using valence levels rather than values allows values to be changed in load-rule-parameters without editing all rule names
        
  set rule-number n-values length(rule-list) [?]
  set rule-voteshare n-values length(rule-list) [0]
  set rule-pcount n-values length(rule-list) [0]
  set rule-eccent n-values length(rule-list) [-1]

  create-parties 1
  ask parties [set fitness 1 set heading random-float 360 jump random-float 30 set old-x xcor set old-y ycor  
               set age 0 set size 2 random-pick load-rule-parameters color-myself ]          
               ;; every run starts with a single party, which has a random position and rule picked from the rule list
          
  ask patches [ 
      let x1 (pxcor - x-mean1) / sd-1
      let y1 (pycor - y-mean1) / sd-1      
      set votes votes1 * exp (-0.5 * ( x1 ^ 2 + y1 ^ 2)) / (2 * pi * sd-1 ^ 2)
        ;; votes1, x_mean1, y_mean1, sd_1 = votes, mean and standard deviation of subpopulation 1, read from interface
        ;; each patch's votes arising from subpopulation 1 =  votes1 * bivariate normal density with mean1, sd_1, rho = 0
      
      if (votes2 > 0)[  
      let x2 (pxcor - x-mean2) / sd-2
      let y2 (pycor - y-mean2) / sd-2      
      set votes votes + votes2 * exp (-0.5 * ( x2 ^ 2 + y2 ^ 2)) / (2 * pi * sd-2 ^ 2)]
        ;; add votes to each patch from subpopulation 2, calculated as above
      
      if (votes3 > 0)[    
      let x3 (pxcor - x-mean3) / sd-3
      let y3 (pycor - y-mean3) / sd-3      
      set votes votes + votes3 * exp (-0.5 * ( x3 ^ 2 + y3 ^ 2)) / (2 * pi * sd-3 ^ 2)]
        ;; add votes to each patch from subpopulation 3, calculated as above
      ]
      
  set total-votes sum [ votes ] of patches
  type "Total votes at all locations = " type round(total-votes)     
        ;; add total of votes on all patches and output this to the command window
  
  ask patches [set vote-share votes / total-votes]      
      ;calculate each patch's vote share
  
  set mean-voterx sum [ pxcor * vote-share ] of patches
  set mean-votery sum [ pycor * vote-share ] of patches      
  type "   Mean voter x = " type round(mean-voterx) 
  type "   Mean voter y = " print round(mean-votery)
      ;; calculate center (mean) of voter distribution on each dimension as sum of (patch positions weighted by vote share)
      ;; output this to the command window 
  
  set max-voteshare max [ vote-share ] of patches
  ask patches [set pcolor scale-color gray vote-share 0 max-voteshare ] 
      ;; color patches red with density proportional to vote shares
  
  update-support
      ;; ask voters to choose closest party and calculate relative success of different rules
      
  update-rule-measures
  
end  

; ******* parameter setup buttons
to random-pop
   set sd-1 5 set sd-2 5 set y-mean1 0 set y-mean2 0
   set x-mean2 precision (random-float 15)  2    set x-mean1 0 - x-mean2              
   set votes1 500000 + random 166667  set votes2 1000000 - votes1
end

to symmetric-pop
   set sd-1 10 set x-mean1 0 set y-mean1 0 set votes1 1000000  set votes2 0 set votes3 0 set x-mean2 0
end


;;____________________________
;
;;   PARTY DYNAMICS
;;____________________________

to stick
      ;; do nothing
end 


to aggregate
   if (mysize > 0) 
  [
     let xbar (sum [votes * pxcor] of patches with [best-party = myself] / mysize)
     let ybar (sum [votes * pycor] of patches with [best-party = myself] / mysize)
     
     if (xbar) > max-pxcor [set xbar max-pxcor]
     if (xbar) < min-pxcor [set xbar min-pxcor]
     if (ybar) > max-pycor [set ybar max-pycor]
     if (ybar) < min-pycor [set ybar min-pycor]

     let dist distancexy xbar ybar
     facexy xbar ybar
     ifelse (dist >= speed) [jump speed] [setxy xbar ybar]
   ]
      ;; identify xbar, ybar, the mean x, y positions of current party members
      ;; if you will not overshoot (xbar, ybar) face this and jump distance "speed" towards (xbar, ybar) 
      ;; maintain current position if zero supporters
end 


to hunt
  ifelse (mysize > old-size) [jump speed] [set heading heading + 90 + random-float 180  jump speed]
      ;; hunter makes a move of size speed in same direction as previous move if this increased party support
      ;; else reverses direction and makes a move of size speed in on a heading chosen from the 180 degree arc now faced 
  set old-size mysize 
      ;; remember party size for next cycle
end


to sat-hunt
   ifelse (mysize / total-votes < comfort-kappa) [hunt] [stick]
end
      ;; hunt if vote share is below comfort-kappa, else stand still


to predate
   let me mysize
   set prey min-one-of other parties with [mysize > me] [distance myself]
       ;; find the closest larger party
   if prey != nobody [
      let xdest [xcor] of prey
      let ydest [ycor] of prey                                                
      facexy xdest ydest
      let dist distancexy xdest ydest
      ifelse (dist >= speed) [jump speed] [setxy xdest ydest]
         ;; if you will not overshoot (xdest, ydest), jump "speed" towards this          
   ]
end


to sat-predate
  ifelse (mysize / total-votes < comfort-kappa) [predate] [stick]
end
      ;; predate if vote share is below comfort-kappa, else stand still

      
to explore
  if (mysize > best-size and abs(best-x) < max-pxcor and abs(best-y) < max-pycor) [set best-x xcor set best-y ycor set best-size mysize]
      ;; NB voter support is updated AFTER all party adaptation is finished 
      ;; so the effect on party support of a move made the previous tick is assessed BEFORE adaptation this tick
      ;; if you found a position with more support than your previous best-size, update your best position
  ifelse (remainder cycle campaign-ticks != 0)           
     [setxy old-x old-y set heading random-float 360 jump random-float neighborhood-eta]
         ;; if there is no election, test a random position within eta of your position at the last election
     [if (best-size > old-size) [setxy best-x best-y] 
         ;; if there is an election, go to a better postion iff you have found one
          set best-size 0]  
              ;;after the election zero best-size, which stores the best size found in inter-electoral exploration
              ;;NB in update-party-measures, all parties update old-x, old-y and old-size after each election  
end 
      ;; Each NON-ELECTION tick, explorer tests random positions "close" to its position at the last election.
      ;; "Close" is defined by neighborhood-eta, the maximum number of patches a party can move between elections.
      ;; During exploration, Explorer updates best-x, best-y, best-size iff tested co-ordinates 
      ;; generate more support than the previous best-xy co-ordinates
      ;; Each ELECTION tick, explorer sets its position as the best location found in inter-electoral exploration,
      ;; Provided this yields more support than the previous electoral position


to sat-explore
  ifelse (mysize / total-votes < comfort-kappa) [explore] [stick]
end
      ;; explore if vote share is below its comfort-kappa, else stand still
      

;;____________________________
;;
;;   MAIN CONTROL SUBROUTINES
;;____________________________

to update-support
  ask patches [set closest-party min-one-of parties [distance myself]]
      ;; patches find their closest party
  ask patches [set best-party max-one-of parties                                                   ;;the valence model enters here ******
      [ valence-lambda * valence - (1 - valence-lambda) * ((distance myself) ^ 2)]]                                                ;*****
      ;; patches find their "best" party - the one that maximizes their valence-weighted quadratic utility                         ******
      ;; NB distances and thus valence are scaled in NetLogo patch units                                                           ******
      ;; i.e. a distance differential of 1 sd = 10 units from the ideal point is balanced when lambda = 0.5                        ******
      ;; by a valence differential of 100
      ;; setting valence-lamda = 0 gives the no-valence model                                                                      ******
  ask parties [set mysize sum [votes] of patches with [best-party = myself]]                                                       ;*****                     
      ;; each party sums the votes on patches for which it is the best party                                                       ******
end

to calculate-election-results
  set election election + 1
  update-party-measures
  update-rule-measures
  measure-enp
  measure-eccentricity
  measure-misery
  party-death
  party-birth
end

to update-party-measures                                   
  ask parties [
      set fitness fitness-alpha * fitness + (1 - fitness-alpha) * mysize / total-votes
                 ;; parties recursively update their fitness as: (alpha)*(previous fitness) + (1-alpha)*(current vote share) 
      set age age + 1 set old-x xcor set old-y ycor
  ]
  
  set n-parties count parties  
end

to update-rule-measures
   (foreach rule-number rule-list [ 
      set rule-voteshare replace-item ?1 rule-voteshare sum [mysize / total-votes] of parties with [rule = ?2]
          ;; calculate the current support level of all parties using each rule
        
      set rule-pcount replace-item ?1 rule-pcount count parties with [rule = ?2]
          ;; count the number of parties using each rule

      ifelse (sum [mysize] of parties with [rule = ?2] > 0) 
        [
        set rule-eccent replace-item ?1 rule-eccent mean [eccentricity] of parties with [rule = ?2] 
        ]
          ;;calculate the mean of eccentricity, policy loss and policy shift of all parties using each rule
          
        [ 
        set rule-eccent replace-item ?1 rule-eccent -1 
        ]
          ;;these measures have no meaning when no party uses a rule
    ])  
end

to measure-enp
  set enp (total-votes ^ 2) / (sum [mysize ^ 2] of parties)
     ;; calculate the enp of the system
end

to measure-eccentricity
  ask parties [set eccentricity sqrt ((xcor - mean-voterx) ^ 2 + (ycor - mean-votery) ^ 2) / 10] 
     ;; calculate each party's eccentricity, its Euclidean distance from the center of the voter distribution
  set mean-eccentricity mean [eccentricity] of parties
     ;; calculate the mean eccentricity of all parties in the system
end

to measure-misery
   ask patches [set misery misery-alpha * misery + (1 - misery-alpha) * ((distance closest-party ^ 2) / 100) * vote-share]
   set voter-misery sum [misery] of patches
      ;; patch misery is misery at t-1, updated by mean quadratic Euclidean distance of patch from closest party, 
      ;; weighted by patch vote share
      ;; mean voter "misery" is thus updated mean quadratic Euclidean distance of each voter from his/her closest party
end

to party-death
   ask parties [if (fitness < survival-threshold and count parties > 2) 
       [
          if (birth-death-file = true) [
              file-write votes1 file-write x-mean1 file-write votes2 file-write x-mean2 file-write fitness-alpha file-write survival-threshold
              file-write campaign-ticks file-write misery-alpha file-write misery-beta
              file-write valence-lambda
              file-write election file-write "death" file-write rule file-write who 
              file-write precision xcor 4 file-write precision ycor 4 file-write "age" file-write age file-print ""
          ] 
          die 
          ask patches [set closest-party min-one-of parties [distance myself]]
       ]]   
                 ;; parties whose updated fitness falls below the survival threshold write out their data and die
                 ;; as long as there are at least two parties
end

to party-birth
  ifelse (endogenous-birth = true)
    [ ask one-of patches with [distancexy 0 0 < 30]
      [ if (random-float 1 < (misery-beta * misery * 1000)) [sprout-parties 1 [initialize-party] ]]]
        ;; pick a random patch within three standard deviations of the origin.
        ;; the probability this patch sprouts a new party is proportional to (misery-beta)*(patch misery)
        ;; the greater patch misery, the higher the probability the patch sprouts a new party.
        ;; NB patch misery is scaled, in measure-misery above, to the share of all voters on the patch, 
        ;; this share maxes at 0.00159 for the (0,0) patch in a unimodal (0,10) distribution and is 0.0002 at patch (20,0) in this distibution
        ;; this explains the scaling up of the patch misery score by 1000 and the units of beta are thus sui generis to the simulation
        ;; the greater beta, however, the more sensitive are voters on a patch to a given level of misery.
        ;; new-born parties intially locate at the position of the "sprouting" patch.
    
    [ create-parties 1 [set heading random-float 360 jump random-float 30 initialize-party] ]
        ;; non-endogenous initial party locations take a random walk within 30 from the origin
end

to initialize-party
  ifelse (count parties = 0) [set fitness 1] [set fitness 1 / count parties] 
  set heading random-float 360 set old-x xcor set old-y ycor set age 0 set size 1.5                                                               
  
  random-pick                      
  load-rule-parameters
  color-myself
  
  if (birth-death-file = true) [
    file-write votes1 file-write x-mean1 file-write votes2 file-write x-mean2 file-write fitness-alpha file-write survival-threshold
    file-write campaign-ticks file-write misery-alpha file-write misery-beta
    file-write valence-lambda 
    file-write election file-write "birth" file-write rule file-write who 
    file-write precision xcor 4 file-write precision ycor 4  file-print ""
  ]
     ;;Write out your starting data before handing control back to the observer
end

to random-pick
    set rule one-of rule-list
    ;; randomly pick a rule from the rule list
end

to load-rule-parameters                    ; set parameters of your decision rule by reading the relevant parts of your rule name
  set species first rule
  set valence 25 * (read-from-string item 1 rule)
  set size 1 + valence / 50
       ;; NB valence takes five evenly spaced values on the [100,500] interval; changing that 100 changes the valence range
       ;; useful to have parties with difference valences have different sizes on the interface
  set comfort-kappa (read-from-string substring rule 2 5) / 100
  set speed 1
  set neighborhood-eta 6
end

to color-myself
  if (species = "S") [set color yellow]
  if (species = "A") [set color lime]
  if (species = "H") [set color violet]
  if (species = "P") [set color red]
  if (species = "E") [set color blue]
end
 
to adapt
  if (species = "S") [stick]
  if (species = "A") [aggregate]
  if (species = "H") [sat-hunt]
  if (species = "P") [sat-predate]
  if (species = "E") [sat-explore]
end

;;____________________________
;
;;   MAIN CONTROL ROUTINE
;;____________________________

to go
  repeat campaign-ticks
  [
    set cycle cycle + 1
    ask parties [adapt]
    update-support
    if (remainder cycle campaign-ticks = 0 and cycle != 0) [calculate-election-results]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
499
8
1050
580
35
35
7.62
1
10
1
1
1
0
0
0
1
-35
35
-35
35
0
0
1
ticks

BUTTON
8
12
72
55
Setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
50
342
164
375
votes1
votes1
0
1000000
1000000
1000
1
NIL
HORIZONTAL

SLIDER
162
342
263
375
x-mean1
x-mean1
-40
40
15
1
1
NIL
HORIZONTAL

SLIDER
261
342
362
375
y-mean1
y-mean1
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
361
342
461
375
sd-1
sd-1
0
40
10
0.5
1
NIL
HORIZONTAL

SLIDER
50
373
163
406
votes2
votes2
0
1000000
1000000
1000
1
NIL
HORIZONTAL

SLIDER
162
373
262
406
x-mean2
x-mean2
-40
40
-15
1
1
NIL
HORIZONTAL

SLIDER
261
373
361
406
y-mean2
y-mean2
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
360
373
460
406
sd-2
sd-2
0
40
10
0.5
1
NIL
HORIZONTAL

SLIDER
50
404
163
437
votes3
votes3
0
1000000
0
1000
1
NIL
HORIZONTAL

SLIDER
162
404
262
437
x-mean3
x-mean3
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
261
404
361
437
y-mean3
y-mean3
-40
40
0
1
1
NIL
HORIZONTAL

SLIDER
360
404
460
437
sd-3
sd-3
0
40
1
0.5
1
NIL
HORIZONTAL

TEXTBOX
120
444
451
464
Population designer: Equilateral = (0 17)(-15 -9)(15 -9)
11
0.0
0

BUTTON
72
12
142
57
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
289
52
354
97
Eccentricity
mean-eccentricity
2
1
11

MONITOR
70
52
142
97
Cycle
cycle
0
1
11

MONITOR
226
52
291
97
Misery
voter-misery
2
1
11

SLIDER
184
125
320
158
fitness-alpha
fitness-alpha
0
.99
0.8
.01
1
NIL
HORIZONTAL

SLIDER
184
156
320
189
survival-threshold
survival-threshold
0
1.0
0.1
.01
1
NIL
HORIZONTAL

SLIDER
184
187
320
220
campaign-ticks
campaign-ticks
1
50
20
1
1
NIL
HORIZONTAL

MONITOR
163
52
228
97
N parties
n-parties
0
1
11

MONITOR
353
52
418
97
ENP
enp
2
1
11

TEXTBOX
224
29
365
49
Measures from last election
11
0.0
1

MONITOR
8
52
72
97
Campaign
election
1
1
11

SWITCH
27
222
162
255
birth-death-file
birth-death-file
1
1
-1000

SLIDER
27
125
162
158
misery-alpha
misery-alpha
0
1
0.5
.01
1
NIL
HORIZONTAL

SLIDER
27
158
162
191
misery-beta
misery-beta
0
1
1
.01
1
NIL
HORIZONTAL

SWITCH
27
191
162
224
endogenous-birth
endogenous-birth
0
1
-1000

TEXTBOX
217
110
289
129
Environment parameters
11
0.0
1

TEXTBOX
30
110
150
128
Party birth parameters
11
0.0
1

TEXTBOX
48
524
478
551
Stickers yellow: Aggregators green; Hunters violet; Predators red; Explorers blue
11
0.0
1

TEXTBOX
84
267
198
285
Valence
11
0.0
1

SLIDER
61
281
199
314
valence-lambda
valence-lambda
0
1
0.5
.01
1
NIL
HORIZONTAL

BUTTON
206
472
319
505
Random population
random-pop
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

INPUTBOX
338
125
458
185
bd-file-name
manual_bd.txt
1
0
String

TEXTBOX
347
189
497
217
Hit file-close after manual run\\nwith birth-death file
11
0.0
1

BUTTON
357
222
437
255
NIL
file-close
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

@#$#@#$#@
WHAT IS IT?
-----------
This implements the model of party competition with endogenous party birth and death, and with party leaders who have different levels of non-policy "valence", specified in Chapter 9 of Michael Laver and Ernest Sergenti's book, Party competition: an agent based model (Princeton University Press, 2012). A full description and analysis of the model, which is a modified version of the model specified in Chapter 7, can be found in Chapter 9 of this book and is not repeated here.

Party positions and voter ideal points are defined in two dimensions of policy/ideology, directly analogous to the dimensions used in other "spatial" models of political competition.  The horizontal dimension might, for example, be seen as describing left-right economic policy positions; the vertical dimension might be a liberal-conservative policy dimension on matters such as abortion, sexuality, euthanasia. 

VOTERS no longer always vote for the closest party, and are also concerned the the non- policy "valence" (charisma, competance) of the party leaders. The distribution of voters' ideal points  can be designed as if this is an aggregate of up to three subpopulations (though only two are investigated by Laver and Sergenti and the default setting on the interface sets the size of the third subpopulation at zero). Voters in each subpopulation have normally distributed ideal points, and each subpopulation is characterized by: the number of voters it comprises; the standard deviation of the distribution of its voters' ideal points, and the means of this distribution on the x and y dimensions. All of these parameters can be set using the sliders in the _Population Designer_ panel near the bottom of the interface. Alternatively, the _random population_ button picks these at random.

A voter evaluates each party by trading off the closeness of party positions to her ideal point, against the non-policy "valence" of the party leader. The trade-off parameter is "valence-lambda", set on the interface and common for all voters. Voter utility is defined in the "update-support" procedure as: 
valence-lambda * valence - (1 - valence-lambda) * ((distance myself) ^ 2).
Voters support the party which maximizes their utility, thus defined.

PARTY LEADERS each have a level of non-policy valence, which they cannot change, and compete with each other by offering policies to potential supporters. They use one of five parameterized _species_ of decision rule to select a party policy position. The species are: Sticker, Aggregator, Hunter, Predator and Exporer. The rule parameters of speed of adaptation (speed) and exploration neighborhood (eta) are now set to central values, while comfort threshold (kappa) and valence vary between leaders. These rules and parameters are described fully in Laver and Sergenti (2012) and implemented in the _party dynamics_ section of the code.

DYNAMICS OF PARTY COMPETITION. The baseline dynamics of the model iterate forever.  (1) Voters support their closest party.  (2) Given profile of voter support for parties, leaders adapt party policy positions using their decision rule. (3) Go to 1.

The set of surviving political parties is fully endogenous to the model. 

EXISTING PARTIES DIE if their updated fitness, denominated in vote share, falls below a system survival threshold. The party survival threshold, and the memory parameter in the fitness updating regime, can be set using the sliders in the _Environment_ panel near the top of the interface.  

NEW PARTIES ARE BORN at the ideal points of disgruntled voters. Parameters of the party birth regime can be set using the sliders in the _Party birth_ panel near the top of the interface.  There is a switch on the interface that turns off endogenous party birth and instead randomly generates new party births at random locations. (This was not investigated systematically by Laver and Sergenti.)

Model ticks are divided into CAMPAIGN TICKS and ELECTION TICKS. Party leaders adapt their positions during campaign ticks but receive no rewards or punishments. Parties can only die or be born on election ticks. The number of campaign ticks per election tick can be set using the slider in the _Environment_ panel near the top of the interface.  

HOW TO USE IT
-------------
The set of available decision rules, and the set of available parameterizations of these rules, are hard coded in the "set-rule-list" procedure in the setup routine. (This is obviously less than elegant and it would be better to load this list from a file). Comments below this procedure explain the coding of a given rule parameterization, which is equivalent to a party leader's "decision-making DNA". Some parameters are redundant for some rules. The research design of the Laver-Sergenti experiments requires that all rule SPECIES have an equal chance of random selection, and "clones" of Sticker and Aggregator rules, all with the same parameterization, are included to enable this. The current, easily editable, rule list is the one used for the experiment reported in Chapter 9 of Laver and Sergenti.

SETUP sets up parties, supporters and system parameters as specified above.  GO starts and stops the simulation using current parameters.
(Hitting SETUP while GO is still pressed very occasionally causes an error depending where precisely the program is when setup is hit; this easily retrieved by unpressing GO and pressing SETUP again.)

RUNNING EXPERIMENTS.  Laver and Sergenti designed a large computational experiment, and report results of this, in Chapter 9 of their book. Although the _production_ run was executed on a high performance cluster, precisely equivalent smaller scale experiments can easily be run using Behavior Space. Sketch runs for all results reported in Laver and Sergenti  were generated using Behavior Space on a normal laptop.

DATA OUTPUT. Standard data output is via Behavior Space experiments used in the normal way. There is a separate data channel that writes out information on party births and deaths only when these occur.  This is activated by a switch and a file name on the interface.


WHAT TO PLAY WITH
-----------
The most obvious new source of potential experimentation concerns party leader valences. These can easily be changed by editing the rule-list in the setup procedure. Any number of parametizations can enter this list, which can be as large and varied as you like. The larger the rule list, however, the longer it will take to "map out" the steady state distribution of model outputs. The less systematically it is constructed, the harder it will be to make sense of model outputs.

The key substantive question to explore, as discussed in Laver and Sergenti Chapter 9, evolutionary settings in which LOW valence parties can survive.

The rule parameters of speed of adaptation (speed) and exploration neighborhood (eta) might be varied, while another interesting possibility is to make valence-lambda a voter variable, drawing this from some distribution when the simulation is set up.

Laver and Sergenti report results from a carefully controlled computational experiment and only investigate electorate with two subpopulations. There are infinitely many alternative populations for you to explore using the population designer. There are also many parameterizations of the competitive environment, and the party birth regime, not explored by Laver and Sergenti. You may, for example, want to specify a parameterization of the model you feel corresponds to some real political system that interests you.

By far the most exciting and callenging way forward is to specify and program your own decision rule for party leaders.  Just drop in your coded new rule as a procedure in the party dynamics section, add its name to the rule list, edit it in to the _adapt_ and _color-myself_ procedures, and add a reporter for your rule_s vote share to the interface. You_re good to go!

CREDITS AND REFERENCES
----------------------
Programmed by:
 
Michael Laver, Department of Politics, New York University
ml127@nyu.edu

Ernest Sergenti, The World Bank
esergenti@gmail.com
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="valence_test" repetitions="1" runMetricsEveryStep="true">
    <setup>  random-seed 79103
  set votes1 647096 
  set x-mean1 -7.564
  set y-mean1 0
  set sd-1 5
  set votes2 352904
  set x-mean2 7.564
  set y-mean2 0
  set sd-2 5
  set votes3 0
  set x-mean3 0
  set y-mean3 0
  set sd-3 1.0
  set endogenous-birth true
  set misery-alpha 0.5
  set misery-beta 1
  set birth-death-file false
  set valence-lambda 0.3839
  set fitness-alpha 0.4277
  set survival-threshold 0.2607
  set campaign-ticks 12
  setup</setup>
    <go>go</go>
    <final>file-close</final>
    <timeLimit steps="251"/>
    <metric>n-parties</metric>
    <metric>voter-misery</metric>
    <metric>mean-eccentricity</metric>
    <metric>enp</metric>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
