<?xml version="1.0" encoding="utf-8"?>
<model version="NetLogo 7.0.3" snapToGrid="true">
  <code><![CDATA[; FantasticLogo: Predator-Prey Dynamics on Planet Magy
;
; Version: 1.0.2
; Author: Daniel Vartanian
; Maintainer: Daniel Vartanian <https://github.com/danielvartan>
; License: GPLv3 <https://www.gnu.org/licenses/gpl-3.0>
; Repository: <https://github.com/sustentarea/fantasticlogo>
;
; Require: NetLogo >= 7.0.3

globals [max-oopalamca]

breed [bleurghs bleurgh]
breed [oopalamcas oopalamca]
breed [plasticocos plasticoco]

bleurghs-own [energy]
oopalamcas-own [energy]

to setup
  clear-all

  set max-oopalamca 10000

  ask patches [
    let #color (sky + 4)

    set pcolor one-of (list
      (#color + (random-float 0.9))
      white
    )
  ]

  create-plasticocos 30 [
    set shape  "plasticoco"
    set size 2
    setxy random-xcor random-ycor
  ]

  create-oopalamcas oopalamca-n [
    set shape  "oopalamca"
    set size 2
    setxy random-xcor random-ycor

    set energy random random-between
      (0.25 * bleurgh-gain-from-food)
      (1.25 * bleurgh-gain-from-food)

    while [any? other turtles-here] [
      setxy random-xcor random-ycor
    ]
  ]

  create-bleurghs bleurgh-n [
    set shape "bleurgh"
    set size 3
    setxy random-xcor random-ycor

    set energy random-between
      (0.25 * bleurgh-gain-from-food)
      (1.25 * bleurgh-gain-from-food)

    while [any? other turtles-here] [
      setxy random-xcor random-ycor
    ]
  ]

  reset-ticks
end

to go
  if ((not any? oopalamcas) and (not any? bleurghs)) [
    stop
  ]

  if (count oopalamcas > max-oopalamca) [
    user-message "The oopalamca have inherited the planet!"

    stop
  ]

  ask oopalamcas [
    move

    ifelse (random 100 < 50) [
      set energy energy - 1
    ] [
      set energy energy + 1
    ]

    if (energy < 0) [die]

    reproduce-oopalamcas
  ]

  ask bleurghs [
    eat-oopalamca
    set energy energy - 1

    if (energy < 0) [die]

    if (energy > (5 * bleurgh-gain-from-food)) [
      reproduce-bleurghs
    ]
  ]

  tick
end

to move
  rt random 50
  lt random 50
  fd 1
end

to reproduce-oopalamcas
  if (random-float 100 < oopalamca-reproduce) [
    set energy (energy / 2)

    hatch 1 [
      right random-float 360
      forward 1
    ]
  ]
end

to eat-oopalamca
  let prey one-of oopalamcas-here ; (oopalamcas in-radius 1)

  if prey != nobody  [
    ask prey [die]
    set energy energy + bleurgh-gain-from-food
  ]
end

to reproduce-bleurghs
  if (random-float 100 < bleurgh-reproduce) [
    set energy (energy / 2)

    hatch 1 [
      set energy 1
      setxy random-xcor random-ycor

      while [any? other turtles-here] [
        setxy random-xcor random-ycor
      ]
    ]
  ]
end

to-report random-between [#x #y]
  report (random (#y - #x) + #x)
end
]]></code>
  <widgets>
    <view x="270" wrappingAllowedX="true" y="10" frameRate="30.0" minPycor="-25" height="561" showTickCounter="true" patchSize="11.0" fontSize="14" wrappingAllowedY="true" width="561" tickCounterLabel="ticks" maxPycor="25" updateMode="1" maxPxcor="25" minPxcor="-25"></view>
    <button x="10" actionKey="S" y="10" height="40" disableUntilTicks="false" forever="false" kind="Observer" width="125" display="Setup">setup</button>
    <slider x="10" step="1" y="275" max="100" width="250" display="bleurgh-reproduce" height="50" min="0.0" direction="Horizontal" default="30.0" variable="bleurgh-reproduce" units="%"></slider>
    <slider x="10" step="1.0" y="220" max="100.0" width="250" display="bleurgh-gain-from-food" height="50" min="0.0" direction="Horizontal" default="50.0" variable="bleurgh-gain-from-food"></slider>
    <slider x="10" step="1" y="165" max="100" width="250" display="oopalamca-reproduce" height="50" min="0.0" direction="Horizontal" default="15.0" variable="oopalamca-reproduce" units="%"></slider>
    <slider x="10" step="1" y="110" max="100" width="250" display="bleurgh-n" height="50" min="0" direction="Horizontal" default="50.0" variable="bleurgh-n"></slider>
    <slider x="10" step="1" y="55" max="100" width="250" display="oopalamca-n" height="50" min="0" direction="Horizontal" default="50.0" variable="oopalamca-n"></slider>
    <button x="140" actionKey="G" y="10" height="40" disableUntilTicks="true" forever="true" kind="Observer" width="120" display="Go">go</button>
    <plot x="840" autoPlotX="true" yMax="100.0" autoPlotY="true" yAxis="Frequency" y="10" xMin="0.0" height="235" legend="true" xMax="100.0" yMin="0.0" width="375" xAxis="Steps" display="Populations">
      <setup></setup>
      <update></update>
      <pen interval="1.0" mode="0" display="Oopalamcas" color="-5825686" legend="true">
        <setup></setup>
        <update>plot count oopalamcas</update>
      </pen>
      <pen interval="1.0" mode="0" display="Bleurghs" color="-4079321" legend="true">
        <setup></setup>
        <update>plot count bleurghs</update>
      </pen>
    </plot>
    <monitor x="1030" precision="3" y="250" height="60" fontSize="11" width="185" display="Bleurghs">count bleurghs</monitor>
    <monitor x="840" precision="3" y="250" height="60" fontSize="11" width="185" display="Oopalamcas">count oopalamcas</monitor>
  </widgets>
  <info><![CDATA[## OVERVIEW

`FantasticLogo` is a NetLogo model for simulating the complex [predator-prey dynamics](https://en.wikipedia.org/wiki/Lotka%E2%80%93Volterra_equations) of bleurghs and oopalamcas, two species native to planet Magy's northern hemisphere. It provides a platform for xenobiology research, allowing users to explore how alien species interact, compete, and adapt within an unfamiliar ecosystem.

![NetLogo Web Badge](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/netlogo-web-badge.png)

Click [here](https://danielvartan.github.io/fantasticlogo/) to play with this model online on [NetLogo Web](https://www.netlogoweb.org)!

## HOW IT WORKS

The model operates on a grid of patches, each representing a patch of soil where bleurghs grow and oopalamcas roam.

Oopalamcas are small, purple creatures with voracious appetites and a distinctive croak. Bleurghs are large, carnivorous plants that move slowly and capture prey with their funnel-shaped mouths. In the model, oopalamcas are the prey and bleurghs are the predators.

As the simulation runs, bleurghs grow on patches and, upon reaching maturity, can reproduce by dispersing seeds to random locations. Bleurghs prey on oopalamcas, regulating their population.

> An oopalamca (prey) and a plasticoco (a harmless plant).

![Oopalamca](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/oopalamca.png) ![Plasticoco](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/plasticoco.png)

> An infant and an adult bleurgh (predator).

![Oopalamca](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/infant-bleurgh.png) ![Bleurgh](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/bleurgh.png)

## HOW TO USE IT

Use the sliders on the left to adjust the initial populations and reproduction rates for bleurghs and oopalamcas, as well as other key parameters. When ready, click `Setup` to initialize the simulation, then click `Go` to begin observing the dynamics.

### INTERFACE CONTROLS

#### CHOOSERS, INPUT BOXES, SLIDERS, AND SWITCHES

- **`oopalamca-n`**: Slider for setting the initial number of oopalamcas (default: `50`).
- **`bleurgh-n`**: Slider for setting the initial number of bleurghs (default: `50`).
- **`oopalamca-reproduce`**: Slider for setting the probability (in percentage) of oopalamcas reproducing each tick (default: `15%`).
- **`bleurgh-gain-from-food`**: Slider for setting the energy bleurghs gain from consuming an oopalamca (default: `50`).
- **`bleurgh-reproduce`**: Slider for setting the probability (in percentage) of bleurghs reproducing each tick (default: `30%`).

#### BUTTONS

- **`Setup`**: Initializes the simulation with the selected parameters.
- **`Go`**: Starts or resumes the simulation.

### MONITORS AND PLOTS

- **`Populations`**: Plot displaying the populations of oopalamcas and bleurghs over time.
- **`Oopalamcas`**: Monitor showing the current number of oopalamcas.
- **`Bleurghs`**: Monitor showing the current number of bleurghs.

## THINGS TO NOTICE

- Observe the cyclical nature of predator-prey relationships, where increases in the prey population lead to increases in the predator population, followed by declines in both populations.
- Examine how varying reproduction rates and energy gains affect the stability and fluctuations of both species.
- Analyze the conditions under which either species may face extinction or overpopulation.

## THINGS TO TRY

- Explore the impact of initial population sizes on long-term stability.
- Experiment with different reproduction rates and energy gains to observe their effects on population dynamics.
- Investigate the conditions leading to extinction or overpopulation of either species.

## JOKES ASIDE...

`FantasticLogo` takes inspiration from René Laloux's 1973 masterpiece, [La Planète Sauvage](https://www.imdb.com/title/tt0070544/). The model's visual and conceptual design mirrors the film's hypnotic atmosphere, where the towering, blue-skinned Draags dominate a world teeming with surreal, alien flora and fauna. In this world, humans (known as Oms) are treated as mere pets, struggling to survive in an environment that is both beautiful and perilous.

The model's dynamics are grounded in the classic [Lotka-Volterra equations](https://danielvartan.github.io/lotka-volterra/), originally formulated by Alfred J. Lotka ([1925](http://archive.org/details/elementsofphysic017171mbp)) and Vito Volterra ([1926](https://www.nature.com/articles/118558a0)), to describe predator-prey interactions. The code builds on Wilensky's ([1997](http://ccl.northwestern.edu/netlogo/models/WolfSheepPredation)) Wolf Sheep Predation model.

[(![La Planète Sauvage](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/la-planete-sauvage-cover-netlogo.png)](https://www.imdb.com/title/tt0070544/)

## HOW TO CITE

![DOI Badge](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/doi-badge.png)

If you use this model, please cite it to acknowledge the effort invested in its development and maintenance.

To cite `FantasticLogo` please use the following format:

Vartanian, D. *FantasticLogo: Predator-prey dynamics on planet Magy* [Computer software]. [https://doi.org/10.5281/zenodo.18122347](https://doi.org/10.5281/zenodo.18122347)

A BibLaTeX entry for LaTeX users is:

```latex
@software{vartanian,
  title = {FantasticLogo: Predator-prey dynamics on planet Magy},
  author = {Daniel Vartanian},
  doi = {https://doi.org/10.5281/zenodo.18122347},
  note = {Computer software}
}
```

## HOW TO CONTRIBUTE

![Contributor Covenant 3.0 badge](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/contributor-covenant-3-0-badge.png)

Contributions are welcome! Whether you want to report bugs, suggest features, or improve the code or documentation, your input is highly valued. Please check the code repository [issues tab](https://github.com/sustentarea/fantasticlogo/issues) for existing issues or to open a new one.

## LICENSE

![GPLv3 License Badge](https://raw.githubusercontent.com/danielvartan/fantasticlogo/refs/heads/main/images/gplv3-license-badge.png)

```text
Copyright (C) 2025 Daniel Vartanian

FantasticLogo is free software: you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation, either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program. If not, see <https://www.gnu.org/licenses/>.
```

## REFERENCES

Lotka, A. J. (1925). *Elements of physical biology*. Williams & Wilkins Company. [http://archive.org/details/elementsofphysic017171mbp](http://archive.org/details/elementsofphysic017171mbp)

Volterra, V. (1926). Fluctuations in the abundance of a species considered mathematically. *Nature*, *118*(2972), 558-560. [https://doi.org/10.1038/118558a0](https://doi.org/10.1038/118558a0)

Wilensky, U. (1997). *NetLogo wolf sheep predation model* [Computer software]. Center for Connected Learning and Computer Based Modeling at Northwestern University. [http://ccl.northwestern.edu/netlogo/models/WolfSheepPredation](http://ccl.northwestern.edu/netlogo/models/WolfSheepPredation)
]]></info>
  <turtleShapes>
    <shape name="default" rotatable="true" editableColorIndex="0">
      <polygon color="-1920102913" filled="true" marked="true">
        <point x="150" y="5"></point>
        <point x="40" y="250"></point>
        <point x="150" y="205"></point>
        <point x="260" y="250"></point>
      </polygon>
    </shape>
    <shape name="bleurgh" rotatable="false" editableColorIndex="0">
      <polygon color="496990463" filled="true" marked="false">
        <point x="135" y="120"></point>
        <point x="165" y="165"></point>
        <point x="180" y="210"></point>
        <point x="180" y="240"></point>
        <point x="150" y="300"></point>
        <point x="165" y="300"></point>
        <point x="195" y="240"></point>
        <point x="195" y="195"></point>
        <point x="165" y="135"></point>
      </polygon>
      <circle x="65" y="20" marked="false" color="-303222273" diameter="170" filled="true"></circle>
      <circle x="59" y="14" marked="false" color="-1491375361" diameter="182" filled="false"></circle>
    </shape>
    <shape name="infant-bleurgh" rotatable="false" editableColorIndex="0">
      <polygon color="496990463" filled="true" marked="false">
        <point x="135" y="120"></point>
        <point x="165" y="165"></point>
        <point x="180" y="210"></point>
        <point x="180" y="240"></point>
        <point x="150" y="300"></point>
        <point x="165" y="300"></point>
        <point x="195" y="240"></point>
        <point x="195" y="195"></point>
        <point x="165" y="135"></point>
      </polygon>
      <polygon color="-303222273" filled="true" marked="false">
        <point x="180" y="150"></point>
        <point x="180" y="120"></point>
        <point x="165" y="97"></point>
        <point x="135" y="84"></point>
        <point x="128" y="121"></point>
        <point x="147" y="148"></point>
        <point x="165" y="165"></point>
      </polygon>
      <polygon color="-1491375361" filled="true" marked="false">
        <point x="170" y="155"></point>
        <point x="131" y="163"></point>
        <point x="175" y="167"></point>
        <point x="196" y="136"></point>
      </polygon>
      <polygon color="-1491375361" filled="false" marked="false">
        <point x="150" y="150"></point>
        <point x="120" y="120"></point>
        <point x="135" y="75"></point>
        <point x="165" y="90"></point>
        <point x="180" y="120"></point>
        <point x="180" y="150"></point>
        <point x="165" y="165"></point>
      </polygon>
    </shape>
    <shape name="oopalamca" rotatable="false" editableColorIndex="0">
      <rectangle endX="180" startY="225" marked="false" color="-528509185" endY="255" startX="165" filled="true"></rectangle>
      <rectangle endX="180" startY="225" marked="false" color="-1491375361" endY="255" startX="165" filled="false"></rectangle>
      <rectangle endX="135" startY="225" marked="false" color="-528509185" endY="270" startX="120" filled="true"></rectangle>
      <rectangle endX="135" startY="225" marked="false" color="-1491375361" endY="270" startX="120" filled="false"></rectangle>
      <circle x="63" y="63" marked="false" color="-528509185" diameter="175" filled="true"></circle>
      <circle x="150" y="105" marked="false" color="-1" diameter="60" filled="true"></circle>
      <circle x="165" y="120" marked="false" color="255" diameter="30" filled="true"></circle>
      <circle x="60" y="60" marked="false" color="-1491375361" diameter="180" filled="false"></circle>
    </shape>
    <shape name="plasticoco" rotatable="false" editableColorIndex="0">
      <rectangle endX="165" startY="165" marked="false" color="496990463" endY="300" startX="135" filled="true"></rectangle>
      <polygon color="496990463" filled="true" marked="false">
        <point x="135" y="255"></point>
        <point x="90" y="210"></point>
        <point x="45" y="195"></point>
        <point x="75" y="255"></point>
        <point x="135" y="285"></point>
      </polygon>
      <polygon color="496990463" filled="true" marked="false">
        <point x="165" y="255"></point>
        <point x="210" y="210"></point>
        <point x="255" y="195"></point>
        <point x="225" y="255"></point>
        <point x="165" y="285"></point>
      </polygon>
      <polygon color="496990463" filled="true" marked="false">
        <point x="135" y="180"></point>
        <point x="90" y="135"></point>
        <point x="45" y="120"></point>
        <point x="75" y="180"></point>
        <point x="135" y="210"></point>
      </polygon>
      <polygon color="496990463" filled="true" marked="false">
        <point x="165" y="180"></point>
        <point x="165" y="210"></point>
        <point x="225" y="180"></point>
        <point x="255" y="120"></point>
        <point x="210" y="135"></point>
      </polygon>
      <polygon color="751909887" filled="true" marked="false">
        <point x="135" y="165"></point>
        <point x="120" y="120"></point>
        <point x="150" y="90"></point>
        <point x="180" y="120"></point>
        <point x="165" y="165"></point>
      </polygon>
    </shape>
  </turtleShapes>
  <linkShapes>
    <shape name="default" curviness="0.0">
      <lines>
        <line x="-0.2" visible="false">
          <dash value="0.0"></dash>
          <dash value="1.0"></dash>
        </line>
        <line x="0.0" visible="true">
          <dash value="1.0"></dash>
          <dash value="0.0"></dash>
        </line>
        <line x="0.2" visible="false">
          <dash value="0.0"></dash>
          <dash value="1.0"></dash>
        </line>
      </lines>
      <indicator>
        <shape name="link direction" rotatable="true" editableColorIndex="0">
          <line endX="90" startY="150" marked="true" color="-1920102913" endY="180" startX="150"></line>
          <line endX="210" startY="150" marked="true" color="-1920102913" endY="180" startX="150"></line>
        </shape>
      </indicator>
    </shape>
  </linkShapes>
  <previewCommands>setup repeat 75 [ go ]</previewCommands>
  <resources>
    <resource name="la-planete-sauvage-cover-netlogo" extension="png">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                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             