globals [
  sample-car     ;focus on a car
  special-convoy ; sepcial convoy
  car-ahead     ; one of the cars on the patches ahead each car (depending if the car is closed to the world boundary, since we are wrapping to the lane below)
  cars-ahead    ; all the cars on the "front-assist-range" patches ahead
]

turtles-own [
  speed
  speed-limit
  speed-min

]

to setup
  clear-all
  ask patches [ setup-road ]
  setup-cars
  watch sample-car
  reset-ticks
end

to setup-road ;; patch procedure
  ifelse (pycor mod 4 < 2) [ set pcolor white ] [ set pcolor green - random-float 0.5]
end

to setup-cars
  if number-of-cars > world-width * 12 [
    user-message (word
      "There are too many cars for the amount of road. "
      "Please decrease the NUMBER-OF-CARS slider to below "
      (world-width + 1) " and press the SETUP button again. "
      "The setup has stopped.")
    stop
  ]

  ;set the special-convoy
  create-turtles 1 [
    set special-convoy one-of turtles
    set color red
    set shape "convey"
    set size 3
    set xcor 0
    set ycor 0
    set heading 90
    ;; set initial speed to be speed-limit-convoy
    set speed speed-limit-convoy
  ]
  set-default-shape turtles "car"
  ;creates the sample car to focus on, just ahead the special-convoy (or at least among the randomly created turtles just ahead the special convoy)
  ;(equivalent of placing it as last car behind the special convoy and the <number-of-cars> cars
  create-turtles 1 [

    set color orange
    set sample-car one-of turtles with [color = orange]
    set xcor 4
    set ycor 0
    set heading 90
    ;; set initial speed to be in range 0.7 to 1.0
    set speed 0.7 + random-float 0.3
    set speed-limit 1
    set speed-min 0
  ]
  ;creates the "manual" cars
  create-turtles (1 - front-assist-cars-percentage / 100) * number-of-cars [
    set color blue
    set xcor random-xcor
    set ycor (random  13) * 4 - 24
    set heading 90
    ;; set initial speed to be in range 0.7 to 1.0
    set speed 0.7 + random-float 0.3
    set speed-limit 1
    set speed-min 0

   ]
  ;creates the cars with front assist option or autonomous cars
  create-turtles front-assist-cars-percentage * number-of-cars / 100 [
    set color green
    set xcor random-xcor
    set ycor (random  13) * 4 - 24
    set heading 90
    ;; set initial speed to be in range 0.7 to 1.0
    set speed 0.7 + random-float 0.3
    set speed-limit 1
    set speed-min 0
    separate-cars
  ]

end

; this procedure is needed so when we click "Setup" we
; don't end up with any two cars on the same patch
to separate-cars ;; turtle procedure
  if any? other turtles-here [
    fd 1
    separate-cars
  ]
end

to go
  ;; if there is a car ahead of you, adapt your speed (in a way depending on wether you are a car with "front assist" or a "manual" car:
  ask turtles [
    detect-car-ahead
    ifelse car-ahead != nobody
                     [ slow-down-car ]
                     [ speed-up-car ] ;; otherwise, speed up
               ;; don't slow down below speed minimum
   if speed < speed-min [ set speed speed-min ]
               ;; or don't speed up beyond speed limit
   ifelse color != red [if speed > speed-limit [ set speed speed-limit ]]
                ;special cas for special convoy that has a lower speed limit:
                       [if speed > speed-limit-convoy [ set speed speed-limit-convoy]]   ; The convoy has a lower speed limit

   ifelse xcor + speed > 25.5 [ifelse ycor < -20 [set ycor 24
                              set xcor xcor + speed ]
                              [set ycor ycor - 4
                               set xcor xcor + speed ]]
    [fd speed]
      ]


  tick
end

to detect-car-ahead
  ; if you are a car with "front-assist", you detect precisely the speed of the car ahead in a distance of "front-assist-range" :
  ifelse color = green or color = orange
            [;as we wrap from right to left with the lane below, and from bottom-right to top-left, we must look into the patches ahead :
              ifelse xcor + front-assist-range > 25 [ifelse ycor >= -20 [set cars-ahead other turtles-on patches with
                                                                                     [(pycor = ([ycor] of myself - 4 ) and pxcor >= -25 and pxcor < ([xcor] of myself - 51 + front-assist-range))
                                                                                           or (pycor = ( [ycor] of myself ) and pxcor >= [xcor] of myself and pxcor <= 25 )]]
                                                                          [set cars-ahead other turtles-on patches with
                                                                                     [(pycor = 24 and pxcor >= -25 and pxcor < ([xcor] of myself - 51 + front-assist-range))
                                                                                           or (pycor = ([ycor] of myself ) and pxcor > [xcor] of myself and pxcor <= 25 )]]]
                                 [set cars-ahead other turtles in-cone (front-assist-range) 180 with [ y-distance <= 1 ]]
              set car-ahead min-one-of cars-ahead [ x-distance ]]
    ; if you are a "manual" car , you detect the car ahead in the patch in front of you:
           [;as we wrap from right to left with the lane below, and from bottom-right to top-left:
            ifelse xcor > 24.5 [ifelse ycor >= -20 [set car-ahead one-of turtles-on patch -25 (ycor - 4)]
                                                   [set car-ahead one-of turtles-on patch -25 24] ]
                               [set car-ahead one-of turtles-on patch-ahead 1]]

end
to slow-down-car ;[ car-ahead ] ;; turtle procedure
  ifelse color = green or color = orange
                  [; if you are a car with front-assist : progressive slow down while getting closer to the car ahead of you and then adapt to the same speed
                   ifelse speed > [ speed ] of car-ahead
                             [ ifelse [ x-distance ] of car-ahead <= safety-distance [set speed [speed] of car-ahead - front-assist-deceleration ]
                                                                                     [set speed ((speed - [speed] of car-ahead) / (front-assist-range -  safety-distance)  * ([ x-distance ] of car-ahead - safety-distance) * deceleration-coef  + [speed] of car-ahead ) ]]
                                                                                    ;[set speed speed - front-assist-deceleration]]
                             [speed-up-car]]
                  [;; if you are a "manual" car, slow down so you are driving more slowly than the car ahead of you
                   set speed [ speed ] of car-ahead - deceleration]
end


to speed-up-car ;; turtle procedure
  ifelse color = green or color = orange [set speed speed + front-assist-acceleration] [set speed speed + acceleration]
end

to-report x-distance
  report distancexy [ xcor ] of myself ycor
end

to-report y-distance
  report distancexy xcor [ ycor ] of myself
end
; Copyright 1997 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
790
10
1388
609
-1
-1
11.57
1
10
1
1
1
0
1
0
1
-25
25
-25
25
1
1
1
ticks
30.0

BUTTON
15
90
87
131
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
95
90
166
130
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
15
10
219
43
number-of-cars
number-of-cars
1
100
80.0
1
1
NIL
HORIZONTAL

SLIDER
155
205
330
238
deceleration
deceleration
0.02
.099
0.03
.001
1
NIL
HORIZONTAL

SLIDER
155
245
330
278
acceleration
acceleration
0
.0080
0.0073
.0001
1
NIL
HORIZONTAL

PLOT
5
410
755
607
Car speeds
time
speed
0.0
300.0
0.0
1.1
true
true
"" ""
PENS
"orange car" 1.0 0 -955883 true "" "plot [speed] of sample-car"
"min speed" 1.0 0 -204336 true "" "plot min [speed] of turtles"
"max speed" 1.0 0 -5516827 true "" "plot max [speed] of turtles"

SLIDER
155
145
327
178
speed-limit-convoy
speed-limit-convoy
0
1
0.3
0.1
1
NIL
HORIZONTAL

SLIDER
15
50
222
83
front-assist-cars-percentage
front-assist-cars-percentage
0
100
90.0
1
1
NIL
HORIZONTAL

SLIDER
155
305
330
338
front-assist-deceleration
front-assist-deceleration
0
0.05
0.01
0.001
1
NIL
HORIZONTAL

SLIDER
155
345
330
378
front-assist-acceleration
front-assist-acceleration
0.005
0.0099
0.0099
0.0001
1
NIL
HORIZONTAL

SLIDER
355
305
527
338
front-assist-range
front-assist-range
safety-distance
20
20.0
1
1
NIL
HORIZONTAL

SLIDER
355
345
527
378
safety-distance
safety-distance
1
10
2.0
1
1
NIL
HORIZONTAL

TEXTBOX
40
310
115
365
Cars with \nfront-assist\n parameters
11
0.0
1

TEXTBOX
40
220
190
246
\"Manual\" cars \nparameters
11
0.0
1

TEXTBOX
40
150
120
180
Special-convoy \nparamters
11
0.0
1

SLIDER
550
305
722
338
deceleration-coef
deceleration-coef
1
10
5.0
0.5
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

**Every Day at 3pm**, on the French national highway N171, an **exceptional convoy** starts **from Nantes, towards Saint Nazaire**, situated 50 km ahead. This convoy carries a preassembled A320 floor to be integrated in a pre-final assembly line at Airbus Saint Nazaire factory. This convoy takes the total width of the road, and **can only drive at 50 km/h**, where the normal speed limit of the N171 is 110 km/h.
And every day at 3pm, the traffic is like an accordion up to Saint Nazaire: the cars behind the exceptional convoy brake to full stop, and then start again up to 110 km/h, until they are obliged to stop again, causing every time potential car crashes.
Sometimes, the traffic ends up stabilizing at steady speed of 50 km/h before arriving to Saint Nazaire.


This model models the movement of cars on this N171, and tries to see what are the parameters that improve or worsens this traffic problem, and tries to answer the question: Would **self-driving cars** (or at least cars with **front-assist option**) avoid this traffic problem ? and if yes, with what minimum proportion of self driving cars among the "manual cars" ?

Each **"Manual" car is blue**  and follows a simple set of rules.

Each **car with "Front-assist" is green** and follows a set of more elaborated rules.


The model demonstrates how traffic jams can form or disappear, and tries to understand the influence of the option "front-assist" in modern cars .  

## HOW TO USE IT

Use the Sliders to set total number of cars and the proportion of "front-assist" cars.
Click on the SETUP button to set up the map and cars.


Click on GO to start the cars moving.  Note that they wrap around the world as they move, so the road is like a continuous loop, going one lane down at each loop and wrapping from bottom - right corner to up-left corner

### For a "Manual" car: 
The **acceleration** and **decelaration SLIDERS** control the way it slows down (decelerates) when it sees a car in the patch ahead, and the way speeds up (accelerates) when it doesn't see a car in the patch ahead. 

### A "Front-assist" car: 
* detects the first car ahead in a range adjusable with the **SLIDER "front-assist-range"**
* then slows down progressively until it reaches the  "safety-distance" adjustable with the **SLIDER "safety-distance"**. During this slowing down , the speed is a linear function of difference between the speed and the speed of the car ahead, ponderated by the distance between the car and the car ahead minus safety-distance. This slow-down can be moderated by a coefficient with the **SLIDER "decelration coef" **(increasing the coefficient make the cars speed decrease slower)
* if distance is below safety-distance, then the car slows down below the speed of the car ahead, using a deceleration factor adjustable with the **SLIDER "front-assist-deceleration"**

## THINGS TO NOTICE

### Initialization

The **watch highlight** is set on the "orange" car, wich is a "Front-assist" cars, so that its speed isn't sploiled by the "bad" behaviour that could have a "Manual" cars. 
Its speed must only show the state of the traffic at the **back end of a flow of cars behind the special convoy**. Thus, **this orange car** is set to be among the first cars ahead of the special convoy, which is equivalent to **being the last car** of the flow of cars behind the special convoy, as the world wraps up on itself. 

**The randomness** comes with the random position of the cars at initialization, and with their random initial speed. However, I considered that the speed of the cars on a high way with no obstacles is usually quite steady and closed to the speed limit. So the initial speed is set randomly between 0.7 and 1 (the speed limit), instead of being set between 0 and 1. 

"Even though all of the cars are moving forward, the traffic jams tend to move backwards. This behavior is common in wave phenomena: the behavior of the group is often very different from the behavior of the individuals that make up the group." (See Trafic Basic)

As opposed to Traffic Basic, I **"expanded" the world** here, drawing 13 lanes wrapping one on another, to have more distance between the special-convoy and the last car, and see long distance effects;


### Plot 

The plot shows three values as the model runs:

* the **fastest speed** of any car (this doesn't exceed the speed limit!)

* the **slowest speed** of any car

* the speed of **the last car** (or almost) : the orange car

Notice not only the maximum and minimum, but also the variability -- the "jerkiness" of one vehicle.

The goal is to find conditions where the "jerkiness" of the last car's speed is dampened after a while and every cars are driving at the same speed: the speed of the special convey. 

Even though both ACCELERATION and DECELERATION are very small, the cars can achieve high speeds as these values are added or subtracted at each tick and the jerkiness of the last car is ver difficult to avoid.

## THINGS TO TRY

First set the proportion of "front-assist" cars to 100% and find the best parameters of these cars to have a constant flow as fast as possible. (sliders of the group "front-assist" cars)

Then try the opposite: set the proportion of "front-assist" cars to 0 % and see if it is possible to get a regular flow. 

Now can try to set the proportion to 95%.


## EXTENDING THE MODEL

The fact is here that the "manual" cars are set to be like "really bad" drivers. one could try to improve their "anticipation" range... 

I think that the benefits of the anticipation that "front-assist" cars can have in real life isn't efficiently reproduced here , probably because the slowing down rule needs to be enhanced ! 

And the **"dampening" effect** of these "front-assist" cars could greater. 

## NETLOGO FEATURES

The plot shows both global values and the value for the last car in the queue, which helps one watch overall patterns and individual behavior at the same time.

The `watch` command is used to make it easier to focus on the orange car.


## RELATED MODELS

I have used parts of code from the 2 models below and I thank them for these very helpful examples:

* "Traffic Basic"
Wilensky, U. (1997).  NetLogo Traffic Basic model.  http://ccl.northwestern.edu/netlogo/models/TrafficBasic.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

* "Traffic 2 Lanes": a more sophisticated two-lane version of the "Traffic Basic" model.
Wilensky, U. & Payette, N. (1998). NetLogo Traffic 2 Lanes model. http://ccl.northwestern.edu/netlogo/models/Traffic2Lanes. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.


I used the NetLogo software:

*Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

This model was built by Philippe BALMAS for the student assignment following the lecture of Professor Georgy Bobashev , given to the A20-Cohort of DSTI on April 2021.
Thank you for quoting it if re-used. 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

convey
false
0
Rectangle -7500403 true true 4 15 195 157
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42
Rectangle -7500403 true true 180 75 180 75
Rectangle -7500403 false true 30 45 150 105
Rectangle -1 true false 30 45 165 120

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
setup
repeat 180 [ go ]
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
