; add model procedures here

globals [ main-antenna ; refers to the turtle acting as the emitter and receiver
          antenna-x antenna-y ; the x/y coordinates of the antenna
          antenna-heading     ; the heading of the antenna, in terms
                              ; of -180 to +180 degrees.
          antenna-direction   ; the direction of rotation of the antenna
          main-scope               ; refers to the turtles acting as the display scope
          scope-x scope-y     ; holds the x/y coordinates of the display scope
          scope-radius        ; holds the radius of the display scope
          horizon             ; the lowest that a UFO can fly
          envelope            ; the highest that a UFO can fly
          old-rf-setting
        ]

breed [ waves wave]
breed [ reflected-waves reflected-wave ]
breed [ antennas an-antenna]
breed [ ufos ufo]
breed [ scopes scopex]
breed [ scope-markers scope-marker]


waves-own
[ pips     ; used to measure the travel of the wave
  bounced? ; helps the wave remember that is has bounced off something
           ; used to prevent our simplified waves
           ; from bouncing multiple times
  wave-move
]
reflected-waves-own
[ pips     ; used to measure the travel of the wave
  bounced? ; helps the wave remember that is has bounced off something
           ; used to prevent our simplified waves
           ; from bouncing multiple times
  wave-move
]
ufos-own
[ goal-x goal-y ; where the ufo is heading
  speed         ; relative speed of the ufo
]

scope-markers-own [ pips ]

to setup
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-pips should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-pips at the end
  ;; of the procedure.)
  clear-all
  set-default-shape waves "wave"
  set-default-shape antennas "antenna"
  set-default-shape ufos "ufo"
  set-default-shape scope-markers "phosphor"

  antenna-setup
  scope-setup
  set envelope max-pycor
  set horizon antenna-y + .5 + 1
  create-ufos 3
  [ UFO-setup ]
end

to antenna-setup
  create-antennas 1
  [ set main-antenna self
    set antenna-direction 1
    set antenna-x 0
    set antenna-y 1
    setxy antenna-x antenna-y
    set size 2
    set heading 0
  ]
end

to ufo-setup
    ; start on left or right edge
    set xcor plus-or-minus ( max-pxcor  )
    ; start somewhere between horizon and upper limit of sky
    set ycor envelope - random-float ( envelope - horizon )
    ; head for opposite edge
    set goal-x (- xcor)
    ; at some other altitude
    set goal-y envelope - random-float ( envelope - horizon )
    ; point towards the goal
    set heading towardsxy goal-x goal-y
    ;set the size
    set size 1
    ; set the speed and color
    set speed (.5 + random-float 1) / patch-size
    set color random 13 * 10 +  15

end

to-report plus-or-minus [ value ]
    ; randomly reports either +value or -value
    report value * (((random 2) * 2) - 1)
    ; explanation of "(((random 2) * 2) - 1)"
    ; Operation:    Yields:
    ; random 2   -> 0           or 1
    ;      * 2   -> 0 * 2 =  0  or 1 * 2 = 2
    ;      - 1   -> 0 - 1 = -1  or 2 - 1 = 1
    ; thus, returns -1 or +1

end

to scope-setup
  set scope-x 0
  set scope-y 0 + min-pycor  - .5
  set scope-radius max-pycor + antenna-y ; - .5
  create-scopes 1
  [ set main-scope self
    set shape "scope"
    set size scope-radius * 2
    setxy scope-x scope-y
    set heading 0
    set color green
  ]
end

to go
  every (1 / 20) [  ufo-move ]
  every ((.5 * sweep-speed) / 60) [ antenna-sweep antenna-emit-pulse wave-propigate ]
  every (1 / 20) [ scope-fade ]
end

to-report wave-in
   report [ ->
    set pips pips + 1
    if pycor + dy < (antenna-y - 1) [ die ]
    jump (wave-length / patch-size)
    if ycor <= [ ycor ] of one-of antennas
      [ scope-activate ]
  ]
end

to-report wave-out
  report [ -> ; advance clock timing the pulse
      set pips pips + 1
      ; pulses are considered lost if they do not return in a certain time.
      ; but for display purposes, we will destroy pulses that leave the screen,
      ; or are below the antenna

      let res wave-length / patch-size
      ifelse not can-move? res
      [ die ]
      [ ; wave moves forward
        jump res
        ; if wave has not yet hit something and there's a UFO in the way...
        if any? ufos with [ distance myself <= .25 ]
        [ wave-bounce ]
      ]
  ]
end


to wave-propigate
  if rf-visible? or rf-visible? != old-rf-setting [ clear-drawing set old-rf-setting rf-visible? ]
  while [ any? waves ]
  [ ask waves [ run wave-move if rf-visible? [ stamp ] ]
  ]
  if rf-visible? [ display ]

end

to antenna-sweep
    set antenna-heading antenna-heading + 5 * antenna-direction
    ifelse reciprocate?
    [ if abs antenna-heading > 90
      [ set antenna-heading 90 * antenna-direction
        set antenna-direction antenna-direction * -1
      ]
    ]
    [ if abs antenna-heading >= 180
      [ set antenna-heading -180 * antenna-direction + antenna-direction
      ]
    ]
    ; antenna-dir 0 - antenna-dir
    ask main-scope [ set heading antenna-heading ]
    ask main-antenna [ set heading antenna-heading ]
end

to antenna-emit-pulse
  if abs antenna-heading < 90
  [ create-waves 1
    [ set shape "wave"
      set size wave-length / patch-size
      set heading antenna-heading
      setxy antenna-x antenna-y
      set bounced? false
      set pips 0
      set color white
      set hidden? not rf-visible?
      set wave-move wave-out
    ]
  ]
end

to scope-activate
    ; converts the incoming wave to mark the scope
    ; marks the scope based on the travel-time of the wave
     let detected-distance 0

     hide-turtle
     set breed scope-markers
     set shape "phosphor"
     set size  1

     ; move to center of scope
     setxy scope-x scope-y
     set color green + 1
     ; reverse heading
     set heading ( heading + 180 )
     ; calulate distance, and scale for scope viewing.
     set detected-distance pips * wave-length / patch-size / 2.5
     if detected-distance > scope-radius
     [ set detected-distance scope-radius ]
     jump detected-distance
     show-turtle
end

to scope-fade
    ask scope-markers
    [ ifelse size > (1 / patch-size)
      [ set size size * .96
      ]
      [ die ]
    ]
end

to ufo-move
  ask ufos
  [ ifelse ( can-move? (speed ) )
    [ jump speed ]
    [ ufo-setup ]
    ;; ask (waves in-radius size) with [ not bounced? ] [ wave-bounce ]
  ]
end


to wave-bounce
    set wave-move wave-in
    set heading heading + 180
    set color yellow
    set shape "wave-return"
    set bounced? true
end
@#$#@#$#@
GRAPHICS-WINDOW
145
10
498
364
-1
-1
15.0
1
10
1
1
1
0
0
0
1
-11
11
-11
11
0
0
1
ticks
30.0

BUTTON
75
10
130
43
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
9
10
64
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
10
90
132
123
RF-visible?
RF-visible?
0
1
-1000

SWITCH
10
50
131
83
reciprocate?
reciprocate?
0
1
-1000

SLIDER
10
130
135
163
wave-length
wave-length
1
patch-size * 2
4.0
1
1
NIL
HORIZONTAL

SLIDER
10
170
135
203
sweep-speed
sweep-speed
1
10
4.0
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This model is a simplified demonstration of RADAR.

RADAR is an acronym for RAdio Detection And Ranging

Radar is a detection system that uses radio waves to determine the distance (ranging), angle, or velocity of objects.

The radar unit emits a pulsed or continuous beam of electromagnetic energy. When the beam hits an object, some of the energy is reflected back to the source. The radar unit includes a processor that, depening on sooo many things, interprets the signal.

In our model, we simply use the round-trip time of the returned pulse to calculate the range of the object. This is likely wrong, but works for this demonstration. 

The returning pulse illuminates a simulated phospher display.
 
Sonor works in a similar fashion, but uses sound waves instead of radio waves.

## HOW TO USE IT

Click Setup.  
Click Go.

## CONTROLS

reciprocate?
  Whether the antenna sweeps in a circle, or goes back-and-forth.  

RF-visible?
  Makes the radar waves visible to the naked eye  

wavelength
  adjusts the simulated beam wavelength
  it is more difficult to detect objects with the object size
  is smaller than the wavelength of the beam

sweep-speed
  adjusts the rotation frequency of the antenna


## THINGS TO TRY


Use the slow-down slider to slow down time. Now you can follow the radio waves as they propigate out, bounce back, and are received again.

## EXTENDING THE MODEL

How about doppler radar?

## NETLOGO FEATURES

To get the sweeping single line of the scope, a custom turtle shape of a circle with a line from the center is used.  
Since patch resolution is not important, but large shapes are, the model uses few patches and a large patch size.
@#$#@#$#@
default
true
3
Polygon -13840069 true false 75 255 30 225 45 225 45 210 60 210 60 195 90 240
Polygon -13840069 true false 135 60 120 30 135 0 165 0 180 30 165 60
Polygon -6459832 true true 120 60 180 60 210 75 225 105 225 210 210 240 180 255 120 255 90 240 75 210 75 120 90 75 120 60 120 90 180 90 105 210 120 225 195 105 195 210 180 225 120 225 105 210 105 105 120 90
Polygon -13840069 true false 135 255 150 270 135 300 165 270 165 255
Polygon -13840069 true false 225 255 270 225 255 225 255 210 240 210 240 195 210 240
Polygon -13840069 true false 225 90 270 60 255 60 255 45 240 45 240 30 210 75
Polygon -13840069 true false 75 90 30 60 45 60 45 45 60 45 60 30 90 75
Polygon -10899396 true false 105 210 105 105 120 90 180 90
Polygon -10899396 true false 120 225 195 105 195 210 180 225

antenna
true
0
Polygon -7500403 false true 0 150 46 256 148 299 255 255 298 151 225 240 181 255 120 255 74 241
Polygon -7500403 false true 119 254 150 150 180 255

antenna2
true
0
Polygon -7500403 false true 0 150 46 256 148 299 255 255 298 151 225 240 181 255 120 255 74 241
Polygon -7500403 false true 119 254 150 150 180 255

phosphor
true
5
Circle -10899396 true true 16 16 268

scope
true
5
Circle -10899396 false true 15 15 270
Line -10899396 true 150 30 150 150
Circle -10899396 false true 15 15 270
Circle -10899396 false true 15 15 270
Circle -10899396 false true 15 15 270

turtlezero
true
3
Polygon -13840069 true false 225 90 270 60 255 60 255 45 240 45 240 30 195 90
Polygon -13840069 true false 75 90 30 60 45 60 45 45 60 45 60 30 105 90
Polygon -13840069 true false 225 240 270 210 255 210 255 195 240 195 240 180 195 240
Polygon -13840069 true false 75 240 30 210 45 210 45 195 60 195 60 180 105 240
Polygon -13840069 true false 135 60 120 30 135 0 165 0 180 30 165 60
Polygon -6459832 true true 120 60 180 60 210 75 225 105 225 210 210 240 180 255 120 255 90 240 75 210 75 120 90 75 120 60 120 90 180 90 105 210 120 225 195 105 195 210 180 225 120 225 105 210 105 105 120 90
Polygon -13840069 true false 135 255 150 270 135 300 165 270 165 255
Polygon -10899396 true false 105 210 105 105 120 90 180 90
Polygon -10899396 true false 120 225 195 105 195 210 180 225

ufo
false
4
Rectangle -1184463 false true 15 135 285 180
Rectangle -1184463 false true 60 120 239 194
Rectangle -1184463 false true 104 104 197 211

wave
true
2
Polygon -955883 false true 150 0 105 60 105 90 150 150 195 210 195 240 150 300 180 225 120 75

wave-return
true
4
Polygon -1184463 false true 150 0 120 60 120 90 150 150 180 210 180 240 150 300 165 225 135 75
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
