; Searching for Kevin Bacon model.
;
; Shows how agents can distribute information to aid network searches
; using word-of-mouth communication and blackboard communication.
;
; Copyright 2010 William John Teahan. All Rights Reserved.

breed [nodes node]
breed [searchers searcher]

nodes-own
[net-depth]         ; used when building some of the networks

searchers-own
[location            ; the node where the searcher is located
 time                ; time step
 height              ; used for hill climbing search
 path-cost           ; actual cost of search path so far (used for A-star search)
 estimated-cost]     ; estimated cost of cheapest path to goal

globals
[bacon-set           ; nodes linked with Kevin Bacon node
 kevin-bacon-node    ; node associated with Kevin Bacon
 kevin-bacon-xcor    ; x co-ordinate of Kevin Bacon (goal node)
 kevin-bacon-ycor    ; y co-ordinate of Kevin Bacon (goal node)
 search-time-step    ; indicates what the current time step is during the search
 visited-nodes       ; indicates which nodes have aleady been visited - do not revisit them in that case
 search-completed    ; true when search is completed
 searchers-used      ; number of searcher agents used
 max-active-searchers; maximum active searcher agents
 path-found          ; true if we have found a path to expand; e.g. used by DFS to signal a dead-end
 person-colour       ; colour of person shown doing the search in the visualisation
 IDS-depth ]         ; current depth limit for IDS (Iterative Deepening Search)


to create-network [number-of-nodes]
;; Creates number-of-nodes new nodes in the network.
  create-nodes number-of-nodes
  [ set color blue
    set label (word "b" who "    ") ]

  set bacon-set nodes
end

to setup-network
  clear-all
  set-default-shape nodes "circle 2"
  ;; create a random network

  create-network no-of-nodes

  set kevin-bacon-node min-one-of bacon-set [ who ]
  ask kevin-bacon-node
  [ set color white
    set size 3
    set shape "star" ] ; Kevin Bacon is a star

  create-network-links

  reset-layout
end

to create-network-links
;; creates the network links for the nodes that have none
;; according to the network-type
  ifelse (network-type = "P2P-no-super-nodes")
    [ create-network-P2P-no-super-nodes ]
    [ ifelse (network-type = "P2P-has-super-nodes")
      [ create-network-P2P-has-super-nodes ]
      [ ifelse (network-type = "P2P-random-single-link")
        [ create-network-P2P-random-single-link ]
        [ ifelse (network-type = "P2P-incremental")
          [ create-network-P2P-incremental ]
          [ ifelse (network-type = "P2P-incremental-1")
            [ create-network-P2P-incremental-1 ]
              [ ifelse (network-type = "Star-central-hub")
                [ create-network-star-central-hub ]
                [ create-network-hierarchical]]]]]]
end

to reset-layout
  clear-drawing

  layout

  ;; leave space around the edges
  ask nodes [ setxy 0.95 * xcor 0.95 * ycor ]

  ask kevin-bacon-node ; recalculate Kevin Bacon's position
  [ set kevin-bacon-xcor xcor
    set kevin-bacon-ycor ycor ]
end

to change-layout
  ask searchers [ die ] ;; kill off all current searchers as where their current location may no longer be a correct node position
  reset-layout
end

to layout
  ifelse layout-type = "spring"
    [ ifelse (network-type = "P2P-no-super-nodes" or network-type = "P2P-has-super-nodes")
    ;;incremental" or network-type = "P2P-incremental-1" or network-type = "hierarchical")
      [ repeat 500 [ layout-spring nodes links 0.1 9 5 ]]
      [ repeat 500 [ layout-spring nodes links 0.5 2 1 ]]]

;;else
    [ ifelse layout-type = "circle"
      [ let radius ifelse-value (max-pxcor <= max-pycor) [max-pxcor - 1] [max-pycor - 1]
        layout-circle nodes radius ]
      [ layout-radial nodes links kevin-bacon-node ]
    ]
end

to create-network-P2P-no-super-nodes
;; creates a P2P (peer-to-peer) layout without "super-nodes"
  let bacon-count count bacon-set
  let bacon-links ifelse-value (links-per-node < bacon-count) [links-per-node] [bacon-count]

  let n 0
  ask bacon-set
  [ if count my-links = 0
    [ set n random (bacon-links - 1) + 1
      create-links-with n-of n other bacon-set ]];; create at least one link, but not to itself
end

to create-network-P2P-has-super-nodes
;; creates a P2P (peer-to-peer) layout without "super-nodes"
  let bacon-count count bacon-set
  let bacon-links ifelse-value (links-per-node < bacon-count) [links-per-node] [bacon-count]
  let bacon-slinks ifelse-value (links-per-super-node < bacon-count - 1) [links-per-super-node] [bacon-count - 1]

  let n 0
  ask bacon-set
  [ if count my-links = 0
    [ set n random bacon-links + 1
      create-links-with n-of n other bacon-set ]];; create at least one link, but not to itself

  ;; now create super-nodes, making sure kevin bacon is included
  ask kevin-bacon-node
  [ if count my-links = 0
    [ set n random bacon-slinks + 1 ;; create at least one link
      create-links-with n-of n other bacon-set ]] ;; but not to itself
  ask n-of (no-of-super-nodes - 1) bacon-set
  [ if count my-links = 0
    [ set n random bacon-slinks + 1 ;; create at least one link
      create-links-with n-of n other bacon-set ]] ;; but not to itself
end

to create-network-P2P-random-single-link
  ask bacon-set
      [ if count my-links = 0
        [ create-link-with one-of other bacon-set ]]
end

to create-network-P2P-incremental
  let bacon-set1 (list kevin-bacon-node)

  foreach but-first sort bacon-set
  [ ask ?
      [ if count my-links = 0
        [ create-link-with one-of bacon-set1
          set bacon-set1 fput ? bacon-set1 ]]]
end

to create-network-P2P-incremental-1
  ask nodes [ set net-depth 0 ]

  ask kevin-bacon-node
      [ create-link-with one-of other bacon-set
        ask link-neighbors [set net-depth 1 ]]
  ask bacon-set
      [ if (net-depth = 0) and (count my-links = 0)
        [
          create-link-with one-of bacon-set with [net-depth = 1]
          set net-depth 1
        ]
      ]
end

to create-network-star-central-hub
  let bacon-count-1 count bacon-set - 1
  ask kevin-bacon-node
      [ if count my-links = 0
        [ create-links-with n-of bacon-count-1 other bacon-set ]]
end

to create-network-hierarchical
  ask nodes [ set net-depth 0 ]
  ask kevin-bacon-node ;; root of Kevin-Bacon sub-network
      [ set net-depth 1
        ifelse (count other bacon-set < links-per-node)
          [ create-links-with other bacon-set ]
          [ create-links-with n-of links-per-node other bacon-set ]
        ask link-neighbors [ set net-depth 2 ]]
  let depth 2
  let this-links 0
  let this-count 0
  while [count bacon-set with [net-depth = 0] > 0] ;; while still more nodes to place in network
      [ ask bacon-set with [net-depth = depth]
          [ set this-count count bacon-set with [net-depth = 0]
            set this-links ifelse-value (links-per-node < this-count) [links-per-node] [this-count]
            create-links-with n-of this-links bacon-set with [net-depth = 0]
            ask link-neighbors [ set net-depth depth + 1 ]]
        set depth depth + 1]

end

to setup-searcher
;; creates some new searchers
  set search-completed false
  if (visited-nodes = 0) ; not initialised yet
    [ set visited-nodes []]
  create-searchers 1
  [
    set size 2
    set pen-size 3
    set color magenta
    set shape "person"
    set person-colour magenta
    ifelse (start-node-number = 0)
      [ set location one-of nodes ] ; random start node
      [ set location node start-node-number ]
    set time search-time-step
    move-to location
    set height hill-height xcor ycor
  ]
  set searchers-used searchers-used + 1
end

to reset-searchers
;; resets the searchers by first killing off all existing ones, then
;; creating new ones, and resetting related counts and plots
  clear-drawing
  ask searchers [ die ] ;; kill off existing searchers
  set searchers-used 0
  ask links [ set thickness 0 ] ;; reset all network links to uncrossed
  setup-searcher
  set search-completed false
  set search-time-step 0
  set max-active-searchers 1
  set visited-nodes []
  reset-ticks
end

to go-searchers
  if (count searchers = 0)
    [ user-message "Couldn't find Kevin Bacon:\nCan't find any more paths to search or no searchers to do search!"
      stop ]

  if search-completed
    [ user-message "Found Kevin Bacon!"
      stop ]

  ; uninformed search strategies (blind search)
  if search-behaviour = "Breadth First Search"
    [ expand-breadth-first-search ]
  if search-behaviour = "Uniform Cost Search"
    [ expand-depth-first-search ]
  if search-behaviour = "Depth First Search"
    [ expand-depth-first-search ]
  if search-behaviour = "Multi-agent Depth First Search"
   [ expand-MA-depth-first-search ]
  if search-behaviour = "Depth Limited Search"
    [ expand-depth-limited-search ]
  if search-behaviour = "Iterative Deepening Search"
    [ expand-iterative-deepening-search ]

  ; informed search strategies
  if search-behaviour = "Greedy Best First Search"
    [ expand-greedy-best-first-search ]
  if search-behaviour = "A* Search"
    [ expand-A-star-search ]

  ; local search strategies
  if search-behaviour = "Hill Climbing Search"
    [ expand-hill-climbing-search ]

  set search-time-step search-time-step + 1

  if (count searchers = 0)
    [ user-message "No more searchers! Abort!"
      stop ]

  tick
end

to-report state [searcher-agent searcher-behaviour]
  ;; reports the state as a list of the current co-ordinates
  report (list [xcor] of searcher-agent [ycor] of searcher-agent)
end

to-report goal-node [this-node]
;; returns true if the searcher agent has reached the goal

  report this-node = kevin-bacon-node
end

to expand-breadth-first-search
  ; expand the search by adding more searcher-agents

  ask searchers
  [
    expand-paths (searcher who)
    die ; prune away parent agents used at lower time-steps
  ]
end

to expand-uniform-cost-search [curr-time]
  ; expand the search by adding more searcher-agents

  set path-found false
  ask first (sort-by [[path-cost] of ?1 < [path-cost] of ?2] searchers)
  [
    expand-paths (searcher who)
    die ; this agent has done its work; it's children are now doing the work
  ]
end

to expand-depth-first-search
  ; expand the search by following newest path
  ; do not follow the other paths in parallel; just follow one of them

  set path-found false
  ask first (sort-by [[time] of ?1 > [time] of ?2] searchers)
  [
    expand-paths (searcher who)
    die ; this agent has done its work; it's children are now doing the work
  ]
end

to expand-MA-depth-first-search
  ; expand the search by following longest path
  ; follow the other paths in parallel; but do not follow all of them

  set path-found false
  let agents ifelse-value (count searchers < max-agents-to-expand)
    [ count searchers ]
    [ max-agents-to-expand ]
  ask n-of agents turtle-set (sort-by [[time] of ?1 > [time] of ?2] searchers)
  [
    expand-paths (searcher who)
    die ; this agent has done its work; it's children are now doing the work
  ]
end

to expand-depth-limited-search
  ; expand the search by following longest path
  ; do not follow the other paths in parallel; just follow one of them
  ; limit the search depth to max-depth

  expand-depth-limited-search-1 max-depth
end

to expand-depth-limited-search-1 [maxdepth]
  ; expand the search by following longest path
  ; do not follow the other paths in parallel; just follow one of them
  ; limit the search depth to maxdepth

  set path-found false
  ask first (sort-by [[time] of ?1 > [time] of ?2] searchers)
  [
    if (time <= maxdepth)
      [ expand-paths (searcher who) ] ; only expand if not exceeded depth limit
    die ; this agent has done its work; it's children are now doing the work
  ]
end

to expand-iterative-deepening-search
  ; expand the search by iteratively performing depth-limited-search
  ; to increasingly greater depths
  set IDS-depth 1
  set person-colour magenta
  while [ IDS-depth <= max-depth ]
  [
    while [count searchers != 0]
      [ expand-depth-limited-search-1 IDS-depth ]
    set IDS-depth IDS-depth + 1
    ; change colour of person to reflect the increased maxdepth of search
    if (person-colour > 5)
      [ set person-colour person-colour - 5 ]
    setup-searcher
  ]
  set person-colour magenta ; restore default colour
end

to expand-greedy-best-first-search
  ; expand the search by adding more searcher-agents

  set path-found false
  ask first (sort-by [[estimated-cost] of ?1 < [estimated-cost] of ?2] searchers)
  [
    expand-paths (searcher who)
    die ; this agent has done its work; it's children are now doing the work
  ]
end

to expand-A-star-search
  ; expand the search by adding more searcher-agents

  set path-found false
  ask first (sort-by [[estimated-cost + path-cost] of ?1 < [estimated-cost + path-cost] of ?2] searchers)
  [
    expand-paths (searcher who)
    die ; this agent has done its work; it's children are now doing the work
  ]
end

to expand-hill-climbing-search
  ; expand the search using hill-climbing search method

  set path-found false
  ask searchers ; there should always be only one searcher at this stage
  [ expand-paths (searcher who) ] ; look to see where I can go next

  foreach but-first (sort-by [[height] of ?1 < [height] of ?2] searchers)
  [ ; kill off all but the first
    ask ? [ die ]; only let the best of the new searchers continue
  ]
end

to-report count-active-searchers
  report count searchers
end

to-report maximum-active-searchers
  let c count searchers
  if (c > max-active-searchers)
  [ set max-active-searchers c]

  report max-active-searchers
end

to-report total-searchers
  report searchers-used
end


to-report euclidean-distance [x y x1 y1]
;; reports the euclidean distance between points (x,y) and (x1,y1)
  report sqrt ((x1 - x) ^ 2  + (y1 - y) ^ 2)
end

to-report manhattan-distance [x y x1 y1]
;; reports the euclidean distance between points (x,y) and (x1,y1)
  report abs (x1 - x) + abs (y1 - y)
end

to-report heuristic-function [x y]
;; reports the heuristic evaluation function value

  let goalx kevin-bacon-xcor
  let goaly kevin-bacon-ycor

  if (heuristic = "Zero")
    [ report 0 ]
  if (heuristic = "Euclidean distance")
    [ report euclidean-distance x y goalx goaly ]
  if (heuristic = "Manhattan distance")
    [ report manhattan-distance x y goalx goaly ]
end

to-report hill-height [x y]
;; reports the "height" of the current search position
;; the zero height is the goal

  report heuristic-function x y
end

to expand-paths [searcher-agent]
;; expands all the possible paths for the searcher-agent

  foreach sort [link-neighbors] of [location] of searcher-agent
    [ expand-path searcher-agent ? ]
end

to expand-path [searcher-agent node]
  ; the searcher-agent creates a new searcher-agent that draws a path in the network from its
  ; current position to the node

  let xcor1 0
  let ycor1 0
  if not search-completed
    [ ; create a new path by creating an agent to search it
      ; check to see if the path has already been visited
      if allow-revisited-nodes? or not member? node visited-nodes
        [
          set path-found true
          if not allow-revisited-nodes?
              [set visited-nodes fput node visited-nodes] ; add to front of visited-nodes set

          hatch-searchers 1
            [ ; clone searcher
              set searchers-used searchers-used + 1

              set size 2
              set pen-size 5
              set color person-colour
              set shape "person"
              set xcor [xcor] of searcher-agent
              set ycor [ycor] of searcher-agent
              set xcor1 xcor ; copy xcor
              set ycor1 ycor ; copy ycor

              set heading [heading] of searcher-agent
              set time [time] of searcher-agent + 1
              set path-cost [path-cost] of searcher-agent
              pen-down

              ; move to the node
              set location node
              move-to location
              set xcor [xcor] of self
              set ycor [ycor] of self
              ; increment path cost when executing the behaviour using Euclidean distance
              set path-cost path-cost + euclidean-distance xcor1 ycor1 xcor ycor
              set estimated-cost (heuristic-function xcor ycor)

              set height hill-height xcor ycor
              stamp
            ]
        ]

      if goal-node node
        [ set search-completed true ]
    ]
end
;
; Copyright 2010 by William John Teahan.  All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from William John Teahan.
; Contact William John Teahan for appropriate licenses for redistribution for
; profit.
;
; To refer to this model in publications, please use:
;
; Teahan, W. J. (2010).  Searching For Kevin Bacon NetLogo model.
;   Artificial Intelligence. Ventus Publishing Aps.
;
@#$#@#$#@
GRAPHICS-WINDOW
210
10
647
448
-1
-1
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
