globals [
  map-canvas
  turf-colors
  map-valids
  map-openings
  turf-color
  path-color
  opening-color
  obstacle-color
  has-setup?
]
patches-own [
  turf-health
  patch-type
  visited?
  active?
  path-cost
  next-path-cell
]
turtles-own [
  destination
  pavement-preference
  grass-aversion
  upcoming-path
]

to setup-map
  clear-all
  set has-setup? true
  set-default-shape turtles "person"
  set turf-colors (list 37 56 56 54 53)

  let t 0
  let p 1
  let x 2
  let o 3

  ;; setup an initial map canvas
  set map-canvas (list
   ( list o p p p p p p p p p o )
   ( list p t t t t p t t t t t )
   ( list p t t t t p t t t t t )
   ( list p t t t t p t t t t t )
   ( list p t t t p p p t t t t )
   ( list p t t t p x p t t t t )
   ( list p t t t p p p t t t t )
   ( list p t t t t t t t t t t )
   ( list p t t t t t t t t t t )
   ( list p t t t t t t t t t t )
   ( list o p p p p p p p p p o )
  )

  resize-world 0 (length (item 1 map-canvas) - 1) 0 (length map-canvas - 1)
  ask patches [ set patch-type (item pxcor (item pycor map-canvas)) ]

  restart
end

to new-canvas
  clear-patches
  let #width-ratio (world-width - 1) / custom-world-width
  let #height-ratio (world-height - 1) / custom-world-height
  set-patch-size floor (patch-size * min (list #width-ratio #height-ratio))
  resize-world 0 custom-world-width 0 custom-world-height
  ask patches [ set patch-type 0 ]
  update-map-data
end

to-report patch-name-to-type [ #name ]
  if #name = "turf" [ report 0 ]
  if #name = "pavement" [ report 1 ]
  if #name = "obstacle" [ report 2 ]
  if #name = "opening" [ report 3 ]
end

to patch-draw
  if mouse-down? [
    ask patch mouse-xcor mouse-ycor [
      set patch-type patch-name-to-type draw-patch-type
    ]
    update-map-data
    ask patch mouse-xcor mouse-ycor [ display ]
  ]
end

to update-map-data
  ; Store a list of valid patches to traverse (non-obstacles)
  set map-valids ( patches with [ patch-type != 2 ] )
  set map-openings ( patches with [ patch-type = 3 ] )

  ask patches with [ patch-type = 0 ] [
    set pcolor last turf-colors
    set turf-health 1.0
  ]
  ask patches with [ patch-type = 1 ] [ set pcolor 8 ]
  ask patches with [ patch-type = 2 ] [ set pcolor black ]
  ask patches with [ patch-type = 3 ] [ set pcolor red ]
end

to restart
  if (has-setup? != true) [
    setup-map
    stop
  ]

  ask patches [ set plabel "" ]
  clear-turtles
  reset-ticks
  clear-all-plots
  update-map-data
  create-turtles population [
    ;; randomly assign agent prefs using bounded normal distribution
    set pavement-preference max (list 0.1 (random-normal avg-pavement-preference (avg-pavement-preference * relative-standard-deviation / 100)))
    set grass-aversion max (list 0 (random-normal avg-grass-aversion (avg-grass-aversion * relative-standard-deviation / 100)))
    move-to one-of map-openings
    pick-destination
  ]
end

to pick-destination
  ;; pick an opening that isn't the one here
  set destination ( one-of map-openings with [ self != [patch-here] of myself ] )
  set upcoming-path path-to-goal destination
end

to go
  if has-setup? != true [ setup-map ]
  ask turtles [ turtle-go ]
  ask patches with [ patch-type = 0 ] [ turf-go ]
  tick
end

to turtle-go
  ;; update destination if necessary
  if length upcoming-path = 0 [
    ;; set a new destination
    pick-destination
  ]
  ;; move one cell down path to destination
  move-to item 0 upcoming-path
  set upcoming-path remove-item 0 upcoming-path
end

to turf-go
  set turf-health  min list (turf-health + (growth-rate / 100)) 1
  set turf-health max list 0 (turf-health - (turf-fragility / 100 * (count turtles-here)))
  set pcolor item (round (turf-health * 4)) turf-colors
end

to-report cell-cost [ #traveler ]
  if (patch-type = 0) [ report 1 + ([grass-aversion] of #traveler * turf-health) ]
  if (patch-type = 1 or patch-type = 3) [ report 1 / [pavement-preference] of #traveler ]
  report 1
end

to-report list-path [ #path-so-far ]
  let #next-patch ([next-path-cell] of (last #path-so-far))

  if ([path-cost] of (last #path-so-far)) = 0 [
    report #path-so-far
  ]

  report list-path lput #next-patch #path-so-far
end

to-report path-to-goal [ #Goal ]
  let #Start patch-here
  let #traveler self

  ; clear all the information in the patches, and reset them
  ask map-valids [
    set next-path-cell nobody
    set path-cost (2 ^ 20)
    set visited? false
    set active? false
  ]

  ask #Start [
    set next-path-cell nobody
    set visited? false
  ]
  ask #Goal [
    set path-cost 0
    set visited? true
    set active? true
  ]

  while [not [visited?] of #Start] [
    let #options map-valids with [active?]

    ifelse any? #options [
      ask min-one-of #options [path-cost] [
        let #path-cost-father path-cost
        set active? false
        let #valid-neighbors neighbors with [member? self map-valids]
        ask #valid-neighbors [
          ;; get potential path cost, multiply cell cost by distance to account for diagonals
          let #new-path-cost #path-cost-father + (distance myself * cell-cost #traveler)
          if #new-path-cost < path-cost [
            set next-path-cell myself
            set visited? true
            set active? true
            set path-cost #new-path-cost
          ]
        ]
      ]
    ] [
      ;; We've run out of options before finding the Start
      show "No path available"
    ]
  ]

  let #future-path list-path (list #Start)

  report #future-path
end
@#$#@#$#@
GRAPHICS-WINDOW
280
10
529
260
-1
-1
21.91
1
10
1
1
1
0
0
0
1
0
10
0
10
1
1
1
ticks
30.0

BUTTON
401
302
517
335
NIL
new-canvas
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
194
136
257
169
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
7
10
179
43
population
population
1
50
50.0
1
1
NIL
HORIZONTAL

SLIDER
7
92
251
125
turf-fragility
turf-fragility
0
100
5.0
1
1
% damage
HORIZONTAL

SLIDER
7
51
186
84
growth-rate
growth-rate
0
10
0.8
0.2
1
percent
HORIZONTAL

SLIDER
7
179
223
212
avg-pavement-preference
avg-pavement-preference
0.1
2
1.2
0.05
1
NIL
HORIZONTAL

PLOT
8
301
259
473
Overall Turf Health
time
turf-health-pct
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -15040220 true "" "plot mean [turf-health] of patches with [patch-type = 0]"

CHOOSER
283
346
421
391
draw-patch-type
draw-patch-type
"turf" "pavement" "obstacle" "opening"
1

BUTTON
104
136
184
169
NIL
restart
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
283
302
386
335
NIL
patch-draw
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
283
439
456
472
custom-world-width
custom-world-width
3
20
20.0
1
1
NIL
HORIZONTAL

SLIDER
283
401
462
434
custom-world-height
custom-world-height
3
20
20.0
1
1
NIL
HORIZONTAL

BUTTON
7
136
95
169
NIL
setup-map
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
7
220
184
253
avg-grass-aversion
avg-grass-aversion
0
2
1.0
0.1
1
NIL
HORIZONTAL

SLIDER
8
259
243
292
relative-standard-deviation
relative-standard-deviation
0
30
30.0
1
1
%
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

The purpose of the agent based model is to understand trail creation and patterns of turf destruction from foot traffic on a college campus. Given a spatial map corresponding to an existing or planned campus or park, results from this model could inform adjusted path layouts or revegetation efforts to mitigate erosion damage from foot traffic. By running the model with a map of the target area with paths removed, it could inform future path layouts to optimize foot traffic and minimize erosion damage.

## HOW IT WORKS

People on the campus are all modeled as the same type agent. Agents are initialized with constant parameters for pavement preference and grass aversion. Agent location and destination are updated throughout the simulation after being initialized to one of the opening cells.

## HOW TO USE IT

### Global Settings

**population** - the number of students (turtles) on the map
**growth-rate** - percent of turf health regeneration per tick
**turf-fragility** - percent of damage traversing agents cause to turf per tick

### Agent Settings

**avg-pavement-preference** - amount agents prefer pavement over dead turf
**avg-grass-aversion** - amount agents dead turf over live turf
**relative-standard-deviation** - standard deviation of agent preferences as percentage of mean

## THINGS TO NOTICE

If you run the model for a long time (hint: turn off view updates for a while), turf destruction can build up after even thousands of ticks.

## THINGS TO TRY

Drag stochasticity to 0% to simulate a population of agents with identical preferences.

The `reset` button also regenerates agent preferences. Using this, you can see different desire paths formed with different agent preferences by turning on `go` and hitting `reset` after the turf stabalizes. You can do this faster by turning off view updates and keeping an eye on the turf health plot.

## EXTENDING THE MODEL

An interesting addition would be to model seasons, where maximum health of the grass oscillates gradually with time. For example, it could oscillate between a maximum of 0.5 for winter and 1 for spring. Making this model match reality would involve settings specific to the climate in question and the maintenance of the turf.

## NETLOGO FEATURES

This uses a variation of Dijkstra's pathfinding algorithm with weights based on terrain type and agent preferences.

You can also use the path-drawing tools to draw your own map:

1. turn on `patch-draw`
2. choose your `draw-patch-type`
3. draw on the canvas

If you want to resize the map canvas, set the `custom-world-height` and `custom-world-width` sliders and click `new-canvas`

## CREDITS AND REFERENCES

This model was created by Benjamin Duncan for an undergraduate Agent-Based Modeling course at Portland State University.

The model was used using the Netlogo software:

> Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2018 Benjamin Duncan.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Benjamin Duncan at benjamin.duncan@pdx.edu.

<!-- 2018 Cite: Duncan, B. -->
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
