globals
[
  max-money    ; maximum money a given patch can hold
  gini-index-reserve
  lorenz-points
]

patches-own
[
  money-here      ; the current amount of money on this patch
  max-money-here  ; the maximum amount of money this patch can hold
]

turtles-own
[
  age              ; how old a citizen is
  wealth           ; the amount of money a citizen has
  life-expectancy  ; maximum age that a citizen can reach
  spending         ; how much money a citizen spends
  opportunity      ; how far beyond their own patch a citizen can see
]

;;;
;;; SETUP AND HELPERS
;;;

to setup
  clear-all
  ;; set global variables to appropriate values
  set max-money 100 ;; this can be tuned
  ;; call other procedures to set up various parts of the world
  setup-patches
  setup-turtles
  update-lorenz-and-gini
  reset-ticks
end

;; set up the initial amounts of money each patch has
to setup-patches
  ;; give some patches the highest amount of money possible --
  ;; these patches are the "most money" where economic benefit is greatest
  ask patches
    [ set max-money-here 0
      if (random-float 100.0) <= percent-most-money
        [ set max-money-here max-money
          set money-here max-money-here ] ]
  ;; spread that money around the window a little and put a little back
  ;; into the patches that are the "most money" found above
  repeat 5
    [ ask patches with [max-money-here != 0]
        [ set money-here max-money-here ]
      diffuse money-here 0.25 ]
  repeat 10
    [ diffuse money-here 0.25 ]          ;; spread the money around some more
  ask patches
    [ set money-here floor money-here    ;; round money levels to whole numbers
      set max-money-here money-here      ;; initial money level is also maximum
      recolor-patch ]
end

to recolor-patch  ;; patch procedure -- use color to indicate money level
  set pcolor scale-color orange money-here 0 max-money
end

;; set up the initial values for the citizen variables
to setup-turtles
  set-default-shape turtles "dot"
  crt num-people
    [ move-to one-of patches  ;; put citizens on patch centers
      set size 1.5  ;; easier to see
      set-initial-turtles-vars
      set age random life-expectancy ]
  recolor-turtles
end

to set-initial-turtles-vars
  set age 0
  face one-of neighbors4
  set life-expectancy life-expectancy-min +
                        random (life-expectancy-max - life-expectancy-min + 1)
  set spending 1 + random spending-max
  set wealth spending + random 50
  set opportunity 1 + random max-opportunity
end

;; Set the class of the citizens -- if a given citizen has less than 1/3
;; the wealth of the richest citizen, color it RED. If between 1/3 - 2/3, color it GREEN
;; Over 2/3 color it BLUE.
to recolor-turtles
  let max-wealth max [wealth] of turtles
  ask turtles
     [ ifelse (wealth <= max-wealth * 0.90)  ;; default is / 3
        [ set color red ]
      [ ifelse (wealth <= max-wealth * 0.10) ;; default is 2/3
        [set color green]
        [ set color blue ]]]
end

;;;
;;; GO AND HELPERS
;;;

to go
  ask turtles
    [ turn-towards-money ]  ;; choose direction holding most money within the citizen's opportunity
  harvest
  ask turtles
    [ move-eat-age-die ]
  recolor-turtles

  ;; grow money every money-growth-interval clock ticks
  if ticks mod money-growth-interval = 0
    [ ask patches [ grow-money ] ]

  update-lorenz-and-gini
  tick
end

;; determine the direction which is most profitable for each citizen in
;; their environment. The 'opportunity index' represents the entrepreneurial
;; vision and/or how much the structure of their world limits or gives
;; economic opporunity to them
to turn-towards-money  ;; citizen procedure
  set heading 0
  let best-direction 0
  let best-amount money-ahead
  set heading 90
  if (money-ahead > best-amount)
    [ set best-direction 90
      set best-amount money-ahead ]
  set heading 180
  if (money-ahead > best-amount)
    [ set best-direction 180
      set best-amount money-ahead ]
  set heading 270
  if (money-ahead > best-amount)
    [ set best-direction 270
      set best-amount money-ahead ]
  set heading best-direction
end

to-report money-ahead  ;; citizens procedure
  let total 0
  let how-far 1
  repeat opportunity
    [ set total total + [money-here] of patch-ahead how-far
      set how-far how-far + 1 ]
  report total
end

to grow-money  ;; patch procedure
  ;; if a patch does not have it's maximum amount of money, add
  ;; num-money-grown to its money amount
  if (money-here < max-money-here)
    [ set money-here money-here + num-money-grown
      ;; if the new amount of money on a patch is over its maximum
      ;; capacity, set it to its maximum
      if (money-here > max-money-here)
        [ set money-here max-money-here ]
      recolor-patch ]
end

;; each citizens accumulates the money on its patch.  if there are multiple
;; citizens on a patch, divide the money evenly among the citizens
to harvest
  ; have citizens accumulate money before any other citizen sets the patch to 0
  ask turtles
    [ set wealth floor (wealth + (money-here / (count turtles-here))) ]
  ;; now that the money has been accumulated, have the citizens make the
  ;; patches which they are on have no money
  ask turtles
    [ set money-here 0
      recolor-patch ]
end

to move-eat-age-die  ;; citizen procedure
  fd 1
  ;; consume some money according to spending
  set wealth (wealth - spending)
  ;; grow older
  set age (age + 1)
  ;; check for death conditions: if you have no money or
  ;; you're older than the life expectancy or if some random factor
  ;; holds, then you "die" and are "reborn" (in fact, your variables
  ;; are just reset to new random values)
  if (wealth < 0) or (age >= life-expectancy)
    [ set-initial-turtles-vars ]
end

;; this procedure recomputes the value of gini-index-reserve
;; and the points in lorenz-points for the Lorenz and Gini-Index plots
to update-lorenz-and-gini
  let sorted-wealths sort [wealth] of turtles
  let total-wealth sum sorted-wealths
  let wealth-sum-so-far 0
  let index 0
  set gini-index-reserve 0
  set lorenz-points []

  ;; now actually plot the Lorenz curve -- along the way, we also
  ;; calculate the Gini index.
  ;; (see the Info tab for a description of the curve and measure)
  repeat num-people [
    set wealth-sum-so-far (wealth-sum-so-far + item index sorted-wealths)
    set lorenz-points lput ((wealth-sum-so-far / total-wealth) * 100) lorenz-points
    set index (index + 1)
    set gini-index-reserve
    gini-index-reserve +
    (index / num-people) -
    (wealth-sum-so-far / total-wealth)
  ]
end


; Copyright 1998 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
184
10
600
427
-1
-1
8.0
1
10
1
1
1
0
1
1
1
-25
25
-25
25
1
1
1
ticks
30.0

BUTTON
10
10
177
43
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
10
48
178
81
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
614
138
782
171
max-opportunity
max-opportunity
1
15
9.0
1
1
NIL
HORIZONTAL

SLIDER
616
231
784
264
money-growth-interval
money-growth-interval
1
10
6.0
1
1
NIL
HORIZONTAL

SLIDER
617
46
785
79
spending-max
spending-max
1
25
14.0
1
1
NIL
HORIZONTAL

SLIDER
10
142
178
175
num-people
num-people
2
1000
446.0
1
1
NIL
HORIZONTAL

SLIDER
7
393
176
426
percent-most-money
percent-most-money
5
25
5.0
1
1
%
HORIZONTAL

SLIDER
10
180
178
213
life-expectancy-max
life-expectancy-max
1
100
33.0
1
1
NIL
HORIZONTAL

PLOT
1039
194
1291
374
Class Plot
Time
Turtles
0.0
50.0
0.0
250.0
true
true
"set-plot-y-range 0 num-people" ""
PENS
"poor" 1.0 0 -2674135 true "" "plot count turtles with [color = red]"
"mid" 1.0 0 -10899396 true "" "plot count turtles with [color = green]"
"rich" 1.0 0 -13345367 true "" "plot count turtles with [color = blue]"

SLIDER
617
315
785
348
num-money-grown
num-money-grown
1
10
6.0
1
1
NIL
HORIZONTAL

SLIDER
10
217
178
250
life-expectancy-min
life-expectancy-min
1
100
15.0
1
1
NIL
HORIZONTAL

PLOT
1038
10
1291
190
Class Histogram
Classes
Turtles
0.0
3.0
0.0
250.0
false
false
"set-plot-y-range 0 num-people" ""
PENS
"default" 1.0 1 -2674135 true "" "plot-pen-reset\nset-plot-pen-color red\nplot count turtles with [color = red]\nset-plot-pen-color green\nplot count turtles with [color = green]\nset-plot-pen-color blue\nplot count turtles with [color = blue]"

PLOT
798
194
1032
374
Lorenz Curve
Pop %
Wealth %
0.0
100.0
0.0
100.0
false
true
"" ""
PENS
"lorenz" 1.0 0 -2674135 true "" "plot-pen-reset\nset-plot-pen-interval 100 / num-people\nplot 0\nforeach lorenz-points plot"
"equal" 100.0 0 -16777216 true "plot 0\nplot 100" ""

PLOT
798
10
1032
190
Gini-Index v. Time
Time
Gini
0.0
50.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -13345367 true "" "plot (gini-index-reserve / num-people) / 0.5"

TEXTBOX
625
269
775
297
How often economic opportunity aries.
11
0.0
1

TEXTBOX
627
353
777
381
How big the new economic opportunities are.
11
0.0
1

TEXTBOX
15
429
165
460
Orange most money and black least.
11
0.0
1

TEXTBOX
626
176
776
204
How much economic foresight a citizen has.
11
0.0
1

TEXTBOX
627
81
777
109
Maximum amount a citizen can spend in one time interval.
11
0.0
1

TEXTBOX
16
119
166
137
CITIZEN FEATURES
13
0.0
1

TEXTBOX
16
324
166
342
PLOT FEATURES
13
0.0
1

TEXTBOX
623
21
773
39
ECONOMIC DYNAMICS
13
0.0
1

TEXTBOX
36
260
186
302
RED      poor\nGRN     middle\nBLU      rich
11
0.0
1

TEXTBOX
14
343
164
385
These are characteristics of the landscape relative to citizens, money, and space.
11
0.0
1

@#$#@#$#@
## WHAT IS IT?

This model attempts to simulate a socio-economic structure observed by Michael B. Katz in his 1975 book, "The People of Hamilton, Canada West: Family and Class in a Mid-Nineteenth-Century City". In the book, he discusses how assessment rolls, city directories and other documents from 1851-1861 reflect two key dynamics: inequality and transience. The wealthy, established class of 1%-10% of the population stays largely the same, including the social institutions and structures of the city. The lower class, however, has high turn-over, low economic growth, and low property ownership. As with the original "Sugarscape" models and the secondary "Wealth Distribution" model, this adaptation attempts to set initial conditions that begin with disparity and, amid the dynamics of life, death, and economic activity, leave the economic gap largely unchanged. This is expressed in a near constant Gini index with significant lower class turnover, including possible movement with in the lower economic order but very little entry into the upper class. 

Does lower class volatility / transience prevent social resources from growing? Does this impair solidarity? Do the wealthy get both more money and more social resources in the form of stability and institutional continuity, etc. It seems likely that these types of dynamics lead to mutual reinforcement.

The wealthy have dynamic stability which is likely a function of the social continuity noted above - both personal networks and institutional enrichment.

## WHAT I NEED HELP WITH

I've been adapting the model but need help with the following:

1. It seems the split between the classes should be more clear with a probability function of some kind determining how likely it is that upper-class would move down and lower-class would move up. Maybe this isn't an important dynamic and can be excluded.
 
2. The split between classes should somehow record stability - ie. wealthy stay put, stay wealthy, while the lower class is more turbulent internally and in terms of movement in and out of the system. Katz research suggess as many as 2/3 of the population of 14,000 may have experienced moving in out betweeen 1851-1861. That transience takes the form of being in the assessment rolls at one point and then not thereafter.
                   * Perhaps the classes could have different death rates to stand in for the                  higher transience of the lower classes. This may be easier than trying to                 design a spatially transient characteristic into the lower class.
 
3. It would be useful to add a social relation function visible in the form of ties so that the wealthy would have strong, bonding ties with few bridging ties to the lower class. The lower class would have small numbers of close ties (as in a family arrangement of some kind - transience was not a solitary male type but included whole families), fewer ties within the lower class, and many fewer with the upper class.
 
4. The wealthy would have a much better economic patch (or patches) to begin with and those patches would remain so. Perhaps a patch that is economically high remains fixed proportional to its value and has the potential to disappear or degrade in proportion with its lower value. 
 
5. The image of the wealthy is something like a cluster of stable pillars of concrete that the lower class flows between and around. That doesn't quite get it because the water and concrete are two mutually exclusive materials whereas the wealthy and lower class are the same entities but with very different ranges of possibility. The feeling may be as exclusive as water flowing around concrete but the dynamic needs to be different than a straight flow model.


## HOW IT WORKS

What follows is an adaptation of the original "Wealth Distribution" model. Money is used instead of grain. Each patch has some economic value in terms of money and an ability to generate more. Citizens get money from their patch but also use some up to live. The more money they keep, the wealthier a citizen gets. 

The model begins with a roughly equal wealth distribution. THIS MAY NEED TO BE CHANGED AS A CITY DOES NOT BEGIN THIS WAY AND WEALTH WAS NOT DISTRIBUTED EQUALLY IN 1851 HAMILTON.

The citizens are mobile and can move around. They are trying to earn money and makes a decision about where to go based on the best opportunities they can see. They keep doing this until they die of old age [             ] or they run out of money and die of depravation [             ]. What they leave behind is an offspring. CHANGE IS NEEDED SO THAT WEALTH AND CHARACTERISTICS ARE PASSED TO THE OFFSPRING WITH PERHAPS A SMALL RANDOM VARIATION. IT IS IMPORTANT THAT LACK OR SURPLUS IS PASSED ALONG.

To observe the equity (or the inequity) of the distribution of wealth, a graphical tool called the Lorenz curve is utilized. The population is ranked by their wealth (as a percentage) and the percentage of the population that wealth represents (e.g. 30% of the wealth is owned by 50% of the population). The scales on the axes range from 0% to 100%. 

A 45 degree angle means 30% or people have 30% of wealth, 90% have 90% of the wealth, and so on. In an extreme case, one person (1%) could have all the money (100%) in which case the curve would look like an "L" but backwards. This is expressed in the Gini coefficient by a "0" if wealth is equally distributed and "1" if one person has all the wealth - the bigger the value, the more unequal it is. 

(0)    Communist idealism <---------------> Hyper wealthy   (1)

## HOW TO USE IT

The NUM-PEOPLE slider controls size of initial population and limits can be set in code.
The LIFE-EXPECTANCY-MAX slider is the oldest a citizen can be (number of ticks).
The LIFE-EXPECTANCY-MIN slider is the earliest point at which a citizen can die (fewest ticks).
The PERCENT-MOST-MONEY slider says how many citizens initially can have maximum wealth.
The MONEY-GROWTH-INTERVAL is a probability for how often economic opportunity occurs on a patch.
The NUM-MONEY-GROWN slider determines how big new economic opportunities are.
The MAX-OPPORTUNITY slider determines how good far beyond their own patch a citizen can see.
The SPENDING-MAX is the maximum amount a citizen can spend in one time interval.

## EXPERIMENTAL OBSERVATIONS

Settings of 1/3 of most, between 1/3 - 2/3 and greater than 2/3 
1. When the diffusion values are changed (ranging from 0.10 - 0.99) very little changes.
2. If initial diffusion and "spread it around some more" are highly contrasted very little changes (eg. 0.10 initial and 0.90 spread around some more).
3. Maximum economic foresight makes for a more volatile balance of high - mid - low income
4. When spending limit is very low, classes are closer to each other.
5. When spending is middle level, the high - low divide is more evident.
6. When spending is high, the high - low divide is evident.
7. The divides are very persistent across all variables.
8. When the spending, opportunity, etc. are lowest, the have / have-not divide is most striking.
9.It is really tough to change the divide much - to a degree, yes but in any fundamental way, no.

Settings of 1/5 of most, between 1/5 - 4/5 and greater than 4/5
* Not much notable change

Settings of 1/10 of most, between 1/10 - 9/10 and greater than 9/10
* With mid-point settings, a very low number of wealth with anlargely equal mid and low pattern emerges.

Settings of max-wealth * (0.90) for red and max-wealth * (0.10) for green
* Yields a significant number of poor and very few rich and middle income.


## NETLOGO FEATURES

Examine how the landscape of color is created --- note the use of the `scale-color` reporter.  Each patch is given a value, and `scale-color` reports a color for each patch that is scaled according to its value.

Note the use of lists in drawing the Lorenz Curve and computing the Gini index.

## CREDITS AND REFERENCES

This original model is based on a model described in Epstein, J. & Axtell R. (1996). Growing Artificial Societies: Social Science from the Bottom Up. Washington, DC: Brookings Institution Press.

For an explanation of Pareto's Law, see http://www.xrefer.com/entry/445978.

For more on the calculation of the Gini index see:

 * Deltas, George (2003).  The Small-Sample Bias of the Gini Coefficient:  Results and Implications for Empirical Research.  The Review of Economics and Statistics, February 2003, 85(1): 226-234.

In particular, note that if one is calculating the Gini index of a sample for the purpose of estimating the value for a larger population, a small correction factor to the method used here may be needed for small samples.

## HOW TO CITE

If you mention this model in a publication, we ask that you include these citations for the model itself and for the NetLogo software:

* Wilensky, U. (1998).  NetLogo Wealth Distribution model.  http://ccl.northwestern.edu/netlogo/models/WealthDistribution.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

This version of the model: Non-commercial. Copyright Milton Friesen, ingenuityarts@gmail.com, August 2017

![CC BY-NC-SA 3.0](http://i.creativecommons.org/l/by-nc-sa/3.0/88x31.png)

Original copyright 1998 Uri Wilensky.

![CC BY-NC-SA 3.0](http://i.creativecommons.org/l/by-nc-sa/3.0/88x31.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.

The "Wealth Distribution" original model was created as part of the project: CONNECTED MATHEMATICS: MAKING SENSE OF COMPLEX PHENOMENA THROUGH BUILDING OBJECT-BASED PARALLEL MODELS (OBPML).  The project gratefully acknowledges the support of the National Science Foundation (Applications of Advanced Technologies Program) -- grant numbers RED #9552950 and REC #9632612.

The original model was converted to NetLogo as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and/or INTEGRATED SIMULATION AND MODELING ENVIRONMENT. The project gratefully acknowledges the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227. Converted from StarLogoT to NetLogo, 2001.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225
@#$#@#$#@
0
@#$#@#$#@
