;;;;;;;;;;;;;;;;;;;
;;  Global Info  ;;
;;;;;;;;;;;;;;;;;;;
;; global variables are either declared here on in the interface (but not both)
;; if added to the interface, global variables are converted to comments
;; this maintains the description of the variable here 
globals [
;; control variables 
  clock                                           ;; tracks in minutes the simulation duration
  ;;sim-runs                                      ;; number of times to run scenario
  sim-run                                         ;; number of scenarios run
  ;;sim-type                                      ;; false for legacy scenario, true for FUWS scenario
;;Environment
  FieldSize                                       ;; used to control distances in the world
;; FUWS
  ;;Minelines                                     ;; used in the layout/planning of legacy minefields
  ;;MineQty                                       ;; number of mines in a legacy scenario
  ;;MobileWeaponQty                               ;; number of weapons in a FUWS scenario
;; communicator parameters
  s-comms-range                                   ;; range of undersea comms between sensors
  w-comms-range                                   ;; range of undersea comms from sensors to weapons
;; threat parameters
  ;;t-number                                      ;; number of enemy vessels per scenario
  t-number-list                                   ;; list of the t-numbers
  t-count                                         ;; number of enemy vessels generated in a scenario
  t-goal                                          ;; the patch to which the enemy vessles are transisting
  t-yline                                         ;; the line used to transit the minefield
  ;;t-speed                                       ;; speed of enemy vessel
  ;;t-tactic                                      ;; used to select the enemy vessel tactic
;; measures of effectiveness
  fails                                           ;; Used to count weapon system failures (threat transit success)
  kills                                           ;; Used to count threat kills by transitor sequence
  ExpectedCasualties                              ;; This is used to track the mean number of casualties in a sim-run
  missed                                          ;; Counts weapons launched that failed to acquire a target
  Threat1                                         ;; This is used to calculate the simple initial threat point estimate
  Threat1SE                                       ;; This is used to calculate the standard error of the simple initial threat point estimate
  ThreatN                                         ;; This is used to calculate the threat number
  PkDegradation                                   ;; This may be used to calculate how the risk decays
  FieldConfigDegradation                          ;; This may be used to calculate how the minefield decays
  FieldConfigCost                                 ;; This may be used to report the cost of a COA
]

;;;;;;;;;;;;;;
;;  Agents  ;;
;;;;;;;;;;;;;;
;; this section describes the agent classes (breeds)
;; and the variables assigned to each member of the agent class
;; this maintains the description of the variable here 
;; comments are used to describe the purpose & use of each variable

;; FUWS architecture components
  breed [ sensors  sensor ]                       ;; FUWS sensors
  sensors-own [ s-detect s-line s-sensitivity]
    ;; s-detect is used to control the state of the sensor (1 = threat detected, 0 = no threat detected)
    ;; s-line is used to assign sensors to a mine-line (integer)
    ;; s-sensitivity is used to calculate probability of detection based on the reference target

  breed [ weapons  weapon ]                       ;; FUWS weapons
  weapons-own [ w-exploder w-line w-tnt w-patrolspeed w-interceptspeed w-priority w-energy]
    ;; w-exploder is used to control the state of the warhead (1 = detonation ordered, 0 no detonation ordered)
    ;; w-line is used to assign mines to a mine-line or to a UUV battery of weapons (integer)
    ;; w-tnt is used to assign the explosive power of warheads
    ;; w-patrolspeed sets the speed for a mobile platform  (UUV type weapon)
    ;; w-interceptspeed sets the speed for a torpedo launched at the enemy vessel
    ;; w-priority determines the firing order

  directed-link-breed [ commslinks  commslink ]   ;; FUWS comms nodes link sensors to weapons
  undirected-link-breed [ SensorNetworkLinks SensorNetworkLink ]  ;; FUWS comms nodes link sensors to sensors

;; Targets
  breed [ targetvessels targetvessel ]            ;; Enemy Vessels
  targetvessels-own [ t-damage ] 
    ;; t-damage is used to control the level of damage to the Enemy Vessel (%)
    ;; t-class is used to designate target classification
    ;; t-depth for submerged targets
    ;; t-vulnerability is used to describe the 
    ;; t-size is used to describe the 
    ;; t-signature is used to scale the target's signature to the reference target

;; Environment
  patches-own [ e-depth e-svp e-bottomtype e-ambientnoise]
    ;; environment parameters are not used in this demo
    ;; future development could extend to include this information
    ;; in calcualtion of the p(detection) and p(kill)



;;;;;;;;;;;;;;;;;;;;
;; Scenario Logic ;;
;;;;;;;;;;;;;;;;;;;;
;; These procedures are used to initiate and control the action of the actors in the scenario
;;------------------------------------------------------------------------------;;
to go
;; this procedure initiates the scenario when directed by the user
;; and defines the scenario sequence
  ifelse (any? TargetVessels) [ tick (set clock (clock + .1) )]             ;; if a target exists, time advances in six second increments
     [ifelse  t-count < t-number [setup-nexttarget] [                       ;; if a target doesn't exist, either setup the next target
         ifelse sim-run = sim-runs [endsim stop] [nextsim] ]]                      ;; or set up the next scenario unless all runs are complete
  move-targets
  ifelse sim-type = "fuws" [ fuws-intercept ] [ explode-in-place ]          ;; selects targeting logic consitent with deployed system
  check-kills
  check-failure
end

 
;;------------------------------------------------------------------------------;;
to fuws-intercept
;; this procedure is one possible model of a fuws interaction
;; determines if a sensor detects the enemy vessel
  let i 1              ;; local variables used to aim the weapon 
  let j FieldSize / 2
  let k 0              ;; local variable to identify the initial detecting sensor
  let l [ who ] of targetvessels
  
  ask targetvessels [ ask sensors in-radius figure-of-merit [set s-detect 1 set color yellow ]]
  ;; sensors that detect an enemy vessel send a detection signal to sensors in the network
  ask sensors [ if ( s-detect = 1 ) [set i pxcor set j pycor set k who
      ask SensorNetworkLink-neighbors [if s-detect = 0 [set s-detect 0.5 set color green ]]]]
  ask sensors [ if ( s-detect = 0.5 ) [ask SensorNetworkLink-neighbors [if s-detect = 0 [set s-detect 0.5 set color lime ]]]]
  ask sensors [ if ( s-detect = 0.5 ) [ask SensorNetworkLink-neighbors [if s-detect = 0 [set s-detect 0.5 set color turquoise ]]]]
  ask sensors [ if ( s-detect = 0.5 ) [ask SensorNetworkLink-neighbors [if s-detect = 0 [set s-detect 0.5 set color sky ]]]]
  ;; sensors that detect an enemy vessel send a detection signal to sensors in the network
  ask sensors [ if ( s-detect > 0 ) [ask out-commslink-neighbors [ 
      ask weapons [face sensor k rt (heading - 270) / 4]                ;; aims the weapon assumes target course 090 speed 11kts
      ask weapons with-max [w-priority] [set w-exploder 0.5 set shape "torpedo"]] ]]
  ask weapons [ if (w-exploder = 0.5) [                         ;; warheads with w-exploder 0.5 attempt to intercept the target
      ask my-in-commslinks [die]
      if any? targetvessels in-cone .1 180 [set w-exploder 1]
      if any? targetvessels in-cone .2 90 [set w-exploder 1]
      if any? targetvessels in-cone  2 45 [face targetvessel first l show "homing"]
      fd (w-interceptspeed / 300 )
      set w-energy ( w-energy - 1 ) ]]                      ;; moves weapon forward by a distance equal to 6 sec

;; warheads damage enemy vessels within the blast radius
   ask weapons [ if (w-exploder = 1 ) [ask targetvessels in-radius blast-radius [set color pink set t-damage 1 ]]]
   ask weapons [ if (w-exploder = 1 ) [ show "boom" die ] ]
;; warheads are removed after they deplete available energy
   ask weapons [ if w-energy = 0 [ set missed missed + 1 show "weapon shut-down" die] ]
   ask sensors [set s-detect 0 set color cyan ]

end

;;------------------------------------------------------------------------------;;
to move-targets
;; this procedure determines how targets in the scenario move
    ifelse t-tactic = "direct path" [
      ask TargetVessels [ facexy (item 0 t-goal) (item 1 t-goal)]]   ;; this logic drives directly to the goal point
      [ ask TargetVessels [                                          ;; this logic drives to the selected t-yline to cross the minefield
             ifelse pxcor = item 0 t-goal [facexy (item 0 t-goal) (item 1 t-goal)]
                                          [ifelse pycor = t-yline [facexy (item 0 t-goal) t-yline ]
                                                                  [facexy (-4)  t-yline ] ] ] ]
   ask TargetVessels [ fd (t-speed / 300 ) ]                       ;; moves target forward by a distance equal to 6 sec
end 

;;------------------------------------------------------------------------------;;
to explode-in-place
;; this procedure is used to model a legacy mine interaction
;; determines if a sensor detects the enemy vessel
ask targetvessels [ ask sensors in-radius figure-of-merit [set s-detect 1 set color pink ]]
;; sensors that detect an enemy vessel send a denonation signal to the warheads
ask sensors [ if ( s-detect = 1 ) [ask in-commslink-neighbors [ set w-exploder 1 set color pink ]]]
;; warheads send damage to enemy vessels within the blast radius
ask weapons [ if (w-exploder = 1 ) [ask targetvessels in-radius blast-radius [set color pink set t-damage 1 ]]]
;; warheads are removed after they explode
if any? weapons with [w-exploder = 1] [ show "boom!"]
ask weapons [ if (w-exploder = 1 ) [ die ] ]
end


;;;;;;;;;;;;;;;
;;  Outputs  ;;
;;;;;;;;;;;;;;;
;; These procedures are used to track and report on MOE's/MOP's
;; Still working on this 
;;------------------------------------------------------------------------------;;
to check-kills
;; this procedure removes targets that are killed by the minefield
  ask targetvessels [ if ( t-damage >= 1 ) [ die ]]
  ifelse (any? TargetVessels) [ ]
         [ set kills replace-item (t-count - 1) kills (1 + item (t-count - 1) kills)
           output-moe]
end

;;------------------------------------------------------------------------------;;
to check-failure
;; this procedure removes targets that the minefield "fails" to kill
  if any? targetvessels-on patch (FieldSize + 5) (FieldSize / 2) [
          set fails replace-item (t-count - 1) fails (1 + item (t-count - 1) fails)
          output-moe]
  ask patch (FieldSize + 5) (FieldSize / 2) [ask targetvessels-here [die] ]
end


;;------------------------------------------------------------------------------;;
to output-moe
;; this proceedure updates each MOE when called
;; will call additional MOE calculations when developed
  Update-ThreatProfile
  Update-SIT
  Update-EC
end

;;------------------------------------------------------------------------------;;
to Update-ThreatProfile
;; this proceedure updates the threat profile MOE
ifelse sim-run = 1  [set ThreatN map [(sim-run - ?) / sim-run] fails]
                    [set ThreatN (map [ precision (?1  / ( ?1 + ?2 )) 3 ] kills fails)]
;;  output-print ThreatN                       ;; prints the Threat Profile
end

;;------------------------------------------------------------------------------;;
to Update-SIT
;; this proceedure updates the SIT and associated standard error
  set Threat1 first ThreatN
  set Threat1SE (1 / sim-run) * SQRT ((1 / sim-run) * first kills * (sim-run - first kills))
end

;;------------------------------------------------------------------------------;;
to Update-EC
;; this proceedure updates the EC point estimate
let j 0
  foreach kills [set j (j + ?)]
  set ExpectedCasualties j / sim-run
end




;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Weapon System Setup  ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;
;; One of the following proceedures is used to set-up the inital weapon system deployment
;;------------------------------------------------------------------------------;;
to setup-weaponsystem
    if FixedField = True [random-seed 1941]          ;;the definition of a random seed will ensure the same minefield is generated each time
    ifelse sim-type = "fuws" [ setup-fuws ] [
    ifelse sim-type ="radial minefield" [ setup-minefield_radialminelines ] [
      ifelse sim-type ="vertical minefield" [ setup-minefield_verticalminelines] []]]
    ;; selects the weapon system setup for use **to to expand options**
    random-seed new-seed
end

;;------------------------------------------------------------------------------;;
to setup-fuws
;; this stub procedure would be used to set up a fuws scenario - it may need to call separate procedures to setup sensor and weapon fields
  create-sensors SensorQty
  ask sensors [set color cyan set s-detect 0
    setxy (1)  ( FieldSize / 2 )                                   ;; moves sensors to left edge of minefield
    set s-line (remainder who Lines)                               ;; assigns to sensor line
    set heading 90] 
  let i 0     ;; local variable used to iterate for each sensorline
  let j 0     ;; local variable used to determine sensor line axis
  let k 0     ;; local variable used to offset sensorline from edge
  repeat Lines [                                                              ;; lays each line of sensors
    set j (260 + random-float 20 ) 
    set k (Fieldsize * .6 * random-float (i + 1) / Lines)
    ask sensors [if (s-line = i) [ fd k rt j 
    fd (2 * (random-float (0.45 * FieldSize) ) - (0.45 * FieldSize)) ]]           
    set i ( i + 1 )]
  ask sensors [ create-SensorNetworkLinks-with other sensors in-radius s-comms-range ]
  ;;ask SensorNetworkLinks [ hide-link ]

  create-weapons MobileWeaponQty
  ask weapons [ set color cyan set w-exploder 0
    set w-tnt 1                                                                     ;; sets explosive power on scale of 0-1
    set w-patrolspeed 0                                                             ;; sets patrol speed in kts
    set w-interceptspeed 45                                                         ;; sets intercept speed in kts
    set w-energy 100                                                                ;; energy used at 1-unit per 6 sec at intercept speed
    set w-priority (remainder who MobileWeaponQty)
    setxy ( 3 * FieldSize / 4 ) ( FieldSize / 2 )                                   ;; moves mines to right of minefield
    set w-line (remainder who UUVQty)]
  set i 0    ;; local variable used to iterate for each UUV weapon bank
  repeat UUVQty [                                                                   ;; positions each UUV
    set j (350 + random-float 20 )
    set k (random-float (0.45 * FieldSize))
    ask weapons [ if (w-line = i) [ set heading j fd (2 * k - (0.45 * FieldSize)) ] ]           
    set i ( i + 1 )]

  ask sensors [ create-commslinks-to weapons in-radius w-comms-range ] 
  ;;ask commslinks [ hide-link ]
end

;;------------------------------------------------------------------------------;;
to setup-minefield_radialminelines
;; this procedure is used to setup a legacy minefield with minelines concentrating near the center of the minefield
  create-weapons MineQty                                 ;; generates
  ask weapons [ set color cyan
    setxy ( FieldSize / 2 ) ( FieldSize / 2 )            ;; moves mines to center of minefield
    hatch-sensors 1]                                     ;; creates a sensor atop each mine

  ask weapons [
    create-commslink-to sensor (who + MineQty) [ tie ]     ;; creates a tied link from sensor to mine
    set w-line (remainder who Lines)                       ;; assigns each mine to a mineline
    set heading (w-line * 360 / Lines)]                    ;; sets direction of offset for the mineline from the center of the minefield
    ask commslinks [ hide-link ]
  
  let i 0                      ;; local variable used to iterate for each mineline
  let j 0                      ;; local variable used to determine mineline axis
  let k 0                      ;; local variable used to offset mineline from center
 
  repeat Lines [               ;; lays each mineline - other logics for laying the initial minefield could be employed
    set j (120 + random-float 60 )                ;; radomizes heading of each mineline by adding 120 to 180 to the offset direction
    set k (random-float (Fieldsize * .1))         ;; randomizes the offset of each mineline from the center between 0 * 10% of the minefield size
    ask weapons [if (w-line = i) [ fd k rt j      ;; moves each mine in the mineline out
    fd (2 * (random-float (0.4 * FieldSize) ) - (0.4 * FieldSize)) ]]
    set i ( i + 1 )]  
end 

;;------------------------------------------------------------------------------;;
to setup-minefield_verticalminelines
;; this procedure is used to setup a legacy minefield with minelines concentrating near the center of the minefield
  create-weapons MineQty                                 ;; generates
  ask weapons [ set color cyan
    setxy ( 2 ) ( FieldSize / 2 )                        ;; moves mines to left edge of minefield
    hatch-sensors 1                                      ;; creates a sensor atop each mine
    set heading 90
    set w-line (remainder who Lines)]
  ask weapons [create-commslink-to sensor (who + MineQty) [ tie ] ] ;; creates a tied link from sensor to mine
  ask commslinks [ hide-link ]
  
  let i 0                      ;; local variable used to iterate for each mineline
  let j 0                      ;; local variable used to determine mineline axis
  let k 0                      ;; local variable used to offset mineline from center

  repeat Lines [                                                             ;; lays each line of mines
    set j (260 + random-float 20 ) 
    set k (.85 * Fieldsize * random-float (i + 1) / Lines)
    ask weapons [if (w-line = i) [ fd k rt j 
    fd (2 * (random-float (0.48 * FieldSize) ) - (0.48 * FieldSize)) ]]           
    set i ( i + 1 )]
end 

;;;;;;;;;;;;;;;;;;
;; Calculations ;;
;;;;;;;;;;;;;;;;;;
;; These procedures are used to calculate and return numbers 
;;------------------------------------------------------------------------------;;
to-report figure-of-merit
  ;;this procedure returns the range at which a target is detected
  ;;this may be a randomized number based on t-signal and s-sensitivity or a fixed number
  report random-normal 0.2 (0.2 / 8)
end

;;------------------------------------------------------------------------------;;
to-report blast-radius
  ;;this procedure returns the range at which a target damage from the warhead is a mission kill
  ;;this may be a randomized number based on t-vulnerability and w-power or a fixed number
  ;;we will want to update this logic to support calculating damage as a function of range
  report random-normal 0.2 (0.2 / 16)
end


;;;;;;;;;;;;;;;;
;; Initialize ;;
;;;;;;;;;;;;;;;;
;; These procedures are used to setup and establish the scenario components and geographic layout
;;------------------------------------------------------------------------------;;
to setup
  clear-all
  reset-simulationvariables             ;; these global variables are used to execute the simulation
  reset-scenariovariables               ;; these global variables are not (yet) controlled by the interface and describe the scenario
  setup-environment                     ;; establishes the environmet/patches initial conditions
  setup-weaponsystem
  setup-nexttarget                      ;; generates the first target
  reset-ticks
end

;;------------------------------------------------------------------------------;;
to reset-scenariovariables
;; initial values for global variables not controlled by the interface are set here
;; simulation variables are used to execute the scenario
;; these values may change in the course of a scenario and can be reset by calling this procedure
  set FieldSize                  20                       ;; 20 = 10nm * 10nm area each patch is 1kyd
;; component measures of performance
  set s-comms-range                4                       ;; sets range in kyds betweens comms nodes
  set w-comms-range                10                      ;; sets range in kyds betweens comms nodes
;; threat parameters
  ;;set t-signal                 0.1                      ;; enemy vessel signal used to calculate sensor probability of detection
  ;;set t-vulnerability          1                        ;; enemy vessel vulnerability to damage factor
  set t-goal                     [ ]
end

;;------------------------------------------------------------------------------;;
to reset-simulationvariables
;; initial values for global variables not controlled by the interface are set here
;; simulation variables are used to execute the scenario
;; these values may change in the course of a scenario and can be reset by calling this procedure
;; component measures of performance
;; control variables
  set clock                    0                        ;; tracks time in min
  set sim-run                  1                        ;; counts simulation runs 
  set t-count                  0                        ;; counts enemy vessels generated
  set t-yline                  FieldSize / 2 
;; default shapes
  set-default-shape weapons "x"                         ;; mines appear as x's
  set-default-shape sensors "circle 2"                  ;; sensors will appears as circles
  set-default-shape TargetVessels "surf_hostile"
;; measures of effectiveness
  set fails n-values t-number   [ 0 ]                   ;; records the success=0 or failure=1 of the minefield in each trial
  set kills n-values t-number   [ 0 ]                   ;; records the success=0 or failure=1 of the minefield in each trial
  set missed                    0
  set ExpectedCasualties        0                       ;; This is used to track the mean number of casualties in a sim-run
  set Threat1                   0                       ;; calculates the threat to the single initial transister (SIT)
  set Threat1SE                 0.5
  set ThreatN n-values t-number [ 0 ]                   ;; creates a list of the threat presented to the nth transiter
  set PkDegradation             0                       ;; available metric
  set FieldConfigDegradation    0                       ;; available metric
  set FieldConfigCost           0                       ;; would support CAIV analysis
  set t-number-list             [ ]                     ;; creates a list of the transiter number for plotting against the threat  
  let i 1 repeat t-number [set t-number-list lput i t-number-list set i (i + 1)]
end

;;------------------------------------------------------------------------------;;
to setup-environment
;; sets minefield size - note each patch is 1kyd*1kyd
  resize-world -5 ( FieldSize + 5 ) -5  (FieldSize + 5 )
  ;; to provide for maneuver outside the minefield, sets the size of the world 5kys beyond the minefield in all directions
  ;; the origin is moved to the south-west corner of the minefield area
  ask patches [ set pcolor black set e-depth 200 ]
  ;; this section would be extended to support file based inputs to describe real environments
;; draws minefield boundry
  crt 1 [set color cyan set pen-mode "down" ]
  ask turtle 0 [setxy 0 FieldSize setxy FieldSize FieldSize setxy FieldSize 0 setxy 0 0 ]
  ct
;; places land at top and bottom of screen to visually simulate chokepoint
  ask patches [
    if ( pxcor >= 0 and pxcor < FieldSize and (pycor = -5 or pycor = ( FieldSize + 5 )))
      [set pcolor orange set e-depth 0]
    if ( pxcor >= 2 and pxcor < (FieldSize - 2) and (pycor = -4 or pycor = ( FieldSize + 4 ) ))
      [set pcolor orange set e-depth 0]
    if ( pxcor >= 4 and pxcor < (FieldSize - 4) and (pycor = -3 or pycor = ( FieldSize + 3 ) ))
      [set pcolor orange set e-depth 0]  ]
;; sets and marks the target objective
   set t-goal lput (FieldSize + 5) t-goal 
   set t-goal lput (FieldSize / 2) t-goal 
   ask patch (FieldSize + 5) (FieldSize / 2) [set pcolor 5]
end

;;------------------------------------------------------------------------------;;
to setup-nexttarget
;; this procedure is used to create a new enemy vessel
   create-TargetVessels 1                              ;; creates initial enemy target, future targets will spawn as specified by the scenario
   ask TargetVessels [ set color red set size 2
     setxy -4 random-ycor pd]                          ;; sets enemy vessel on west edge of minefield
   set t-count (t-count + 1 )                          ;; incriments target vessles
;; sets the path the target will take through the minefield
;; if edge line is used all targets in a simulation will take the same 1ky wide corridor thru
;; however, the location of the path is randomized between simulations
   if t-tactic = "center line" [ set t-yline FieldSize / 2 ]
   if t-tactic = "edge line" [ ask TargetVessels [
      ifelse t-count = 1 [set t-yline round (1 + random-float (FieldSize / 4)) ][set t-yline t-yline]]]
end

;;------------------------------------------------------------------------------;;
to nextsim
  if (sim-run < sim-runs) [
    set sim-run (sim-run + 1)
    set t-count 0
    ct
    setup-weaponsystem
    setup-nexttarget
    go]
end

;;------------------------------------------------------------------------------;;
to endsim
  file-open "MFSA_prototype_output.csv"
  file-write ExpectedCasualties
  file-write ThreatN
  file-close-all
end
@#$#@#$#@
GRAPHICS-WINDOW
400
10
818
449
-1
-1
13.1613
1
10
1
1
1
0
0
0
1
-5
25
-5
25
0
0
1
ticks
30.0

BUTTON
10
10
92
43
Initialize
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
100
10
186
43
Start Sim
Go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
5
140
177
173
Lines
Lines
1
20
5
1
1
NIL
HORIZONTAL

SLIDER
5
180
177
213
MineQty
MineQty
0
100
0
1
1
NIL
HORIZONTAL

TEXTBOX
15
105
165
131
Undersea Weapon System\n               Parameters
11
0.0
1

SLIDER
195
140
367
173
t-speed
t-speed
1
25
12
1
1
NIL
HORIZONTAL

TEXTBOX
200
120
350
138
Enemy Vessel Parameters
11
0.0
1

SLIDER
195
180
367
213
t-number
t-number
0
10
6
1
1
NIL
HORIZONTAL

INPUTBOX
195
50
280
110
sim-runs
100
1
0
Number

PLOT
835
10
1240
235
Threat by Transiter
Transiter
Threat Presented
0.0
10.0
0.0
1.0
true
false
"clear-plot\nset-plot-x-range 0 t-number\nset-plot-y-range 0 1" ""
PENS
"default" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item 0 t-number-list - 1) (item 0 threatn)"
"pen-1" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (1 mod t-number) t-number-list - 1) (item (1 mod t-number) threatn)"
"pen-2" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (2 mod t-number) t-number-list - 1) (item (2 mod t-number) threatn)"
"pen-3" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (3 mod t-number) t-number-list - 1) (item (3 mod t-number) threatn)"
"pen-4" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (4 mod t-number) t-number-list - 1) (item (4 mod t-number) threatn)"
"pen-5" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (5 mod t-number) t-number-list - 1) (item (5 mod t-number) threatn)"
"pen-6" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (6 mod t-number) t-number-list - 1) (item (6 mod t-number) threatn)"
"pen-7" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (7 mod t-number) t-number-list - 1) (item (7 mod t-number) threatn)"
"pen-8" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (8 mod t-number) t-number-list - 1) (item (8 mod t-number) threatn)"
"pen-9" 0.8 1 -16777216 true "" "plot-pen-reset\nplotxy (item (9 mod t-number) t-number-list - 1) (item (9 mod t-number) threatn)"

CHOOSER
195
220
333
265
t-tactic
t-tactic
"direct path" "center line" "edge line"
2

CHOOSER
10
50
162
95
sim-type
sim-type
"radial minefield" "vertical minefield" "fuws"
2

SLIDER
5
220
177
253
SensorQty
SensorQty
0
100
100
1
1
NIL
HORIZONTAL

SLIDER
5
265
177
298
MobileWeaponQty
MobileWeaponQty
0
20
6
1
1
NIL
HORIZONTAL

SLIDER
5
305
177
338
UUVQty
UUVQty
0
10
3
1
1
NIL
HORIZONTAL

MONITOR
1130
295
1237
340
Misses/Sim Run
Missed / sim-run
4
1
11

MONITOR
835
240
1240
285
Threat by Transiter
ThreatN
3
1
11

MONITOR
835
290
925
355
SIT
Threat1
4
1
16

SWITCH
285
50
390
83
FixedField
FixedField
1
1
-1000

MONITOR
930
310
987
355
+/-
Threat1SE
3
1
11

MONITOR
835
365
925
430
EC
ExpectedCasualties
4
1
16

TEXTBOX
930
295
1000
313
standard error
9
0.0
1

@#$#@#$#@
## WHAT IS IT?
This model simulates naval mine warefare (MIW) scenarios.

It was designed to contrast two mine warefare system architectures.
1) Traditional, explode-in-place mines deployed in fields
and 
2) Future undersea weapon systems (FUWS) consisting of distributed sensor fields and mobile weapons

## HOW TO USE IT
1) Select the "sim-type" from the drop down, this will determine the type of
weapon architecture modeled

2) Type in the desired number of Simulation Runs. This is the number of times the
model will generate a minefield and run targets ships through it. The fixed field toggle switch will generate the same field in each simulation run when set to on. When set to off, a new distribution of sensors and weapons will be generated for each simulation run.

3) If simulating a traditional minefield, 
   Use the slider bars to set the number of lines and total number of mines deployed.
   (i.e. 5 lines and 30 mines will result in 5 lines of 6 mines each)

   If simulating a FUWS,
   Use the slider bars to set the number of lines, sensors, mobile weapons, and UUVs. 
   In this architecture the sensors will be deployed in lines and the mobile weapons
   will be deployed on the UUVs

Additional sensor and weapon parameters may be available as the model is built out.

4) Use the slider bars to set the number of threat vessels in each scenario
Additional threat vessel parameters may be available as the model is built out.

5) Select the tactic the threat will use to transit the minefield
In center tactic, the theat vessels transit a 1ky channel in the center of the 
minefield area.

In edge tactic, the theat vessels transit a randomly selected, off center 1ky channel.

In direct tactic, the theat vessels transit directly to the destinaiton.

6) Initialize the scenario and Start the Simulations

## HOW IT WORKS
The model contains three breeds of turtles:

1) Hostile TargetsVessels attempt to transit the minefield using the selected tactic.

2) Sensors, represented as O icons, detect the presence of a target when transiting within a givin range. They pass data about the threat to weapons.

3) Weapons, represented as X icons, can explode and damage targets.


## THINGS TO NOTICE
1) Watch the threat profile.  This shows the risk that the nth transitor is delayed/
destroyed by the minefield. These risks are numerically shown beneath the graph.

2) The SIT is the threat to the first transiter in a series.

## THINGS TO TRY
1) Note how for the same minefield setup, the profile shape differs for different threat tactics. Also note how the profile changes for the same tactic when changing the minefield setup rules.

## EXTENDING THE MODEL
1) Detection and damage could be modeled as probabilistic functions of range vs fixed

2) Consider upgrading the FUWS targeting and decision logic. For example, is there a better way to select the weapon to fire? or when to fire? or where to aim?


## CREDITS AND REFERENCES
This model was built as part of the NPS MENTAL FOCUS Capstone Project.


Copyright 2015
Robert Patchin
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

boat
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 158 33 230 157 182 150 169 151 157 156
Polygon -7500403 true true 149 55 88 143 103 139 111 136 117 139 126 145 130 147 139 147 146 146 149 55

boat 2
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 157 54 175 79 174 96 185 102 178 112 194 124 196 131 190 139 192 146 211 151 216 154 157 154
Polygon -7500403 true true 150 74 146 91 139 99 143 114 141 123 137 126 131 129 132 139 142 136 126 142 119 147 148 147

boat 3
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6459832 true false 150 32 157 162
Polygon -13345367 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7500403 true true 158 37 172 45 188 59 202 79 217 109 220 130 218 147 204 156 158 156 161 142 170 123 170 102 169 88 165 62
Polygon -7500403 true true 149 66 142 78 139 96 141 111 146 139 148 147 110 147 113 131 118 106 126 71

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

cannon
true
0
Polygon -7500403 true true 165 0 165 15 180 150 195 165 195 180 180 195 165 225 135 225 120 195 105 180 105 165 120 150 135 15 135 0
Line -16777216 false 120 150 180 150
Line -16777216 false 120 195 180 195
Line -16777216 false 165 15 135 15
Polygon -16777216 false false 165 0 135 0 135 15 120 150 105 165 105 180 120 195 135 225 165 225 180 195 195 180 195 165 180 150 165 15

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sail
true
0
Polygon -7500403 true true 75 225 225 225 255 195 45 195 75 225
Polygon -7500403 true true 150 30 75 150 150 150 150 45 150 30 165 30 165 195 150 195 150 150

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

surf_hostile
false
2
Line -955883 true 30 150 150 270
Line -955883 true 30 150 150 30
Line -955883 true 150 30 270 150
Line -955883 true 270 150 150 270
Rectangle -955883 true true 135 135 165 165

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tile stones
false
0
Polygon -7500403 true true 0 240 45 195 75 180 90 165 90 135 45 120 0 135
Polygon -7500403 true true 300 240 285 210 270 180 270 150 300 135 300 225
Polygon -7500403 true true 225 300 240 270 270 255 285 255 300 285 300 300
Polygon -7500403 true true 0 285 30 300 0 300
Polygon -7500403 true true 225 0 210 15 210 30 255 60 285 45 300 30 300 0
Polygon -7500403 true true 0 30 30 0 0 0
Polygon -7500403 true true 15 30 75 0 180 0 195 30 225 60 210 90 135 60 45 60
Polygon -7500403 true true 0 105 30 105 75 120 105 105 90 75 45 75 0 60
Polygon -7500403 true true 300 60 240 75 255 105 285 120 300 105
Polygon -7500403 true true 120 75 120 105 105 135 105 165 165 150 240 150 255 135 240 105 210 105 180 90 150 75
Polygon -7500403 true true 75 300 135 285 195 300
Polygon -7500403 true true 30 285 75 285 120 270 150 270 150 210 90 195 60 210 15 255
Polygon -7500403 true true 180 285 240 255 255 225 255 195 240 165 195 165 150 165 135 195 165 210 165 255

torpedo
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Rectangle -16777216 true false 0 0 315 150
Rectangle -7500403 true true 135 135 150 210
Rectangle -7500403 true true 90 135 195 150

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Sweep" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <enumeratedValueSet variable="MineQty">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t-number">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sim-type">
      <value value="&quot;fuws&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="UUVQty">
      <value value="1"/>
      <value value="2"/>
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="sim-runs">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Lines">
      <value value="2"/>
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t-tactic">
      <value value="&quot;edge line&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="SensorQty">
      <value value="50"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MobileWeaponQty">
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="FixedField">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t-speed">
      <value value="12"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
1
@#$#@#$#@
