;;;;;;;;;;;;;;;;;;
;; Declarations ;;
;;;;;;;;;;;;;;;;;;

globals
[
  run-number
  ;; counter used to keep the model running for a little
  ;; while after the last turtle gets infected
  delay
  
]

breed [ humans human ]


humans-own [ 
  energy  ;; human energy level
  affability ;; how friendly the human is (helps determine whether they will group with others)
  infected? ;; boolean - are they infected with the zombie virus or not?
  aggression ;; measures likelihood of fighting zombies
  fighting? ;; boolean - are they currently fighting or not
  ]

patches-own [resource-level num-humans-on-patch is-growing-food? ]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

;; clears the plot too
to setup-clear
  clear-all
  set run-number 1
  setup-world
end

;; note that the plot is not cleared so that data
;; can be collected across runs
to setup-keep
  clear-turtles
  clear-patches
  set run-number run-number + 1
  setup-world
end

;; scatters food patches across the world in 3/4 quadrants. The patches also have varying food density levels
to setup-world
  set-default-shape humans "person"
  ask patches [
    ifelse random-float 100 < food-density [ 
      ifelse pxcor > 0 or pycor < 0 
        [set pcolor yellow
          set is-growing-food? true]
        [set pcolor gray ]]
    [set pcolor gray] 
  ] 
  set delay 0

  create-some-humans
  create-some-zombies
  reset-ticks
end


;; creates humans and gives them certain properties 
to create-some-humans
  create-humans num-humans
  [
    setxy random-pxcor random-pycor   ;; put humans on patch centers
    set color black
    set heading 90 * random 4
    set energy 20 
    set infected? false 
    set fighting? false
    set affability 5
    set aggression 5
  ]
end

;; for infect button - user initially infects a few humans with the zombie virus
to create-some-zombies 
  create-humans num-zombies [
   setxy random-pxcor random-pycor 
   set heading 90 * random 4
    set infected? true 
    set color green
    set affability 0
    set energy 30
    set aggression 10 ]               
end



;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

to go
  humans-wander 
  increase-aggression 
  reproduce
  fight 
  ;; Humans will cluster together if they have the same affability level (if they are nice to each other, they form links/
  ;; groups) and will avoid each other if one person is less nice than the other. 
  if cluster? [
    ask humans with [not infected?] [
      let x self
      ask humans-on neighbors [
        if not infected? [
          let aff-h [affability] of x
          if aff-h = affability [
            create-link-with x
            ask links [ set color blue] 
            set affability affability + 1
          ]
          if affability < aff-h [
            avoid
          ]
        ]
      ]
    ]
  ]
  
  eat-food 
  run-away 
  infect-human
  human-death-natural  
  fight-color

  regrow-resources 

  set num-zombies count humans with [infected?]
  set num-humans count humans with [not infected?]
  
  if num-zombies = 0 or num-humans = 0 [
    stop
  ]
  tick
end



;; controls the general motion of the humans
to humans-wander
  ask humans[ 
    rt (random 4) * 90 
    fd 1
    if not infected? [
    set energy energy - 0.1 ]]
end




;;;;;;;;;;;;;;;;;;;;;;;
;;; Human Behaviors ;;;
;;;;;;;;;;;;;;;;;;;;;;;

;; If a human's energy runs to 0, they die automatically. 
to human-death-natural 
  ask humans [ 
  if not infected? and (energy <= 0) [
    die 
    set num-humans num-humans - 1]     
  ]                 
end


;; Humans turn around and avoid each other if they encounter each other. Used in the clustering behavior. 
to avoid 
  let candidates patches in-radius 1 with [ not any? humans-here with [ infected? ] ]
  ifelse any? candidates
    [ face one-of candidates ]
    [ rt (random 4) * 90 ]
end

;; Humans will run-away from a zombie by moving a farther distance (essentially a sort of jump away from the zombies
;; if they are nearby)
to run-away
  ask humans with [infected?] [
    ask humans-on neighbors [
      if not infected? [
        rt (random 4) * 90
        fd 2
      ]
    ]
  ]
end

;; If humans are hungry, they get cranky and more aggressive. 
to increase-aggression
  ask humans with [energy <= 20 and not infected?] [
    set aggression aggression + 0.5
  ]
end

;; If a human is strong enough to fight zombies, he calls over the people he is linked to in order to kill the zombie
;; with help. Humans also get more aggressive when gathering for a fight, like they are feeding off each other's energy. 
to gather-for-fight
  ask humans with [not infected?] [
    let x myself
      ask link-neighbors [
          move-to x
          set fighting? true
          ask humans-here with [infected?] [ 
            set energy energy - num-humans
            if energy <= 0 [die]]
        ]
    set aggression aggression + 1
  ]
end

;; makes it easier to see who is fighting
to fight-color
  ask humans with [not infected?] [
   if fighting? 
      [set color red]
  ]
end

;; Fight behavior for humans - gathers up friends and tries to kill a zombie.

;;and energy >= 30 and aggression >= 6
to fight
  ask humans with [not infected? and energy >= 30 and aggression >= 6] [
    if any? humans-here with [infected?] [
      ifelse cluster?
        [gather-for-fight]
        [set fighting? true 
          ask humans-here with [infected?] [
            die
          ]
        ]
    ]
  ]
end

;; In order to keep their energy up, humans have to keep eating food. 
to eat-food
  ask humans with [not infected? ] [
    if pcolor = yellow [ 
      set pcolor gray    
      set energy energy + energy-gain-from-food ;; increment human energy level
    ]
  ]
end

to reproduce
  if reproduce? [
  ask humans with [not infected?] [
    let x self
    if energy >= 100 [
      set energy energy - 50  ;; reproduction transfers energy
      hatch 1 [ 
        set energy 20 
        set affability [affability] of x
        set aggression [aggression] of x
        ] ;; to the new agent
    ]
  ]
  ]
end

;;; Zombie procedures

;; Zombies have to eat too, but instead of food, they feast on fresh human brains
to eat-brains
  ask humans with [infected?] [
    ask other humans-here with [not infected?] [
      if energy < 10 [
        die 
      ]
      set energy energy + 3
    ]
  ]
end

;; If a human encounters a zombie and their energy level is between a certain threshold, then the human gets infected by 
;; the virus
to infect-human 
  ask humans with [infected?] [ 
    ask other humans-here [
      if not infected? and (energy < 15 and energy > 10) [
        set infected? true 
        set color green
        set affability 0
        set energy 30 
        ask my-links [die] ]
    ] 
  ]         
end


;;;;;;;;;;;;;;;;;;;;;;
;; Patch procedures ;;
;;;;;;;;;;;;;;;;;;;;;;
;to resource-level-energy
;  set energy resource-level
;end
;
;;; recolor the food to indicate how much has been eaten
;to recolor-resources
;set pcolor scale-color yellow (10 - resource-level) -10 20 
;end

;; regrow the food
to regrow-resources
  ask patches with [pcolor = gray] [
    if is-growing-food? != 0 [
      if (random-float 100 < resource-replenish-rate) and (pxcor > 0 or pycor < 0) [
        set pcolor yellow 
      ] 
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
367
10
806
470
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

SLIDER
8
21
211
54
num-humans
num-humans
0
100
10
1
1
NIL
HORIZONTAL

BUTTON
12
260
115
293
NIL
setup-clear
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
125
259
227
292
NIL
setup-keep
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
85
307
148
340
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
9
72
211
105
energy-gain-from-food
energy-gain-from-food
0
100
4
1
1
NIL
HORIZONTAL

SLIDER
9
122
211
155
resource-replenish-rate
resource-replenish-rate
0
100
15
1
1
NIL
HORIZONTAL

SLIDER
9
170
212
203
food-density
food-density
0
100
50
1
1
NIL
HORIZONTAL

PLOT
7
352
343
573
Zombie Virus Growth
time
number
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"zombies" 1.0 0 -7500403 true "" "plot num-zombies"
"humans" 1.0 0 -2674135 true "" "plot num-humans"

SWITCH
231
21
358
54
cluster?
cluster?
1
1
-1000

MONITOR
397
500
497
545
NIL
num-zombies
17
1
11

MONITOR
511
501
608
546
NIL
num-humans
17
1
11

SLIDER
10
216
212
249
num-zombies
num-zombies
0
100
0
1
1
NIL
HORIZONTAL

SWITCH
231
72
358
105
reproduce?
reproduce?
1
1
-1000

@#$#@#$#@
## WHAT IS IT?

This is a zombie apocalypse model intended to show human behavior in the face of extreme adversity. It is essentially a survivalist situation in which people can fight zombies and eat the resources around them in order to attempt to stay alive. 

## HOW IT WORKS

AGENTS

Humans

- Initially I start out having levels of ENERGY AFFABILITY AGGRESSION and have checks to see whether or not I am infected with the zombie virus (INFECTED?) and/ or whether or not I am fighting (FIGHTING?)

- On every clock-tick, I move in a random direction and check a few things: 

	- If I can cluster and if my neighbors have the same affability level as me, we 		become friends and link together. If not, I AVOID them. 

	- If there is a zombie in my vicinity, I RUN-AWAY from it

	- If there is a zombie on the same patch as me, I check to see if I can fight it 		based on my aggression levels and energy. If I can and I can cluster with other 		humans, I call my friends over with GATHER-FOR-FIGHT and we attempt to kill the 		zombie together. Some of us may die or become infected. If I can, but cannot 			cluster with other humans, I attempt to fight off the zombie on my own resulting 		in either my infection, death, or the death of the zombie. If I can't fight the 		zombie, I either become infected or die. 

	- If I am fighting, I turn red momentarily. 

	- If my energy drops below a certain level (aka, I get hungry) my aggression 			levels start to increase.  

	- If my energy hits 0, I die.

	- If I am on a yellow food patch that is not depleted, I eat and gain energy. 
	
	- If I have enough energy to reproduce and the reproduce? toggle is on, then I 			make a child that has the same affability and aggression levels as me. 

Zombies

- Initially, the user assigns me to have the zombie virus. This means I have no AFFABILITY, a newly set ENERGY level, and a newly set AGGRESSION level. 

- On every clock tick, I move in a random direction and check a few things:

	- If I am on the same patch as a human and it has a certain energy level, I 			either infect it, or eat its brains for energy, or get killed by the human(s) 			depending on the situation. 

Patches 

- Initally, I am assigned to either be a background patch, or a patch with food on it.
 
- There are only food patches in three quadrants of the environment. 

- On every clock tick: 

	- If there is a human on me, they eat food and deplete the store that I have, but 	I regrow food according to a RESOURCE-REPLENISH-RATE sometimes as well. 


## HOW TO USE IT

NUM-HUMANS slider : sets the number of humans in the simulation

NUM-ZOMBIES slider : sets the number of humans with the zombie virus in the simulation

ENERGY-GAIN-FROM-FOOD slider: sets the amount of energy a human gains by eating food 

RESOURCE-REPLENISH-RATE slider: sets the rate at which the food in the model grows back

FOOD-DENSITY slider: sets the density of the distribution of food patches throughout the model

SETUP-CLEAR button: sets up the model and clears all plots

SETUP-KEEP button: sets up the model but doesn't clear plots from previous runs

CLUSTER? toggle: allows the user to toggle whether or not the humans will make connections. 

REPRODUCE? toggle: allows the user to toggle whether or not the humans will reproduce or not during the run. 

GO (continuous) button: runs the model 

ZOMBIE-VIRUS-GROWTH plot: shows the growth of the zombie virus over time

## THINGS TO NOTICE

Regional grouping of humans is interesting. The clustering becomes tighter as time goes on. The number of people in a cluster also increase. Sometimes there are a few large clusters or a bunch of little ones. It's interesting to see under which conditions these clusters of humans form. 

## THINGS TO TRY

Toggle the clustering behavior and see what happens. 

Vary the food density in the model. Can you make it so that the zombie virus is eradicated? 

## EXTENDING THE MODEL

In order to extend the model, try adding different types of resources that different types of people use could produce potentially interesting results (e.g. only aggressive people can use weapons, and fight zombies, while submissive people gather food for the "tribes") 

It might also be interesting to see whether or not different tribes of people form and whether they barter their resources in order to stay alive. 

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

Virus Model
Wolf-sheep predation model 

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Z-Energy-Gain (Cluster off Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="1"/>
      <value value="2"/>
      <value value="4"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Energy-Gain (Cluster off Repro on)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="1"/>
      <value value="2"/>
      <value value="4"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Energy-Gain (Cluster on Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="1"/>
      <value value="2"/>
      <value value="4"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Energy-Gain (Cluster on Repro on)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="1"/>
      <value value="2"/>
      <value value="4"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Replenish Rate (Cluster off Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Replenish Rate (Cluster off Repro on)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Replenish Rate (Cluster on Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Replenish Rate (Cluster on Repro on)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Num-Humans (Cluster off Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Num-Humans (Cluster off Repro on)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Num-Humans (Cluster on Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Num-Humans (Cluster on Repro on)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
      <value value="4"/>
      <value value="6"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Num-Zombies (Cluster off Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Num-Zombies (Cluster off Repro on)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Num-Zombies (Cluster on Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Z-Num-Zombies (Cluster on Repro on)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Energy-Gain (Cluster off Repro off)" repetitions="1" runMetricsEveryStep="true">
    <setup>setup-keep</setup>
    <go>go</go>
    <metric>count humans with [infected?]</metric>
    <metric>count humans with [not infected?]</metric>
    <enumeratedValueSet variable="reproduce?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-gain-from-food">
      <value value="1"/>
      <value value="2"/>
      <value value="4"/>
      <value value="8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="resource-replenish-rate">
      <value value="15"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cluster?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-humans">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="food-density">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-zombies">
      <value value="2"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
