breed [care-givers care-giver]
breed [beds bed]
breed [patients patient]


globals [
  overflow ;;holds overflow count
  patient-arrival-chance ;; List containing probability of arrival for each hour
]

patients-own [initial-apache apache acuity TTL LOS stable-ticks my-caregiver recieving-care? care-count death-prob bed-number ]
beds-own [occupied?]
care-givers-own [my-patients treat-count acuity-score]

to setup
  ca
  
  ;;;;;;;;;;;;;;;
  ;;Setup Layout
  ;;;;;;;;;;;;;;;
  
  ask patches with [pycor > 10 and pycor < 40] [set pcolor white]
  ask patches with [pycor < 10 and pxcor < 20] [set pcolor cyan]
  ask patches with [pycor < 10 and pxcor > 20] [set pcolor cyan]
  
  
  ;;;;;;;;;;;;
  ;;Setup Beds
  ;;;;;;;;;;;;
  
  ask patches with [pycor = 20 or pycor = 30 and pxcor > 5 and pxcor < 35 and pxcor mod 3 = 0] [set pcolor 99]
  ask patches with [pycor = 21 or pycor = 31 and pxcor > 5 and pxcor < 35 and pxcor mod 3 = 0] [set pcolor 99.1]
  set-default-shape beds "square"
  ask patches with [pcolor = 99] [sprout-beds 1 [set size 1 set color 99 set heading 0]]
  
  ;;;;;;;;;;;;;;;;;;;;
  ;;Setup Care-Givers
  ;;;;;;;;;;;;;;;;;;;;
  
  set-default-shape care-givers "person"
  ask n-of n patches with [pcolor = white] [sprout-care-givers 1 [
    set size 2
    set color blue
    set my-patients[]
  ]]
  set-default-shape patients "person"
  
  
  
  ;;;;;;;;;;;;;;;;;;;;;;;;
  ;;Setup Initial Patients
  ;;;;;;;;;;;;;;;;;;;;;;;;
  let initial-patient random-normal initial-patient-mean 2 ;; normal intitializtion of how many patients there are. Initial conditions likely affect system performance.
  if initial-patient > count beds [set initial-patient 20] 
  ask beds [set occupied? false]
  ask n-of initial-patient beds [hatch-patients 1 [set color red set size 2 set bed-number [who] of myself  ] set occupied? true]
  ask patients [ 
    set-apache                                 ;; based off of Naved et al. (2011) - APACHE-II Score Correlation with Mortality and Length of Stay in an ICU
    set-ttl 
    set my-caregiver nobody
    set recieving-care? false
    set-death-prob
    set-patient-color 
  ] 
  assign-patient
  ;;Patient Arrival Probability List
  set patient-arrival-chance [ 12 12 10 10 11 15 18 15 15 13 11 10 ] ;; Assuming that more patients come in the middle of the day. in two hour blocks 
  reset-ticks
end






to go
  ask beds [
    if not any? patients-here [set occupied? false]  
  ]
  patient-inflow
  assign-patient
  treat-patients
  ask patients [advance-health] ; patients either improve or get sicker depending on quality of care
  ask patients [if random-float 1 < death-prob [remove-patient] ]       ;;patients can randomly die according to their accuity
  ask care-givers [clean-list]
  discharge-patients 
  ask care-givers [clean-list]
  ask patients [set LOS LOS + 1]
  ask patients [set-patient-color] ;; to visualize how sick patients are
  
  tick
end






;; admits patients if there are available beds. Patients arrive based on the defined patient arrival probability list with added noise
to patient-inflow
  if random 100 < random-normal 0 2 + item (remainder ticks 12) patient-arrival-chance [
    ifelse any? beds with [occupied? = false]  [admit-patient] [overflow-patient] ] ;; each tick, patients either flow in or are overflowed
end


;; Creates the patient on an open bed
to admit-patient
  ask one-of beds with [occupied? = false] [hatch-patients 1 [
    set color red
    set size 2
    set-apache                                 
    set my-caregiver nobody
    set recieving-care? false
    set-apache
    set-ttl
    set-death-prob
    set bed-number [who] of myself
    set-patient-color
    
  ]
    set occupied? true]
  ask care-givers [clean-list]
end

;;Increases the overflow count variable
to overflow-patient
  set overflow overflow + 1
end

;; This procedure holds the logic for treating patients
to treat-patients
  ;; Aiken et al, 2002: increase in one patient to nurses assignment increases mortalilty rate 7%
  
  
  let dedicated-care-givers care-givers with [length my-patients = 1 and item 0 my-patients != nobody] ;; select care-givers that only have one patient
  ask dedicated-care-givers [
    ifelse random 100 < care-time [
      move-to item 0 my-patients
      fd 2
      set treat-count treat-count + 1
      ask item 0 my-patients [
        set recieving-care? true
        set care-count care-count + 1 ;; increasing by ten with each tick?
        set recieving-care? false
      ]
    ]
    [
      ask dedicated-care-givers [move-to one-of patches with [pcolor = cyan]] 
    ]
  ]
  
  
  
  
  
  let multi-care-givers care-givers with [length my-patients > 1]
  ask multi-care-givers [
    ifelse random 2 = 0 [
      ifelse random 100 < care-time [
        move-to item 0 my-patients
        fd 2
        ask item 0 my-patients [
          set recieving-care? true
          set care-count care-count + 1 ;; increasing by ten with each tick?
          set recieving-care? false
        ] ] 
      [move-to one-of patches with [pcolor = cyan]]
    ][
    ifelse random 100 < care-time [
      if item 1 my-patients != nobody [ ;; ensure that the care-giver has a second patient
        move-to item 1 my-patients
        fd 2
        ask item 1 my-patients [
          set recieving-care? true 
          set care-count care-count + 1
          set recieving-care? false
        ]
      ]
    ]
    [move-to one-of patches with [pcolor = cyan]
    ]
    ] 
  ]
end


;;Patients that are stable are discharged  starting at 7am
to discharge-patients
  ;; every morning at 7 am patients are discharged
  ifelse strict-discharge? [
  if remainder ticks 24 = 7 [
    ask patients with [ stable-ticks > stable-hours-needed ] [
      remove-patient
    ] ]]
  [ask patients with [ stable-ticks > stable-hours-needed ] [
      remove-patient]
  
  ]
end

;; This procedure abstracts quality of care into how often the care-giver treats the patient. 
;; If the quality of care is high the patient is more likely to improve their health.

to advance-health
  if LOS > 0 [
    let quality-of-care (care-count / LOS) + random-normal 0 .1              ;; patient-centric quality of care plus a little noise
    if random-float 1 < quality-of-care [improve-health] ;; 
    
    
    if random-float 1 < 1 - quality-of-care [worsen-health] ;; includes error rates like nonsocomial infections, falls, HAI, and others
    
                                                            ;;If patient is low acuity, they add to a stability counter which determines if they are ready for discharge
    ifelse apache < stable-apache [set stable-ticks stable-ticks + 1] [set stable-ticks 0]]
  set-death-prob
end




;;;;;;;;;;;;;;;;;;;;               
;; Helper Methods ;; Methods that are called throughout the model
;;;;;;;;;;;;;;;;;;;;

;; a patient procedure to remove the patient from the ICU
to remove-patient
  if my-caregiver != nobody [ 
    
    ask my-caregiver [ set my-patients remove myself my-patients]
  ]
  die
  ask beds [
    if not any? patients-here [set occupied? false] 
  ]
  
end


;; decreaing apache score is equivalent to reducing acuity
to improve-health
  set apache apache - random-normal .2 .05
end

;; increasing apache is equivalent to increasing patient acuity
to worsen-health
  set apache apache + random-normal .2 .05
end

;;larger number the darker the color: 
to set-patient-color
  if acuity = "high" [
    set color scale-color red apache 60 25
  ]
  
  if acuity = "medium" [
    set color scale-color yellow apache 50 0
  ]
  
  if acuity = "low" [
    set color scale-color green apache 20 -10 
  ]
end


;; sets the death probability for each patient group which defines how likely it is that a patient will die during that tick.
to set-death-prob
  if acuity = "low" [ 
    set death-prob abs ( random-normal .1 .05 / ttl) ;; based on Naved Et al. results .1 is the mortality % divded by the number of days the patient is expected to stay in the ICU
  ]
  
  if acuity = "medium" [
    set death-prob abs (random-normal .4 .25 / ttl)
  ]
  
  if acuity = "high" [
    set death-prob abs ( random-normal .85 .05 / ttl )
  ]
end



;; This procedure sets the apache score for each patient
;; Imputed from Naved et al. (2011) - APACHE-II Score Correlation with Mortality and Length of Stay in an ICU
to set-apache
  if initial-apache < 10 [set initial-apache random-normal 25 12 set apache initial-apache]
  group-apache
  if initial-apache < 10 [set-apache]    ; makes sure no patient has a negative apache or very small apache
  
end


;;groups each patient based on the apache score.
to group-apache
  if apache <= 10 [ set acuity "low"] ;; 10% death
  if apache >= 30 [ set acuity "high"]  ;; 85% death
  if apache > 10 and apache < 30 [ set acuity "medium"] ;; but around 40-50% death More randomness with these patients
end

;; sets the Time to Live of each patient
to set-ttl
  set ttl abs (random-normal 69 93 )           ;; this is length of stay data from literature                
end


;;;;;;;;;;;;;;;;;;;;;
;;;Patient Assignment
;;;;;;;;;;;;;;;;;;;;;
to assign-patient
  
 
  let sick-patients patients with [ my-caregiver = nobody and acuity = "high"] ;; find high acuity patients without a care-giver 
  let dedicated-caregivers care-givers with [ length my-patients = 0] ;; find care-givers that don't have yet assignment
  ask dedicated-caregivers[
    set my-patients fput one-of sick-patients my-patients ;; adding one of the sick patients to the front of my-patients list
    if item 0 my-patients != nobody[                      ;; accesing first item in list
      ask item 0 my-patients [ set my-caregiver myself]   ;; I ask my patient to set their caregiver to me.
    ]
  ]
  
  
  ;; Assigns remaining care-givers that do not have any patients
  let remaining-patients patients with [(acuity = "medium" or acuity = "low") and my-caregiver = nobody]
  let remaining-empty-caregivers care-givers with [length my-patients = 0 or my-patients = [nobody]] ;; assign rest of caregivers that don't have a patient first
  if any? remaining-patients and any? remaining-empty-caregivers [
    ask remaining-empty-caregivers [
      set my-patients fput one-of remaining-patients my-patients
      ask item 0 my-patients [set my-caregiver myself]
    ]
  ]
  
  
  ;this assignment heuristic makes sure that no patient is left un-cared for
  let leftover-patients patients with [my-caregiver = [] or my-caregiver = nobody ]
  ask care-givers [
    clean-list
    set acuity-score 0
    foreach my-patients [set acuity-score acuity-score + [apache] of ?] ; add up the apache scores of your patients
  ]
  
  ;; asks the neccesarry amount of caregivers with the lowest agregate acuity to take care of the left over patients
  ask min-n-of (count leftover-patients) care-givers [acuity-score]  [    
    set my-patients fput one-of leftover-patients my-patients
    ask item 0 my-patients [set my-caregiver myself]
  ] 
end


;;this makes sure that the my-patient list doesn't have nobody in the list.
;;this code is called throughout the model
to clean-list
  set my-patients remove nobody my-patients
end








;;;;;;;;;;;;;;;           
;;Reporters 
;;;;;;;;;;;;;;;



;; census is taken at 7am every morning
to-report census
  report count patients
  
end

to-report mean-apache
  report mean [apache] of patients
end

to-report bed-utilization
  report count beds with [occupied? = true] / count beds
end

to-report caregiver-patient-ratio
  report precision (count care-givers / count patients) 2
end


;; add in monitors if need be. These measures are useful for adding on new features and debugging
to-report empty-patients
  report count patients with [my-caregiver = [] or my-caregiver = nobody ]
end

to-report empty-caregivers
  report count care-givers with [my-patients = []]
end

@#$#@#$#@
GRAPHICS-WINDOW
561
24
1022
506
-1
-1
11.0
1
10
1
1
1
0
0
0
1
0
40
0
40
1
1
1
hours
30.0

BUTTON
30
22
96
55
NIL
setup
NIL
1
T
OBSERVER
NIL
S
NIL
NIL
1

PLOT
315
362
515
512
Census
Hours
Census
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot census"

MONITOR
306
22
410
67
NIL
bed-utilization
2
1
11

SLIDER
121
22
254
55
n
n
0
count beds
20
1
1
Caregivers
HORIZONTAL

PLOT
16
363
216
513
Cargiver Patient Ratio
Time
Ratio
0.0
10.0
0.0
1.2
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot caregiver-patient-ratio"

BUTTON
32
71
95
104
NIL
go
T
1
T
OBSERVER
NIL
G
NIL
NIL
1

PLOT
12
542
287
692
Mean Length of Stay 
Hours
Mean LOS
0.0
60.0
0.0
10.0
true
true
"" ""
PENS
"LOS" 1.0 0 -2674135 true "" "plot mean [los] of patients"

PLOT
317
542
517
692
Apache Distribution
Apache(interval10)
#ofpatients
0.0
40.0
0.0
10.0
true
false
"" ""
PENS
"default" 10.0 1 -16777216 true "" "histogram [apache] of patients"

PLOT
314
152
515
343
Overflow
hours
patients overflowed
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot overflow"

MONITOR
340
80
496
125
NIL
caregiver-patient-ratio
2
1
11

SLIDER
9
174
202
207
stable-hours-needed
stable-hours-needed
6
100
12
6
1
hours
HORIZONTAL

MONITOR
430
23
525
68
Days Elapsed
ticks / 24
1
1
11

SLIDER
10
125
130
158
care-time
care-time
0
100
100
1
1
%
HORIZONTAL

SWITCH
12
223
175
256
strict-discharge?
strict-discharge?
1
1
-1000

SLIDER
115
76
304
109
initial-patient-mean
initial-patient-mean
10
18
15
1
1
patients
HORIZONTAL

SLIDER
10
268
198
301
stable-apache
stable-apache
3
15
10
1
1
apache
HORIZONTAL

TEXTBOX
120
59
270
77
Modify These before setup
11
0.0
1

TEXTBOX
15
110
165
128
Modify these during go
11
0.0
1

@#$#@#$#@








## WHAT IS IT?

ICU care is costly. Lives are stake. Dollars on the line; some studies show that 2 pennies on every dollar spent in this country are flowing to ICU costs. This model attempts to explore the basic mechanisms that drive patient flow through the ICU. By understanding how patients flow and overflow in the ICU, we may be able to devise better strategies to smooth the flow, bring down costs, and save lives.

You will discover how the number of caregivers, the amount of time that caregivers spend caring for their patients and various discharge standards affect the flow of patients through the ICU. More importantly, this model serves as an object to think with and build with when thinking about patient flow.   

## HOW IT WORKS

There are two important agents: patients and care-givers. Patients come into the ICU with varying degrees of sickness rated on the Acute Physiology and Chronic Health Evaluation (APACHE) score. The higher the apache, the sicker the patient.

### Press Setup
1. Create the ICU layout including beds and care-giver room
2. Create initial patients 
3. Create initial care-givers
4. Care-givers are assigned to patients in order of acuity
	ex. Sick patients (apache > 30) are assigned first...

### Press Go
1. Patients flow in to the ICU with probability based on the hour of the day
2. The new patients are assigned care-givers
3. Care-givers treat their patients. If I have two patients, I split time between each of them.
4. Patients either improve their health or decrease their health based on the quality of care.
5. Depending on how sick I am, there is some chance that I die
6. If I have been healthy for set amount of hours or days, I am ready for discharge at 7am



## HOW TO USE IT

Wash before eating, do not breathe this. But really the fumes are toxic. Feel free to push buttons. Learn about them below:

### Interface Items
n: number of caregivers
initial patient mean: allows adjustment of the mean of how many initial patients are in the icu. 
care-time%: set the percent of the day that caregivers spend on directly providing care.
stable hours needed: set the threshold for how many hours a patient needs to be stable for before being discharged.
strict discharge?: In the on position, patients that are ready for discharge leave at 7am. In the off position, they leave as soon as they hit the threshold set above.
stable-apche: set the apache score to be considered stable for that hour. For example, if stable apache is set to 10 and your apache is 9, you will be considered stable for that hour.

## THINGS TO NOTICE

Visually, you should notice how the colors of the patients are changing. High-risk patients (red), medium-risk patients (yellow), and low risk patients (green) get darker the sicker they get within their category. 

Also notice how the caregivers move to the cyan rooms on the bottom of the world. This is to help visualize how caregivers move in and out of treating their patients. This could be very interesting if the model extended into geospatial constraints. It also helps to visualize how changing the care-time % can change where the care givers are located.

You'll often see, if their are many care-givers to the amount of patients, huddles around a newly admitted sick patient. The care givers typically huddle around until the patient is healed quickly or dies from severe illness. Do you agree with the assumption that the less busy the ICU is, the more likely caregivers will be to work together to heal somebody? How does this affect the flow dynamics? Maybe it makes sense to have standby teams of high performing teams to stablize a patient and then hand off to few caregivers once stablized?

## THINGS TO TRY

Pay attention to how changing the number of care-givers affects the number of patient overflows. Why might this be the case based on how the model functions? How does varying care-time affect the model and why? Do you agree with the assumption that increasing care-time increases the chance of recovery? 

Move around stable-hours-needed and see how it affects the model. Try clicking on a patient and caregiver interaction to see how acuity score changes over time. This gives you a feel for the micro behavior of the model. Compare this to the mean length of stay chart.

You can also try decreasing stable-apache. Why might this parameter have a critical value?

## EXTENDING THE MODEL

This model can be extended in many ways as was the goal of this model. View the list below:

1. Add agents (different nurses) that have a differential ability to treat patients (experience, training, education, etc)
2. Adapt the model to be more specific to a given ICU. Allow input of historical ICU census data, number of beds, number of caregivers
3. Add in geospatial constraints. Allow custom room-layouts and model travel time.
4. Deeper verification of the model with relevant literature.
5. Model quality of care with finer resolution. Add in things like equipment, administrative support, etc. 
6. Implmenent and test different flow smoothing strategies. Either from literature or based off intuition. For example, how should patients be optimally discharged?


## NETLOGO FEATURES

Most of the code is standard NetLogo. Most of the interesting and or tricky components were working with lists. Using the remainder primitive is very useful to turn ticks into hours of the day which allows hour by hour based model dyamics. I used this feature a few times throughout the model, but notably when setting up the patient arrival distribution.

## RELATED MODELS

The base layout for this model was taken from the model published below. Unfortunately, the model below does not follow the ABM framework. This made analyzing the model for validity difficult. As a result, none of the logic was used in the creation of this model other than the physical layout.

Leykum LK, Kumar, P, Parchman ML, McDaniel RR, Lanham HJ, Agar, M. Use of Agent-Based Model to Understand Clinical Systems Journal of Artificial Societies and Social Simulation 2012 Jan;15(3)


## CREDITS AND REFERENCES

Thanks to Northwestern Memorial Hospital Medical Intensive Care Unit for providing reference data as well as observation and interivew opportunities. Also thank you to Arthur Hjorth, Uri Wilensky, Donna Woods, and Jacob Kelter for feedback and advice on the development of this model.

The model can be found on the NetLogo Modeling Commons: http://modelingcommons.org/browse/one_model/3662#model_tabs_browse_discuss
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
