;;A tick is one minute of activity, there are 3600 ticks in a day. 
;;The color of the background represents how fast the body is using energy, i.e. metabolism

;; In this model, the body consumes calories and attempts to consume calories based on the body's needs

;; There are 1440 ticks in a day, each tick represents a minute

;;The 5 macronutrients. One agent represents a calorie of the nutrient
breed [fats fat]
breed [carbohydrates carbohydrate]
breed [proteins protein]
breed [fibers fiber]
breed [water single-water]

;;Persistent agents, body stores
breed [body-fats a-body-fat]  ;; One body fat agent represents 500 calories of body fat

;;A calorie burner is an agent that seeks out energy to accomplish a task. Sleeping and exercise burn calories on top of the BMR
breed [calorie-burners a-calorie-burner]

calorie-burners-own
[
  calories-left-to-burn
]

globals
[ 
  basal-metabolic-rate  ;;Calories/Day at rest
  height                ;;Height in meters
  base-weight           ;;Weight without body fat
  weight                ;;Weight in Kilograms
  age                   ;;Age in years
  
  body-fat-calories              ;;The calories of body fat currently in the body
  
  ticks-per-day
  yesterdays-calorie-balance
  calories-burned-today
  calories-consumed-today
  daily-calorie-intake
  
  body-fat-burned-today
  nutrients-burned-today
  %-nutrients-burned-during-exercise
]

to setup
  clear-all
  
  set weight ( initial-weight-in-lbs * 0.4536 )
  set base-weight ( weight ) * ( 1 - initial-body-fat-% )  ;;The base weight is the weight of the individual without any body fat
  set height ( initial-height-in-inches * 2.54)
  set age initial-age
  
  set ticks-per-day (1440 / minutes-per-tick)
  
  
  ;;Create the initial body fat
  set body-fat-calories ( 7716 * ( weight * initial-body-fat-%) ) ;;7716 calories per kg of body fat
  
  set-default-shape turtles "circle"
  ask patches [
    set pcolor white
  ]
  ask patches with [ pxcor = 0 ]
  [
    set pcolor black
  ]
  
  update-BMR
   
end

to update-BMR
  ;;In this model, BMR is a dynamic variable
  
  ;;Mifflin Equation (1990)
  set basal-metabolic-rate ( 10.0 * ( weight ) + 6.25 * height - 5 * age )
  ifelse(male?) 
  [ set basal-metabolic-rate ( basal-metabolic-rate + 5) ]
  [ set basal-metabolic-rate ( basal-metabolic-rate - 161) ]
  ;;Katch-McArdle Formula
  
end
  
to go
  ;;If there are <= 0 body fat calories, the subject has died, so stop the simulation
  if ( body-fat-calories <= 0 ) [ stop ] 
  if ( ticks mod ticks-per-day = 0 ) 
  [
    ;;Update daily counters
     if( nutrients-burned-today + body-fat-burned-today != 0) 
     [ 
       set %-nutrients-burned-during-exercise (nutrients-burned-today)/(nutrients-burned-today + body-fat-burned-today)
     ]
     set yesterdays-calorie-balance (calories-consumed-today - calories-burned-today)
     set daily-calorie-intake (calories-consumed-today)
     set calories-burned-today 0 
     set calories-consumed-today 0
     set nutrients-burned-today 0
     set body-fat-burned-today 0
     
     set age ( age + ( 1 / 365 ) )
  ] 
  
  ;;Agent actions for each time step
  eat-meals
  exercise
  digest
  burn-calories
  visualize-body-fats
  update-BMR
  if(update-plots? ) [update-plots]
  tick
  
  
end
to eat-meals
  ;;Eat meals
  
  ;;MCDONALDS FOR EVERY MEAL!!!
  if( diet-pattern = "mcdonalds" )
  [
  if ( ( ticks mod ticks-per-day ) = floor(ticks-per-day / 4) )
  [
    ;;Breakfast at 8 am
    eat-food-named "sausage-mcmuffin"
    eat-food-named "orange-juice"
  ]
  
  if (ticks mod ticks-per-day = floor(ticks-per-day / 2) )
  [
    ;;Lunch at 12 pm
    eat-food-named "big-mac"
    eat-food-named "large-fries"
    eat-food-named "large-coke"
  ]
  
  if(ticks mod ticks-per-day = floor(ticks-per-day * 3 / 4))
  [
    ;;Dinner at 6 pm
    eat-food-named "big-mac"
    eat-food-named "large-coke"
    eat-food-named "large-fries"
  ]
  
  if( ticks mod ticks-per-day = floor(ticks-per-day * 7 / 8 ) )
  [
    ;;10 PM snack
    eat-food-named "oreo-mcflurry"
    eat-food-named "soy-milk"
  ]
  ]
  
  ;;Typical Vegan Diet (raw foods only)
  ;;Nurtitional information from Livestrong.com calorie counter
  if( diet-pattern = "vegan" )
  [
  if ( ( ticks mod ticks-per-day ) = floor(ticks-per-day / 4) )
  [
    ;;Breakfast at 8 am
    eat-food-named "mango-smoothie"
    eat-food-named "whole-wheat-bagel"
  ]
  
  if (ticks mod ticks-per-day = floor(ticks-per-day / 2) )
  [
    ;;Lunch at 12 pm
    eat-food-named "kidney-beans"
    eat-food-named "apple-juice"
  ]
  
  if(ticks mod ticks-per-day = floor(ticks-per-day * 3 / 4))
  [
    ;;Dinner at 6 pm
    eat-food-named "pbj-sandwich"
    eat-food-named "fried-brown-rice"
  ]
  
  if( ticks mod ticks-per-day = floor(ticks-per-day * 7 / 8 ) )
  [
    ;;10 PM snack
    eat-food-named "trail-mix"
    eat-food-named "soy-milk"
  ]
  ]
end

to eat-food-named [ food-name ]
  
  ;;VEGAN DIET
  if( food-name = "mango-smoothie" )
  [
    eat-food 10 61 2 3 0
  ]
  if( food-name = "whole-wheat-bagel" )
  [
    eat-food 2 52 13 8 0
  ]
  if( food-name = "kidney-beans" )
  [
    eat-food 2 41 21 16 0
  ]
  if( food-name = "apple-juice" )
  [
    eat-food 0 27 0 1 0
  ]
  if( food-name = "pbj-sandwich" )
  [
    eat-food 10 44 9 3 0
  ]
  if( food-name = "fried-brown-rice" )
  [
    eat-food 6 39 6 4 0
  ]
  if( food-name = "trail-mix" )
  [
    eat-food 10 12 5 1 0
  ]
if( food-name = "soy-milk" )
  [
    eat-food 8 30 16 3 0
  ]
  
  ;;MCDONALDS
  if( food-name = "big-mac" )
  [
    eat-food 29 45 25 6 0  ;;McDonalds Big Mac
  ]
  if( food-name = "sausage-mcmuffin" )
  [
    eat-food 22 29 14 2 0 ;;McDonalds Sausage McMuffin
  ]
  if( food-name = "orange-juice" )
  [
    eat-food 0 39 3 0 0 ;;McDonalds Medium Orange Juice
  ]
  if( food-name = "diet-coke" )
  [
    eat-food 0 0 0 0 0;;Medium Diet Coke
  ]
  if( food-name = "large-coke" )
  [
    eat-food 0 86 0 0 0;; Large Coca-Cola
  ]
  if( food-name = "large-fries" )
  [
    eat-food 25 63 6 6 0 ;;McDonalds Large Fries
  ]
  if( food-name = "oreo-mcflurry" )
  [
    eat-food 19 89 13 3 0;;McDonalds Oreo McFlurry
  ]
  
  ;;Junk foods
  if( food-name = "doritos")
  [
    eat-food 7 18 2 1 0
  ]
  if( food-name = "wheat-thins")
  [
    eat-food 6 21 2 1 0
  ]
  if( food-name = "cheese-its")
  [
    eat-food 8 17 3 0 0
  ]
  if( food-name = "sour-patch")
  [
    eat-food 0 36 0 0 0
  ]
end

to eat-junk-food
  ;;Random snacks
  ;;0 = Doritos
  ;;1 = Wheat thins
  ;;2 = Cheese-its
  let snack-choice (random 4)
  
  if( snack-choice = 0 )
  [
    eat-food-named "doritos"
  ]
  if( snack-choice = 1 )
  [
    eat-food-named "wheat-thins"
  ]
  if( snack-choice = 2 )
  [
    eat-food-named "cheese-its"
  ]
  if( snack-choice = 3 )
  [
    eat-food-named "sour-patch"
  ]
  
end

to eat-food [ grams-fat grams-carbohydrate grams-protein grams-fiber grams-water ]
 
  create-fats ( grams-fat * 9 )  ;; 9 Calories per gram of fat
  [
    set color yellow
    setxy ( ( random max-pxcor )  + 1)  random-ycor
    set calories-consumed-today (calories-consumed-today + 1)
    set size 0.9
  ]
  create-carbohydrates ( grams-carbohydrate * 4 ) ;; 4 Calories per gram of carbs
  [
    set color brown
    setxy ( ( random max-pxcor )  + 1) random-ycor
    set calories-consumed-today (calories-consumed-today + 1)
    set size 0.9
  ]
  create-proteins ( grams-protein * 4 ) ;; 4 calories per gram of protein
  [
    set color red
    setxy ( ( random max-pxcor )  + 1) random-ycor
    set calories-consumed-today (calories-consumed-today + 1)
    set size 0.9
  ]
  create-fibers ( grams-fiber * 4 ) ;; 4 Calories per gram of carbs
  [
    set color green
    setxy ( ( random max-pxcor )  + 1) random-ycor
    set size 0.9
  ]
  create-water ( grams-water )
  [
    set color blue
    setxy ( ( random max-pxcor )  + 1) random-ycor
    set size 0.9
  ]
end

to digest
  let calories-consumed-this-tick ( basal-metabolic-rate / ticks-per-day )
  let frac ( calories-consumed-this-tick - (floor calories-consumed-this-tick) )
  
  ;Compensate for fractional calories-consumed-this-minute
  if ( random-float 1 < frac ) [ set calories-consumed-this-tick ( calories-consumed-this-tick + 1) ]
  let eat-junk-food?  false
  
  repeat calories-consumed-this-tick
  [
    set eat-junk-food? false
    ;;Consume as many available calories as possible, on the last 5 rows. This means they are available for use. If there are not enough in 
    ifelse( count turtles with [ ycor - 5 < min-pycor and xcor > 0 and breed != calorie-burners and breed != fibers] > 0 )
    [  
        ask one-of turtles with [ (ycor - 5 < min-pycor ) and xcor > 0 and breed != calorie-burners and breed != fibers]
        [ 
          ;; With fat interactions turned on, digesting a fat occasionally causes the subject to eat more junk food. This models the phenomenon that diets high in fat often indirectly lead to
          ;; increased insulin levels, which in turn can spike the desire for fats and carbohydrates
          if( fat-interactions? and breed = fats and ( random 200 < 1 ) ) [ set eat-junk-food? true ]        
          set calories-burned-today ( calories-burned-today + 1 )
          die
        ]
    ]
    [
        set body-fat-calories (body-fat-calories - 1)
        set calories-burned-today ( calories-burned-today + 1 )
    ]
    
    if(eat-junk-food?) [ eat-junk-food ]
  ]
  
  
  ask turtles with [xcor > 0 and breed != calorie-burners]
  [
    ;;Speed is influenced by fibers. Fiber in the diet slows down digestion speed, in this model, fiber acts as a bonding agent and slows nearby agents down to 35% normal speed
    let speed ( calories-consumed-this-tick / ( max-pycor * 2 ) )
    if (  fiber-interactions? and count fibers-here > 0)
    [
      set speed (speed * 0.35)
    ]
    
    
    ;; on the last row, the unburned nutrients go into persistent storage, i.e. turn into body fat
    ifelse( ycor - speed < min-pycor)
    [
      if( breed != fibers) [ set body-fat-calories ( body-fat-calories + 1 ) ]
      die  
      ;;setxy (( -1 * ( random max-pxcor )) - 1 )  max-pycor
      ;;set color violet 
    ]
    [
        set ycor (ycor - speed)
    ]
  ]
end

to visualize-body-fats
  if(count body-fats < body-fat-calories / 500)
  [
    create-body-fats( ( body-fat-calories - (count body-fats) * 500 ) / 500 )
    [
      set color violet
      setxy (random min-pxcor - 1) random-ycor
      set size 0.9
    ]
  ]
  
  if( count body-fats > body-fat-calories / 500)
  [
    ask n-of (( (count body-fats) * 500 - body-fat-calories) / 500 ) body-fats
    [
      die
    ]
  ]
end

to burn-calories
  ask calorie-burners
  [ 
    rt (random 20 ) - 10
    forward 0.5
    
    if(count ( other turtles-here with [ breed != calorie-burners and breed != fibers] ) > 0 )
    [
       ask other turtles-here with [ breed != calorie-burners and breed != fibers]
       [
         ifelse( breed = body-fats )
         [
           ;; If the calorie burner reaches a body fat, subtract body fat calories and destroy the calorie burner
           set body-fat-calories (body-fat-calories - 1 )
           set body-fat-burned-today (body-fat-burned-today +  1 )
           ask myself [ set calories-left-to-burn (calories-left-to-burn - 1) ]
         ]
         [
           ;; If the calorie burner reaches a normal nutrient, 
           ask myself [ set calories-left-to-burn (calories-left-to-burn - 1) ]
           set nutrients-burned-today ( nutrients-burned-today + 1 )
           die
         ]
       ]
       if ( calories-left-to-burn <= 0 ) 
       [ 
         set calories-burned-today (calories-burned-today + 10)
         die 
       ]
    ]
  ]
end

to exercise
  ;;Exercising will generate a certain number of calorie-burning agents per minute
    ;;Exercise at 3 pm
  if(ticks mod ticks-per-day = floor(ticks-per-day * 1 / 3))
  [
    create-calorie-burners( (exercise-level / 2) / 10 )
    [
      setxy random max-pxcor random-pycor
      set color pink
      set calories-left-to-burn 10
      set size 0.9
    ] 
  ]
  
  if(ticks mod ticks-per-day = floor(ticks-per-day * 2 / 3))
  [
    create-calorie-burners( (exercise-level / 2) / 10 )
    [
      setxy random max-pxcor random-pycor
      set color pink
      set calories-left-to-burn 10
      set size 0.9
    ]
  ]
end

to update-plots
  set-current-plot "Nutrient Breakdown"
  set-current-plot-pen "carbohydrates"
  plot count carbohydrates
  
  set-current-plot-pen "fats"
  plot count fats
  
  set-current-plot-pen "protein"
  plot count proteins
  
  set-current-plot-pen "fibers"
  plot count fibers
  
  set-current-plot-pen "water"
  plot count water
  
  set-current-plot "Weight"
  ;;Each agent has a gram weight associated with it
  set weight ( base-weight + ( (count fats) / 9 + (count carbohydrates) / 4 + (count proteins) / 4 ) / 1000 ) + (body-fat-calories / 7716 )
  
  ifelse( not weight-in-kg)
  [
    plot ( weight * 2.204 )
  ]
  [
    plot weight
  ]
  
  set-current-plot "Body Composition"
  set-current-plot-pen "% Body Fat"
  let %-body-fat ( ( body-fat-calories / 3500 ) / ( weight * 2.204 ) )
  plot %-body-fat
  
  set-current-plot-pen "% Other"
  plot 1 - %-body-fat
  
  
end
@#$#@#$#@
GRAPHICS-WINDOW
211
30
650
490
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks

BUTTON
7
12
73
45
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
84
12
147
45
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

PLOT
658
11
1160
201
Nutrient Breakdown
Time
kCal
0.0
10.0
0.0
10.0
true
true
PENS
"carbohydrates" 1.0 0 -6459832 true
"fats" 1.0 0 -1184463 true
"water" 1.0 0 -13345367 true
"fibers" 1.0 0 -10899396 true
"protein" 1.0 0 -2674135 true

MONITOR
979
369
1068
414
Days Passed
ticks / ticks-per-day
2
1
11

MONITOR
809
368
905
413
Weight (lbs)
weight * 2.204\\n
4
1
11

PLOT
659
204
898
359
Weight
Time
Weight (lbs. or kg)
0.0
10.0
150.0
175.0
true
false
PENS
"default" 1.0 0 -16777216 true

SWITCH
1074
410
1198
443
weight-in-kg
weight-in-kg
1
1
-1000

SLIDER
8
55
186
88
initial-weight-in-lbs
initial-weight-in-lbs
0
300
110
1
1
NIL
HORIZONTAL

SLIDER
8
94
206
127
initial-height-in-inches
initial-height-in-inches
50
86
60
1
1
NIL
HORIZONTAL

SLIDER
8
132
180
165
initial-age
initial-age
18
70
19
1
1
NIL
HORIZONTAL

SLIDER
9
172
181
205
initial-body-fat-%
initial-body-fat-%
0.02
0.7
0.1
0.01
1
NIL
HORIZONTAL

MONITOR
660
367
803
412
Basal Metabolic Rate
basal-metabolic-rate
0
1
11

SWITCH
10
216
113
249
male?
male?
1
1
-1000

MONITOR
911
369
968
414
Age
age
4
1
11

MONITOR
659
422
790
467
Pounds of Body Fat
body-fat-calories / 3500
3
1
11

SLIDER
7
312
179
345
minutes-per-tick
minutes-per-tick
20
60
60
20
1
NIL
HORIZONTAL

SWITCH
1073
369
1198
402
update-plots?
update-plots?
0
1
-1000

SLIDER
7
273
179
306
exercise-level
exercise-level
0
3000
900
50
1
NIL
HORIZONTAL

TEXTBOX
11
255
209
283
Adjust these sliders mid simulation:
11
0.0
1

SWITCH
5
376
177
409
fiber-interactions?
fiber-interactions?
0
1
-1000

TEXTBOX
8
357
158
375
Agent Interactions:
11
0.0
1

SWITCH
6
415
165
448
fat-interactions?
fat-interactions?
0
1
-1000

CHOOSER
4
457
142
502
diet-pattern
diet-pattern
"mcdonalds" "vegan"
1

MONITOR
929
422
1070
467
Daily Calorie Surplus
yesterdays-calorie-balance
17
1
11

MONITOR
794
422
925
467
Daily Calorie Intake
daily-calorie-intake\\n
17
1
11

TEXTBOX
281
10
362
28
Body Reserves
11
0.0
1

TEXTBOX
508
10
608
28
Digestive System
11
0.0
1

TEXTBOX
212
493
312
511
CARBOHYDRATES
11
35.0
1

TEXTBOX
316
493
346
511
FATS
11
45.0
1

TEXTBOX
439
493
479
511
WATER
11
105.0
1

TEXTBOX
351
493
383
511
FIBER
11
55.0
1

TEXTBOX
386
493
435
511
PROTEIN
11
15.0
1

TEXTBOX
481
493
581
511
CALORIE BURNERS
11
135.0
1

TEXTBOX
586
493
642
511
BODY FAT
11
115.0
1

PLOT
905
204
1161
360
Body Composition
Time
Percentage
0.0
10.0
0.0
1.0
true
true
PENS
"% Body Fat" 1.0 0 -8630108 true
"% Muscle" 1.0 0 -2674135 true
"% Bone" 1.0 0 -7500403 true
"% Other" 1.0 0 -16777216 true

MONITOR
660
474
865
519
% Nutrients burned during Exercise
%-nutrients-burned-during-exercise
3
1
11

@#$#@#$#@
WHAT IS IT?
-----------
This project is an agent-based model of human nutrition, created to study the effects of different macro level phenomenon on health.  In the nutrition model, the simple macronutrients that make up human diets are included.  These are fats, carbohydrates, proteins, fiber, and water.  The model explores the effects of different patterns of diet and exercise on weight and metabolism over different time horizons.  As part of the model, exercise is also included.  Exercise patterns can be changed over time mid-simulation.  Agent interactions are enabled in a modular fashion, and can be experimented with simultaneously for different theoretical conclusions.  The macronutrients are the agents in the model, and the environment is the human subject being studied.


HOW IT WORKS
------------
VISUALIZATION: The visualization aspect of the model is one of the most imporant educational tools. When the subject eats a meal, the digestive system is populated with nutrients that will be consumed by exercise and the body's resting caloric needs. 

MACRONUTRIENTS: The macronutrients are represented by different colors, that flow through the digestive system to supply the body with energy. They can be consumed by the environment, representative of the BMR, or the energy required to pump blood, expand the lungs, etc. Or they can be consumed by calorie burners as energy for exercise.

BODY FATS: The purple agents represent the amount of body fat in the system. They are created and destroyed as the subject gains and loses weight. Both exercise and a calorie defecit burn body fat.

CALORIE BURNERS: The pink agents represent the body's need for energy during exercise. When a subject exercises, their muscles and cardiovascular system need extra energy to operate. The Calorie Burners seek out energy with a preference for fast-fueling nutrients. 


HOW TO USE IT
-------------
First, enter in the initial body information for the subject. This determines BMR among other dynamic variables. You can also give the subject an initial exercise level. This is how many calories the subject will burn every day doing additional work. It is important to note that even sitting at a desk, working on a computer, burns more calories than lying in bed. 

You can also specify which agent interactions to include and the diet pattern of the subject.

THINGS TO NOTICE
----------------
Notice long term weight gain patterns and the very long term ( 1 - 5 year) equilibrium that is reached with any steady diet. If caloric intake and exercise are kept constant, the body will eventually reach a level where it has either completely deteriorated, or BMR has increased to the point where the daily caloric surplus reaches an average of 0. 

Notice the slope of the body composition and weight plots. The BMR serves to decelerate weight gain, and accelerate weight loss, as the body requires more or less energy for daily resting function.

Notice the affect of fiber on keeping the digestive system populated. This is the feeling of being "full" that is discussed as a benefit to a diet high in fiber when attempting weight loss. 

Notice the compounding effect of the fat interaction. When increased insulin levels lead the subject to crave fats and carbohydrates, the extra carbohydrates and fats consumed as a result serve to compound this effect. 

THINGS TO TRY
-------------
Try experimenting with different body types. While a 110 pound teenage woman would undergo massive weight gain on the McDonalds diet, a 190 pound active male could maintain a steady weight in the short run, ignoring the long term consequences of a highly processed diet.

Try varying exercise slightly. What would happen if you ran 30 minutes a day for an extra 400 calories burned? 

Try experimenting with the agent interactions. Does the fat interaction create a viscious cycle of weight gain and insatiable cravings? Does the fiber interaction create a more steady stream of energy for the body?


EXTENDING THE MODEL
-------------------
Human nutrition is fundamentally too complicated to model accurately with our current knowledge of the subject, but many interactions and agents could be added to the model to expand the conclusions generated.

AGENT ADDITIONS:
Micronutrients (Vitamins, Minerals)
Muscle
Bone
Connective Tissue
Intermediate agents (ATP, Glycogen, Amino Acids)

INTERACTIONS:
Metabolic Pathways
Muscle Development
More complex BMR calculation


RELATED MODELS
--------------
Sadly, there are no other agent based models for nutrition as of June 2011. In fact, there are no comprehensive computational models for human nutrition. The closest models that exist are for optomizing beef yeild in mega cattle farms. 


CREDITS AND REFERENCES
----------------------
Dietary Guidelines: www.nutrition.gov

NetLogo Modeling Commons link: 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 4.1.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="1200"/>
    <metric>%-nutrients-burned-during-exercise</metric>
    <enumeratedValueSet variable="fat-interactions?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="weight-in-kg">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="diet-pattern">
      <value value="&quot;vegan&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="update-plots?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-height-in-inches">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-age">
      <value value="19"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="fiber-interactions?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="exercise-level">
      <value value="900"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="male?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="minutes-per-tick">
      <value value="60"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-weight-in-lbs">
      <value value="110"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-body-fat-%">
      <value value="0.1"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
