;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable and Breed declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
globals [
  ;; lists used to create the various turtles
  shape-names       ;; list that holds the names of the non-sick shapes a student's turtle can have
  colors            ;; list that holds the colors used for students' turtles
  color-names       ;; list that holds the names of the colors used for students' turtles
  used-shape-colors ;; list that holds the shape-color pairs that are already being used

  ;; misc
  max-possible-codes ;; total number of possible unique shape/color combinations

  the-points
]

breed [ edges edge ]
breed [ students student ] ;; created and controlled by the clients

students-own [
  base-shape   ;; original shape of a turtle
  user-id  ;; unique id, input by the client when they log in, to identify each student turtle
  step-size
  edge-1
  edge-2
  angle-between-edges
]
edges-own [a b]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;
to startup
  setup
  hubnet-set-client-interface "COMPUTER" []
  hubnet-reset
end

;; Initializes the display, and creates a list that contains the names of the shapes
;; used by turtles in this activity.  Also initializes the data lists.
to setup
  cp
  ct
  clear-output
  setup-vars
  set-default-shape edges "line"
end

;; initialize global variables
to setup-vars
  set shape-names [
    "airplane"
    "android"
    "butterfly"
    "cactus"
    "cat"
    "cow skull"
    "ghost"
    "heart"
    "leaf"
    "monster"
  ]
  set colors      (list brown green yellow (violet + 1)
                        (sky - 1))
  set color-names ["brown" "green" "yellow"
                   "purple" "blue"]
  set max-possible-codes (length colors * length shape-names)
  set used-shape-colors []
  set the-points []
end

;; give the user some information about what the setup button does so they can
;; know whether they want to proceed before actually doing the setup
to setup-prompt
  if user-yes-or-no? (word "The SETUP button should only be used when starting "
              "over with a new group (such as a new set of students) since "
              "all data is lost.  Use the RE-RUN button for continuing with "
              "an existing group."
              "\\n\\nDo you really want to setup the model?")
  [
    user-message (word "Before closing this dialog, please do the following:"
                 "\\n  -Have everyone that is currently logged in, log off and "
                 "then kick all remaining clients with the HubNet Console.")
    setup
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

to go
  every 0.1 [
    ;; get commands and data from the clients
    listen-clients
    display-angles
    update-clients-info
  ]
end

to edgify
  ask edges [ die ]
  let smarties [self] of students with [user-id != 0]
  let dummies [self] of students with [user-id = 0]
  ifelse fixed-together? [
    set the-points sentence (shuffle smarties) (shuffle dummies)
  ] [
    set the-points shuffle (sentence smarties dummies)
  ]
  if length the-points >= 2 [
    foreach n-values length the-points [?] [
      ask item ? the-points [
        connect-to item ((? + 1) mod length the-points) the-points
      ]
    ]
  ]
  display-angles
  update-clients-info
end

to make-fixed-point
  create-students 1 [
    set-unique-shape-and-color
    set color white
    setxy random-pxcor random-pycor
    set heading 0
  ]
end

to move-fixed-point
  let fixed-points students with [user-id = 0 ]
  if mouse-down? [
    let closest min-one-of fixed-points [distancexy-nowrap mouse-xcor mouse-ycor ]
    ask closest 
    [ 
      set xcor mouse-xcor 
      set ycor mouse-ycor 
    ]
    ask edges [reposition]
    display-angles
    update-clients-info
    update-plots
  ]
end

to display-angles
  ask patches [set plabel ""]
  ask edges [
    ask a [
      let e1 myself
      let e2 one-of edges with [ b = myself ] ; here myself is the asking a
      let n1 [b] of e1
      let n2 [a] of e2
      let angle subtract-headings towards n1 towards n2
      if  label-dood != nobody [
      ask label-dood [
        set plabel-color [color] of myself set plabel precision abs angle 0
        ask myself [set angle-between-edges abs angle]
      ]
      ]
      set edge-1 [size] of e1
      set edge-2 [size] of e2
    ]
  ]
end

to-report label-dood
  report one-of neighbors with [
    distance myself = distance-nowrap myself and
    not any? students-here
  ]
end

;; report the amount we can move in the current direction
;; if we don't want to wrap around the screen
to-report move-amount [direction] ;; turtle procedure
  let amount step-size
  let me self
    while [
      abs (xcor + (amount * dx * direction)) > max-pxcor or
      abs (ycor + (amount * dy * direction)) > max-pycor or
      (amount > 0 and
      any? [
        students-here with [self != me]
      ] of patch-ahead (amount * direction))
    ] [
      ifelse amount < 1 [
        set amount 0
      ] [
        set amount amount - 1
      ]
    ]
  report amount
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the clients ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; determines which client sent a command, and what the command was
to listen-clients
  while [ hubnet-message-waiting? ]
  [
    hubnet-fetch-message
    ifelse hubnet-enter-message? [
      create-new-student
    ] [
    ifelse hubnet-exit-message? [
      remove-student
    ] [
      execute-command hubnet-message-tag
    ]]
  ]
end

;; NetLogo knows what each student turtle is supposed to be
;; doing based on the tag sent by the node:
to execute-command [command]
  ifelse command = "Change Appearance" [
    execute-change-turtle
  ] [
  ifelse command = "step-size" [
    ask students with [user-id = hubnet-message-source] [
      set step-size read-from-string hubnet-message
      ;; we only want non integers when it's less than 1
      if step-size > 1 [
        let closest-int round step-size
        ifelse closest-int = ceiling step-size [
          set step-size floor step-size
        ] [
          set step-size ceiling step-size
        ]
      ]
    ]
  ] [
  ifelse command = "Lt" [
    execute-turn -45
  ] [
  ifelse command = "Rt" [
    execute-turn 45
  ] [
  ifelse command = "Up" [
    execute-move 0
  ] [
  ifelse command = "Down" [
    execute-move 180
  ] [
  ifelse command = "Right" [
    execute-move 90 stop
  ] [
  ifelse command = "Left" [
    execute-move 270
  ] [
  ifelse command = "Fd" [
    execute-go 1
  ] [
  ifelse command = "Bk" [
    execute-go -1
  ] [
  ]]]]]]
    ;; executed no matter which movement command it was
    if any? edges [ update-plots ]
  ]
  if hubnet-message-tag = "View" [
    ask students with [ user-id = hubnet-message-source ] [
      move-to-click
    ]
  ]
  ]]]
end

to move-to-click ; student procedure
  let patch-clicked-on patch (round item 0 hubnet-message) (round item 1 hubnet-message)
  setxy [pxcor] of patch-clicked-on [pycor] of patch-clicked-on
  ask edges [ reposition ]
end

;; Create a turtle, set its shape, color, and position
;; and tell the node what its turtle looks like and where it is
to create-new-student
  create-students 1
  [
    setup-student-vars
    send-init-to-client
  ]
end

;; sets the turtle variables to appropriate initial values
to setup-student-vars  ;; turtle procedure
  set user-id hubnet-message-source
  set step-size 1
  set-unique-shape-and-color
  while [ any? other students-here ] [
    setxy random-pxcor random-pycor
  ]
  set heading 0
end

;; sends the appropriate monitor information back to the client
to send-init-to-client
  hubnet-send user-id "You are a:" (word (color-string color) " " base-shape)
  hubnet-send user-id "Located at:" (word "(" pxcor "," pycor ")")
  hubnet-send user-id "Heading:" heading-text
  hubnet-send user-id "Perimeter:" perimeter
  hubnet-send user-id "Area:" area
end

;; pick a base-shape and color for the turtle
to set-unique-shape-and-color
  let code random max-possible-codes
  while [member? code used-shape-colors and count students < max-possible-codes]
  [
    set code random max-possible-codes
  ]
  set used-shape-colors (lput code used-shape-colors)
  set base-shape item (code mod length shape-names) shape-names
  set shape base-shape
  set color item (code / length shape-names) colors
  set label-color color
end

;; report the string version of the turtle's color
to-report color-string [color-value]
  report item (position color-value colors) color-names
end

to-report heading-text
  ifelse heading = 0 [
    report "^"
  ] [
  ifelse heading = 45 [
    report "/^"
  ] [
  ifelse heading = 90 [
    report ">"
  ] [
  ifelse heading = 135 [
    report "\\\\v"
  ] [
  ifelse heading = 180 [
    report "v"
  ] [
  ifelse heading = 225 [
    report "v/"
  ] [
  ifelse heading = 270 [
    report "<"
  ] [
  if heading = 315 [
    report "^\\\\"
  ]]]]]]]]
end

;; Kill the turtle, set its shape, color, and position
;; and tell the node what its turtle looks like and where it is
to remove-student
  ask students with [user-id = hubnet-message-source]
  [
    set used-shape-colors remove my-code used-shape-colors
    die
  ]
end

;; translates a student turtle's shape and color into a code
to-report my-code
  report (position base-shape shape-names) + (length shape-names) * (position color colors)
end

;XXXXXXXXXX these 3 procedures should be combined if possible
;; Cause the students to move forward step-size in new-heading's heading
to execute-move [new-heading]
  ask students with [user-id = hubnet-message-source]
  [
    set heading new-heading
    jump move-amount 1 ; 1 means forward direction
    set heading 0
    ask edges [ reposition ]
  ]
end

;XXXXXXXX terrible name
to execute-go [direction]
  ask students with [user-id = hubnet-message-source]
  [
    jump direction * move-amount direction
    ask edges [ reposition ]
  ]
end

to execute-turn [amount]
  ask students with [user-id = hubnet-message-source]
  [
    rt amount
  ]
end

to update-clients-info
  let user-id-dooder 0
  foreach [ self ] of students with [user-id != 0] [
    set user-id-dooder [user-id] of ?
    hubnet-send user-id-dooder "Perimeter:" perimeter
    hubnet-send user-id-dooder "Area:" area
    hubnet-send user-id-dooder "Heading:" [ heading-text ] of ?
    hubnet-send user-id-dooder "step-size" [step-size] of ?
    hubnet-send user-id-dooder "Located at:" (word "(" precision [xcor] of ? 1 "," precision [ycor] of ? 1 ")")
  ]
end

to-report perimeter
  report precision sum [size] of edges 1
end

to-report area
  let vertices the-points
  let result 0
  foreach n-values length vertices [?] [
    let vertex item ? vertices
    let other-vertex item ((? + 1) mod length vertices) vertices
    let addend (([pxcor] of vertex * [pycor] of other-vertex) -
                ([pycor] of vertex * [pxcor] of other-vertex))
    set result result + addend
  ]
  report precision abs (result / 2) 1
end

to update-plots
  set-current-plot "perimeter"
  plot perimeter
  set-current-plot "area"
  plot area
end

to execute-change-turtle
  ask students with [user-id = hubnet-message-source]
  [
    set used-shape-colors remove my-code used-shape-colors
    show-turtle
    set-unique-shape-and-color
    hubnet-send user-id "You are a:" (word (color-string color) " " base-shape)
  ]
end

to connect-to [other-node]  ;; node procedure
  hatch 1
    [ set breed edges
      set a myself
      set b other-node
      set color white
      set label-color white
      reposition ]
end

to reposition  ;; edge procedure
  setxy ([xcor] of a) ([ycor] of a)
  set heading towards-nowrap b
  set size distance-nowrap b
  jump (distance-nowrap b) / 2
  set label precision size 1
end
@#$#@#$#@
GRAPHICS-WINDOW
440
10
870
461
10
10
20.0
1
18
1
1
1
0
0
0
1
-10
10
-10
10
0
0
1
ticks

CC-WINDOW
5
475
879
570
Command Center
0

BUTTON
18
10
78
43
Setup
setup-prompt
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
89
10
153
43
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
173
10
239
43
NIL
edgify
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

MONITOR
366
273
431
318
NIL
area
1
1
11

BUTTON
16
114
144
147
NIL
make-fixed-point
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
16
188
144
221
remove fixed points
ask students with [user-id = 0] [die]\\nask patches [set plabel \\"\\"]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
16
151
144
184
fixed-together?
fixed-together?
1
1
-1000

PLOT
163
96
363
246
perimeter
NIL
NIL
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 0 -16777216 true

MONITOR
366
96
431
141
NIL
perimeter
3
1
11

PLOT
162
273
362
423
area
NIL
NIL
0.0
10.0
0.0
10.0
true
false
PENS
"default" 1.0 0 -16777216 true

BUTTON
206
57
312
90
NIL
clear-all-plots
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
17
228
142
261
NIL
move-fixed-point
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
269
11
349
44
clean-up
edgify
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

@#$#@#$#@
VERSION
-------
$Id: Perimeters and Areas.nlogo 39722 2008-05-01 18:38:33Z everreau $


@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

android
true
0
Polygon -7500403 true true 210 90 240 195 210 210 165 90
Circle -7500403 true true 110 3 80
Polygon -7500403 true true 105 88 120 193 105 240 105 298 135 300 150 210 165 300 195 298 195 240 180 193 195 88
Rectangle -7500403 true true 127 81 172 96
Rectangle -16777216 true false 119 33 181 56
Polygon -7500403 true true 90 90 60 195 90 210 135 90

butterfly
true
0
Rectangle -7500403 true true 92 135 207 224
Circle -7500403 true true 158 53 134
Circle -7500403 true true 165 180 90
Circle -7500403 true true 45 180 90
Circle -7500403 true true 8 53 134
Line -16777216 false 43 189 253 189
Rectangle -7500403 true true 135 60 165 285
Circle -7500403 true true 165 15 30
Circle -7500403 true true 105 15 30
Line -7500403 true 120 30 135 60
Line -7500403 true 165 60 180 30
Line -16777216 false 135 60 135 285
Line -16777216 false 165 285 165 60

cactus
true
0
Rectangle -7500403 true true 135 30 175 177
Rectangle -7500403 true true 67 105 100 214
Rectangle -7500403 true true 217 89 251 167
Rectangle -7500403 true true 157 151 220 185
Rectangle -7500403 true true 94 189 148 233
Rectangle -7500403 true true 135 162 184 297
Circle -7500403 true true 219 76 28
Circle -7500403 true true 138 7 34
Circle -7500403 true true 67 93 30
Circle -7500403 true true 201 145 40
Circle -7500403 true true 69 193 40

cat
true
0
Line -7500403 true 285 240 210 240
Line -7500403 true 195 300 165 255
Line -7500403 true 15 240 90 240
Line -7500403 true 285 285 195 240
Line -7500403 true 105 300 135 255
Line -16777216 false 150 270 150 285
Line -16777216 false 15 75 15 120
Polygon -7500403 true true 300 15 285 30 255 30 225 75 195 60 255 15
Polygon -7500403 true true 285 135 210 135 180 150 180 45 285 90
Polygon -7500403 true true 120 45 120 210 180 210 180 45
Polygon -7500403 true true 180 195 165 300 240 285 255 225 285 195
Polygon -7500403 true true 180 225 195 285 165 300 150 300 150 255 165 225
Polygon -7500403 true true 195 195 195 165 225 150 255 135 285 135 285 195
Polygon -7500403 true true 15 135 90 135 120 150 120 45 15 90
Polygon -7500403 true true 120 195 135 300 60 285 45 225 15 195
Polygon -7500403 true true 120 225 105 285 135 300 150 300 150 255 135 225
Polygon -7500403 true true 105 195 105 165 75 150 45 135 15 135 15 195
Polygon -7500403 true true 285 120 270 90 285 15 300 15
Line -7500403 true 15 285 105 240
Polygon -7500403 true true 15 120 30 90 15 15 0 15
Polygon -7500403 true true 0 15 15 30 45 30 75 75 105 60 45 15
Line -16777216 false 164 262 209 262
Line -16777216 false 223 231 208 261
Line -16777216 false 136 262 91 262
Line -16777216 false 77 231 92 261

cow skull
true
0
Polygon -7500403 true true 150 90 75 105 60 150 75 210 105 285 195 285 225 210 240 150 225 105
Polygon -16777216 true false 150 150 90 195 90 150
Polygon -16777216 true false 150 150 210 195 210 150
Polygon -16777216 true false 105 285 135 270 150 285 165 270 195 285
Polygon -7500403 true true 240 150 263 143 278 126 287 102 287 79 280 53 273 38 261 25 246 15 227 8 241 26 253 46 258 68 257 96 246 116 229 126
Polygon -7500403 true true 60 150 37 143 22 126 13 102 13 79 20 53 27 38 39 25 54 15 73 8 59 26 47 46 42 68 43 96 54 116 71 126

ghost
true
0
Polygon -7500403 true true 30 165 13 164 -2 149 0 135 -2 119 0 105 15 75 30 75 58 104 43 119 43 134 58 134 73 134 88 104 73 44 78 14 103 -1 193 -1 223 29 208 89 208 119 238 134 253 119 240 105 238 89 240 75 255 60 270 60 283 74 300 90 298 104 298 119 300 135 285 135 285 150 268 164 238 179 208 164 208 194 238 209 253 224 268 239 268 269 238 299 178 299 148 284 103 269 58 284 43 299 58 269 103 254 148 254 193 254 163 239 118 209 88 179 73 179 58 164
Line -16777216 false 189 253 215 253
Circle -16777216 true false 102 30 30
Polygon -16777216 true false 165 105 135 105 120 120 105 105 135 75 165 75 195 105 180 120
Circle -16777216 true false 160 30 30

heart
true
0
Circle -7500403 true true 152 19 134
Polygon -7500403 true true 150 105 240 105 270 135 150 270
Polygon -7500403 true true 150 105 60 105 30 135 150 270
Line -7500403 true 150 270 150 135
Rectangle -7500403 true true 135 90 180 135
Circle -7500403 true true 14 19 134

leaf
true
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

monster
true
0
Polygon -7500403 true true 75 150 90 195 210 195 225 150 255 120 255 45 180 0 120 0 45 45 45 120
Circle -16777216 true false 165 60 60
Circle -16777216 true false 75 60 60
Polygon -7500403 true true 225 150 285 195 285 285 255 300 255 210 180 165
Polygon -7500403 true true 75 150 15 195 15 285 45 300 45 210 120 165
Polygon -7500403 true true 210 210 225 285 195 285 165 165
Polygon -7500403 true true 90 210 75 285 105 285 135 165
Rectangle -7500403 true true 135 165 165 270

@#$#@#$#@
NetLogo 4.1pre1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
VIEW
452
10
872
430
10
10
20.0
1
18
1
1
1
0
0
0
1
-10
10
-10
10

BUTTON
294
173
356
206
Up
NIL
NIL
1
T
OBSERVER
NIL
I

BUTTON
294
206
356
239
Down
NIL
NIL
1
T
OBSERVER
NIL
K

BUTTON
356
206
418
239
Right
NIL
NIL
1
T
OBSERVER
NIL
L

BUTTON
232
206
294
239
Left
NIL
NIL
1
T
OBSERVER
NIL
J

MONITOR
5
96
92
145
Located at:
NIL
3
1

MONITOR
3
10
153
59
You are a:
NIL
3
1

BUTTON
3
57
153
90
Change Appearance
NIL
NIL
1
T
OBSERVER
NIL
NIL

MONITOR
240
35
309
84
Perimeter:
NIL
1
1

MONITOR
315
35
384
84
Area:
NIL
1
1

BUTTON
93
173
156
206
Fd
NIL
NIL
1
T
OBSERVER
NIL
W

BUTTON
30
206
93
239
Lt
NIL
NIL
1
T
OBSERVER
NIL
A

BUTTON
156
206
219
239
Rt
NIL
NIL
1
T
OBSERVER
NIL
D

BUTTON
93
206
156
239
Bk
NIL
NIL
1
T
OBSERVER
NIL
S

MONITOR
96
95
154
144
Heading:
NIL
3
1

CHOOSER
155
268
293
313
step-size
step-size
"0.1" "1" "2" "3" "4" "5"
1

@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
