;; VERSION 3-September-2020 6:00
;; With more cars

extensions [csv table]

breed [ cars car ]
breed [ buyers buyer ]

globals [
  coef_uv_list ; coefficient of utility variables list. Mind the order!
  mu_car       ; mu coefficient table for the nested logit at the second step
  c_car

  ;; buyers variable coefficients
  b-hh-size
  b-spending1 b-spending2 b-spending3 b-spending4
  b-sex
  b-age1 b-age2 b-age3 b-age4 b-age5 b-age6 b-age7 b-age8
  b-ocp1 b-ocp2 b-ocp3 b-ocp4

  ;; cars variabel coefficients
  b-fuelcost
  b-co2
  b-price
  b-kerbweight
  b-dimension
  b-gasoline
  b-manual
]

cars-own [
  variant
  brand
  fuelconsmp
  fuelcost
  co2
  price
  kerbweight
  len
  width
  height
  gasoline
  manual
  chosen?       ; boolean variable mobil dipilih oleh pembeli
  dimension
  cars_uv_list
  car_utility
]

buyers-own [
  hh-size       ; ukuran rumah tangga. Jumlah seluruh orang dalam rumah tangga. (rasio)

  ;; pengeluaran per orang per bulan berdasarkan World Bank (2019), 5 kelas
  spending1  ; <354 ribu
  spending2  ; 354 ribu - 532 ribu
  spending3  ; 532 ribu - 1,2 juta
  spending4  ; 1,2 juta - 6,0 juta
  ;spending5 >6,0 juta. Tapi kan tidak ada variabelnya.

  sex           ; jenis kelamin. (nominal)

  ;; usia, 9 kelas
  age1       ; 21 - 25 tahun
  age2       ; 26 - 30 tahun
  age3       ; 31 - 35 tahun
  age4       ; 36 - 40 tahun
  age5       ; 41 - 45 tahun
  age6       ; 46 - 50 tahun
  age7       ; 51 - 55 tahun
  age8       ; 56 - 60 tahun
  ;age9 >60 tahun.

  ;; pekerjaan, nominal
  occupation1  ; swasta
  occupation2  ; pelajar/mahasiswa
  occupation3  ; TNI/Polri
  occupation4  ; PNS
  ;occupation5  BUMN.

  ;; choice-related variables
  cset-size
  who-choice-set
  car-choice

  buyers_uv_list
  uv_list
  buyer_utility
  utility

  ; tables related to cars agentset
  table_lefthalf
  table_righthalf
  table_exp_nested_sums
  brand_list
  table_car_probs
]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; S E T U P ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to setup
  clear-all         ;; start over the model clean
  file-close-all    ;; close any files open from last run

  set-buyers        ;; setup the car buyers condition using 'set-buyers' procedure
  set-cars          ;; setup the car condition using 'set-cars' procedure

  ;; setting the list for the utility function coefficients in the second step logit model
  ;; make sure not to change the order
  set b-hh-size 1
  set b-spending1 1 set b-spending2 1 set b-spending3 1 set b-spending4 1
  set b-sex 1
  set b-age1 1 set b-age2 1 set b-age3 1 set b-age4 1
  set b-age5 1 set b-age6 1 set b-age7 1 set b-age8 1
  set b-ocp1 1 set b-ocp2 1 set b-ocp3 1 set b-ocp4 1
  set b-fuelcost 1
  set b-co2 1
  set b-price -0.063
  set b-kerbweight 1
  set b-dimension 1
  set b-gasoline 1
  set b-manual 1

  set coef_uv_list (
    list b-hh-size b-spending1 b-spending2 b-spending3 b-spending4 b-sex
         b-age1 b-age2 b-age3 b-age4 b-age5 b-age6 b-age7 b-age8
         b-ocp1 b-ocp2 b-ocp3 b-ocp4 b-fuelcost b-price b-kerbweight
         b-dimension b-gasoline b-manual )

  ;; same goes for the buyers agent
  ask buyers [
    set buyers_uv_list ( list hh-size spending1 spending2 spending3 spending4 spending4 sex
    age1 age2 age3 age4 age4 age5 age6 age7 age8 occupation1 occupation2 occupation3
    occupation4 )

    set buyer_utility
    (b-hh-size * hh-size) + (b-spending1 * spending1) + (b-spending2 * spending2) + (b-spending3 * spending3) +
    (b-spending4 * spending4) + (b-sex * sex) + (b-age1 * age1) + (b-age2 * age2) + (b-age3 * age3) +
    (b-age4 * age4) + (b-age5 * age5) + (b-age6 * age6) + (b-age7 * age7) + (b-age8 * age8) +
    (b-ocp1 * occupation1) + (b-ocp2 * occupation2) + (b-ocp3 * occupation3) + (b-ocp4 * occupation4)
  ]

  ask cars [
    ;; asking cars to count their car-side utility value
    set car_utility (b-fuelcost * fuelcost) + (b-co2 * co2) + (b-price * price) + (b-dimension * dimension) +
                    (b-gasoline * gasoline) + (b-manual * manual) + (b-kerbweight * kerbweight)
  ]

  ;; making the c_car table
  set mu_car table:make
  set c_car table:make

  ;; storing each car brand with its own mu coefficient
  table:put mu_car "Aston Martin" 1.0   table:put mu_car "Audi" 1.5
  table:put mu_car "Bentley" 2.0        table:put mu_car "BMW" 2.5
  table:put mu_car "Chevrolet" 1.0
  table:put mu_car "Daihatsu" 1.2       table:put mu_car "Datsun" 3.5
  table:put mu_car "DFSK" 3.0           table:put mu_car "Dodge" 3.5
  table:put mu_car "Fiat" 1.4
  table:put mu_car "Honda" 1.3          table:put mu_car "Hyundai" -2.5
  table:put mu_car "Isuzu" 2.3
  table:put mu_car "Jaguar" 1.9         table:put mu_car "Jeep" 1.7
  table:put mu_car "KIA" 1.1
  table:put mu_car "Lamborghini" 1.1    table:put mu_car "Land Rover" 1.6   table:put mu_car "Lexus" 1.7
  table:put mu_car "Mahindra" 1.9       table:put mu_car "Maserati" 1.2     table:put mu_car "Mazda" 2.3
  table:put mu_car "Mercedes-Benz" 4.0
  table:put mu_car "MG" 4.0             table:put mu_car "MINI" 2.0         table:put mu_car "Mitsubishi" 3.0
  table:put mu_car "Nissan" 3.1
  table:put mu_car "Peugeot" 3.3        table:put mu_car "Porsche" 3.4
  table:put mu_car "Renault" 2.9        table:put mu_car "Rolls Royce" 1.7
  table:put mu_car "Suzuki" 2.9
  table:put mu_car "Toyota" 1.3
  table:put mu_car "Volkswagen" 1       table:put mu_car "Volvo" 1
  table:put mu_car "Wuling" 1

  ;; storing each car brand with its own car constant
  table:put c_car "Aston Martin" 1      table:put c_car "Audi" 1
  table:put c_car "Bentley" 1           table:put c_car "BMW" 1
  table:put c_car "Chevrolet" 1
  table:put c_car "Daihatsu" 1          table:put c_car "Datsun" 1
  table:put c_car "DFSK" 1              table:put c_car "Dodge" 1
  table:put c_car "Fiat" 1
  table:put c_car "Honda" 1             table:put c_car "Hyundai" 1
  table:put c_car "Isuzu" 1
  table:put c_car "Jaguar" 1            table:put c_car "Jeep" 1
  table:put c_car "KIA" 1
  table:put c_car "Lamborghini" 1       table:put c_car "Land Rover" 1   table:put c_car "Lexus" 1
  table:put c_car "Mahindra" 1          table:put c_car "Maserati" 1     table:put c_car "Mazda" 1
  table:put c_car "Mercedes-Benz" 1
  table:put c_car "MG" 1                table:put c_car "MINI" 1         table:put c_car "Mitsubishi" 1
  table:put c_car "Nissan" 1
  table:put c_car "Peugeot" 1           table:put c_car "Porsche" 1
  table:put c_car "Renault" 1           table:put c_car "Rolls Royce" 1
  table:put c_car "Suzuki" 1
  table:put c_car "Toyota" 1
  table:put c_car "Volkswagen" 1        table:put c_car "Volvo" 1
  table:put c_car "Wuling" 1

  setup-plots
  update-plots
end


to set-buyers    ;; buyers procedure.
  ifelse not file-exists? buyers_file [
    user-message "Tidak ditemukan file 'file_read_buyers.csv'. Mohon periksa kembali nama atau lokasi file tersebut."
    stop ]
  [ file-open buyers_file ]

  ;; Membaca seluruh data dalam satu loop
  while [ not file-at-end? ] [
    ; here the csv extension grabs a single line and puts the read data in a list
    let baca (csv:from-row file-read-line ";")
    ; now we can use that list to create a turtle with the saved properties

    create-buyers 1 [
      set hh-size item 0 baca

      ;; setting spending1 - spending5 value
      ( ifelse
        item 1 baca = 1
        [ set spending1 1  set spending2 0  set spending3 0  set spending4 0 ]
        item 1 baca = 2
        [ set spending1 0  set spending2 1  set spending3 0  set spending4 0 ]
        item 1 baca = 3
        [ set spending1 0  set spending2 0  set spending3 1  set spending4 0 ]
        item 1 baca = 4
        [ set spending1 0  set spending2 0  set spending3 0  set spending4 1 ]
        [ set spending1 0  set spending2 0  set spending3 0  set spending4 0 ])

      ;; setting occupation1 - occupation5 value
      ( ifelse
        item 2 baca = 1
        [ set occupation1 1  set occupation2 0  set occupation3 0  set occupation4 0 ]
        item 2 baca = 2
        [ set occupation1 0  set occupation2 1  set occupation3 0  set occupation4 0 ]
        item 2 baca = 3
        [ set occupation1 0  set occupation2 0  set occupation3 1  set occupation4 0 ]
        item 2 baca = 4
        [ set occupation1 0  set occupation2 0  set occupation3 0  set occupation4 1 ]
        [ set occupation1 0  set occupation2 0  set occupation3 0  set occupation4 0 ] )

      set sex item 3 baca

      ;; setting age1 - age9 value
      ( ifelse
        item 4 baca = 1
        [ set age1 1  set age2 0  set age3 0  set age4 0
          set age5 0  set age6 0  set age7 0  set age8 0 ]
        item 4 baca = 2
        [ set age1 0  set age2 1  set age3 0  set age4 0
          set age5 0  set age6 0  set age7 0  set age8 0 ]
        item 4 baca = 3
        [ set age1 0  set age2 0  set age3 1  set age4 0
          set age5 0  set age6 0  set age7 0  set age8 0 ]
        item 4 baca = 4
        [ set age1 0  set age2 0  set age3 0  set age4 1
          set age5 0  set age6 0  set age7 0  set age8 0 ]
        item 4 baca = 5
        [ set age1 0  set age2 0  set age3 0  set age4 0
          set age5 1  set age6 0  set age7 0  set age8 0 ]
        item 4 baca = 6
        [ set age1 0  set age2 0  set age3 0  set age4 0
          set age5 0  set age6 1  set age7 0  set age8 0 ]
        item 4 baca = 7
        [ set age1 0  set age2 0  set age3 0  set age4 0
          set age5 0  set age6 0  set age7 1  set age8 0 ]
        item 4 baca = 8
        [ set age1 0  set age2 0  set age3 0  set age4 0
          set age5 0  set age6 0  set age7 0  set age8 1 ]
        [ set age1 0  set age2 0  set age3 0  set age4 0
          set age5 0  set age6 0  set age7 0  set age8 0 ] )
    ]

    ask buyers [
      set shape        "person"
      set color        green
      let bound-buyers random -45 - 5
      set xcor         random-xcor
      set ycor         random-ycor ]
  ]

  file-close ;; make sure to close the file
end


to set-cars  ;; cars procedure.

  ifelse not file-exists? cars_file [
    user-message "Tidak ditemukan file 'file_read_cars.csv'. Mohon periksa kembali nama atau lokasi file tersebut."
    stop ]
  [ file-open cars_file ]

  ;; Membaca seluruh data dalam satu loop
  while [ not file-at-end? ][
    ; here the csv extension grabs a single line and puts the read data in a list
    let baca (csv:from-row file-read-line ";")
    ; now we can use that list to create a turtle with the saved properties
    create-cars 1 [
      set variant    item 0 baca
      set brand      item 1 baca
      set fuelconsmp item 2 baca
      set fuelcost   item 3 baca / 10000
      set co2        item 4 baca / 100
      set price      item 5 baca / 1000
      set kerbweight item 6 baca / 1000
      set len        item 7 baca
      set width      item 8 baca
      set height     item 9 baca
      set gasoline   item 10 baca
      set manual     item 11 baca ]

    ask cars [
      set shape        "car"
      set color        yellow
      let bound-cars   random 45 + 5
      set xcor         random-xcor
      set ycor         random-ycor
      set dimension    (len * width) / 10000000 ]
  ]

  file-close ;; make sure to close the file
end




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;   G  0    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to go
  first-step
  second-step
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;   F  I  R  S  T      S  T  E  P    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to first-step
  ask cars [
    set chosen? 0 ]

  first-stage

  ;; this is the equivalence to "tick" if you decide your model won't involve time frame.
  update-plots
end


to first-stage  ;; buyers procedure.
  ask buyers [
    let a 4
    let b 1 / 6

    ;; set the number of cars choice set for each buyer
    ;; by the non-compensatory rule
    set cset-size round random-gamma a b

    set who-choice-set [who] of n-of cset-size cars ;; set buyers' car choice-set as an agentset
    set label cset-size
  ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  S  E  C  O  N  D      S  T  E  P    ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to second-step
  ask buyers [
    ;; table for storing each car's exponentials of utility.
    set table_lefthalf table:make
    foreach who-choice-set [
      x ->
      let c table:get c_car [brand] of car x
      let value (buyer_utility + ([car_utility] of car x) + c)
      table:put table_lefthalf ([who] of car x) (exp (value * table:get mu_car [brand] of car x))
    ]

    set table_righthalf table:make

    ;; procedures to make the table_righthalf table, table for storing aggregate utility
    ;; value for cars within the same brand.
    ; first make a list of list containing [["brand" utility_value]["brand 2" utility_value2]...]
    set brand_list []
    foreach who-choice-set
    [ x -> set brand_list lput (list [brand] of car x table:get table_lefthalf x) brand_list ]
    ; next sort the list of list by the brand name alphabetically
    set brand_list sort-by [[a b] -> first a < first b] brand_list
    ; after that we are going to make aggregated list of car brands and the corresponding value
    set brand_list make-righthalf brand_list
    ; now we assign that list to table_righthalf
    set table_righthalf table:from-list brand_list
    ; we also need a table for the exp(1 / mu_j * ln total exp(mu_j * Vij))
    set table_exp_nested_sums table:from-list make-exp-nested-sums brand_list

    ; now we can compute each car's probability
    let car_probs []
    foreach who-choice-set [
      id -> set car_probs lput list id count-exp id car_probs
    ]
    ; sort the cars based on their probability descending
    set car_probs sort-by [[a b] -> last a > last b] car_probs
    set table_car_probs table:from-list car_probs

    ; after each car has its own probability, we ask buyer to choose the car that
    ; has the highest probability value
    let highestp car first first car_probs
    set car-choice highestp
    ask car-choice [
      set chosen? 1
    ]
  ]

  update-plots
end

;this is my version to make the righthalf table
;to-report make-righthalf [li]
;  let result []
;  let first_item item 0 li     ; first_item is also a list
;  let li-length length li
;  let li-end li-length - 1
;  foreach (range 1 li-length) [
;    ? -> ifelse ? != li-end
;    [ ifelse item 0 item ? li = item 0 first_item
;      [ set first_item replace-item 1 first_item (item 1 first_item + item 1 item ? li) ]
;      [ set result lput first_item result
;        set first_item item ? li ]
;    ]
;    [ ifelse item 0 item ? li = item 0 first_item
;      [ set first_item replace-item 1 first_item (item 1 first_item + item 1 item ? li)
;        set result lput first_item result ]
;      [ set result lput first_item result
;        set result lput item li-end li result ]
;    ]
;  ]
;  report result
;end

;this is JenB's version
to-report make-righthalf [li]
  ; this procedure assumes that li has been sorted
  ; uncomment this following code if above assumption is not used
;  let sorted-list sort-by [[a b] -> first a < first b] li
  ; prime the loop
  let pair-to-add first li
  let active-letter first pair-to-add
  let result []

  ; loop from the second item to the end
  foreach but-first li [
    pair ->
    ifelse first pair = active-letter
    ; in same first letter so add
    [ set pair-to-add (list first pair-to-add (last pair-to-add + last pair)) ]
    ; in different first letter so output and use this item as start of next chain
    [ set result lput pair-to-add result
      set pair-to-add pair
      set active-letter first pair
    ]
  ]
  ; append the final pair
  set result lput pair-to-add result
  report result
end

to-report make-exp-nested-sums [li]
  let result []
  foreach li [
    pair ->
    let the-brand first pair
    let cmu table:get mu_car the-brand
    set result lput list the-brand ((1 / cmu) * (ln last pair)) result
  ]
  report result
end

to-report count-exp [id]
  let p_num_lefthalf table:get table_lefthalf id
  let p_num_righthalf exp (1 / table:get mu_car [brand] of car id) * ln (table:get table_righthalf [brand] of car id)
  let p_denom_lefthalf table:get table_righthalf [brand] of car id
  let p_denom_righthalf sum table:values table_exp_nested_sums

  report (p_num_lefthalf / p_denom_lefthalf) * (p_num_righthalf * p_denom_righthalf)
end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;   C   L   E   A   R     ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to clear
  reset-ticks

  ask cars [
    set chosen? 0 ]

  ask buyers [
    set cset-size 0
    set label ""
  ]

  setup-plots
end
@#$#@#$#@
GRAPHICS-WINDOW
25
130
599
481
-1
-1
5.61
1
10
1
1
1
0
0
0
1
-50
50
-30
30
0
0
1
ticks
30.0

BUTTON
25
10
105
46
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
620
290
860
440
co2 of chosen cars
co2 rate
persentase
50.0
450.0
0.0
70.0
false
false
"clear-plot" ""
PENS
"default" 10.0 0 -16777216 true "" "histogram [co2 * 100] of cars with [chosen? = 1]"

PLOT
870
290
1110
440
cset size
NIL
NIL
0.0
100.0
0.0
100.0
false
false
"clear-plot" ""
PENS
"default" 1.0 1 -955883 true "" "histogram [cset-size] of buyers"

INPUTBOX
490
45
610
105
cars_file
file_read_cars_2.csv
1
0
String

TEXTBOX
490
10
930
40
Isikan nama file cars dan buyers di bawah ini.\nPastikan kedua file berada dalam folder yang sama dengan file model ini: 
11
0.0
1

INPUTBOX
620
45
755
105
buyers_file
file_read_buyers_2.csv
1
0
String

MONITOR
855
10
945
55
NIL
count buyers
17
1
11

MONITOR
855
60
945
105
NIL
count cars
17
1
11

MONITOR
950
60
1055
105
count chosen cars
count cars with [chosen? = 1]
17
1
11

PLOT
620
130
910
280
co2 of cars
NIL
NIL
50.0
450.0
0.0
70.0
true
true
"clear-plot" ""
PENS
"all cars" 10.0 0 -16777216 true "" "histogram [co2 * 100] of cars"
"chosen" 10.0 0 -817084 true "" "histogram [co2 * 100] of cars with [chosen? = 1]"

BUTTON
25
55
105
88
NIL
first-step
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
115
55
215
88
NIL
second-step
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
255
10
318
45
NIL
clear
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
115
10
215
45
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

This model simulates how tax system change is able to affect CO<sub>2</sub> emission from the resulting car sales. This model shows how *feebate* system works: purchase of a car with certain amount of CO<sub>2</sub> emissions number will be rewarded with a rebate or punished with a fee.

`test`


## HOW IT WORKS

The rule conducting the model is adopted from de Haan et al. (2008) and Muller and de Haan (2008) papers. 

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
