extensions [nw]

breed [people person]
breed [dealers dealer]

undirected-link-breed [ friendships friendship ]
undirected-link-breed [ customers customer ]

globals [answer drugs-done-total times-eaten-total work-done-total make-friends-total first-time]
people-own [money dopamine-level times-done times-eaten times-worked times-friend days-since-drug num-friends risk-behavior last-action]
links-own [weight]

;; called upon setup
to setup
  ca
  reset-ticks
  set drugs-done-total 0 ;; total amount of times the population decided to do drugs
  set times-eaten-total 0 ;; total amount of times the population decided to eat
  set work-done-total 0 ;; total amount of times the population decided to do work
  set make-friends-total 0 ;; total amount of times the population decided to make a new friend
  create-network ;; creates the social network
  set first-time true
end

;;uses preferential attachment to create a social network amongst the turtles
to create-network
   nw:generate-preferential-attachment turtles friendships num-people [
     ifelse illegal-world? [
       ifelse random 100 > 5 [make-normal-person] [make-dealer]] ;; 5 dealers for every 100 people
     [make-normal-person] ;; no dealers in this world
     setxy random-xcor random-ycor
     set shape "person"
   ]
end

;; function to create a dealer turtle
to make-dealer
  set breed dealers
  set color white
  let goahead true
  ask link-neighbors [
    if customer-neighbor? myself [set goahead false]]
  if goahead [ask my-links [ set color green set breed customers set thickness .3 ]]
end

;; function to create a normal person
to make-normal-person
  set breed people
  set color sky
  set money random 100 ;; they all start with a random amount of money
  set dopamine-level random 100 ;; they all start with a random amount of happiness
  set times-done 0 ;; no one has tried drugs before
  set risk-behavior random 100 ;; the personality is all random, some people are more risk prone
  set days-since-drug 0
  set times-worked 0 ;; they all have never worked before
  set times-eaten 0 ;; they all have never eaten before
  set times-friend 0 ;; they all have never made a friend before
  set last-action nobody ;; they have never done any action before
  ask my-links [
    ifelse [breed] of other-end = dealers [set color green] ;; to differentiate between dealer links and friendship
      [set weight (random 5) + 1 ;; will be numbers 1 - 5, 1 being the lowest form of friendship, 5 being best friends
        set color 14.9 + weight ;; a visual representation of the friendships, the more red they are the less strong
      ]
    set thickness .3
  ]
end

to go
  ask people [
    let action utility ;; function that picks the option that provides the most utility
    show action
    if action = "work" [work set work-done-total work-done-total + 1]
    if action = "drugs" [do-drugs set drugs-done-total drugs-done-total + 1]
    if action = "eat" [eat set times-eaten-total times-eaten-total + 1]
    if action = "make friend" [make-new-friend set make-friends-total make-friends-total + 1]
    if action != "make friend" [set last-action action]
    if first-time = false [friendships-link-change]
  ]
  if first-time = true [set first-time false]
  show "end of tick"
  tick
end

;; function that calculates which action will give the person the most utility and returns the answer
to-report utility
  let t_work work-change
  let t_drug drug-change
  let t_eat eat-change
  let t_friend make-new-friend-change
  if t_work > t_drug and t_work > t_eat and t_work > t_friend [ set answer "work" ] ;; work gives the most utility
  if t_drug > t_work and t_drug > t_eat and t_drug > t_friend [ set answer "drugs" ] ;; drugs gives the most utility
  if t_eat > t_drug and t_eat > t_work and t_eat > t_friend [ set answer "eat" ] ;; eating gives the most utility
  if t_friend > t_drug and t_friend > t_eat and t_friend > t_work [set answer "make friend" ] ;; making a new friend gives the most utility
  report answer
end

;; function that calculates how much utility a person will get from working
to-report work-change
  let temp-dopamine dopamine-level - energy-of-job ;; your job takes away a certain amount of energy/happiness
  let temp-money money + energy-of-job ;; you get money depending on how much energy your job takes
  report temp-dopamine + temp-money ;; return the decreased dopamine and the increased money added together
end

;; function to actually cause the person to work and gain/lose the benefits
to work
  set dopamine-level dopamine-level - energy-of-job
  set money money + energy-of-job
  set days-since-drug days-since-drug + 1
  set times-worked times-worked + 1
end

;; function that calculates how much utility a person will get from eating
to-report eat-change
  let temp-dopamine dopamine-level + energy-from-food ;; eating gives you energy
  let temp-money money - cost-of-food ;; eating takes away money
  ;; if eating does not put you into debt, report the change, if it does return 0
  ifelse temp-money > 0 [report temp-dopamine + temp-money] [report 0]
end

;; function to actually cause the person to eat and gain/lose the benefits
to eat
  set dopamine-level dopamine-level + energy-from-food
  set money money - cost-of-food
  set days-since-drug days-since-drug + 1
  set times-eaten times-eaten + 1
end

;; function that calculates how much utility a person will get from doing drugs
to-report drug-change
  let ease ease-of-finding
  if ease = 0 [report 0] ;; if the person is not close enough to drugs, he/she will not do it
  let temp-times-done 0
  ;; rehab mechanism, if a person abstains enough then his/her tolerance will decrease
  ifelse days-since-drug > 20 [set temp-times-done 1][set temp-times-done times-done + 1]
  let temp-dopamine dopamine-level + drug-potency ;; the more potent the drug, the more it will increase happiness
  let tolerance 1
  let amount 100 - drug-potency
  ;; the more potent the drug the quicker your tolerance builds, the less potent the more time it takes to build
  if temp-times-done > amount [set tolerance tolerance + 1]
  if temp-times-done > amount * 2 [set tolerance tolerance + 2]
  if temp-times-done > amount * 4 [set tolerance tolerance + 3]
  let temp-money money - (cost-of-drug * tolerance) ;; the drug will get more expensive the more you do it because you build a tolerance
  let total temp-dopamine + temp-money + ease-of-finding
  ifelse random 100 < risk-behavior [ ;; uses probability to calculate whether a person will do the drug
    ifelse temp-money > 0 [report total] ; could afford to do the drug
    [ifelse temp-times-done > amount [report total] ;; the person can't afford it but will do it anyway
      [report 0 ]]] ;; can't afford it and aren't addicted enough to do it anyway
  [report 0]
end

;; function to show how easy it is currently to find drugs
;; different for if it they are legal or illegal
to-report ease-of-finding
  let ease 0
  ifelse illegal-world? [
    ;; a turtle will calculate if they have a connection with a dealer either through
    ;; a direct connection or a friend
    if any? (nw:turtles-in-radius 1) with [breed = dealers and who != [who] of myself ] [set ease 30] ;; the person is connected to the dealer
    if any? (nw:turtles-in-radius 2) with [breed = dealers and who != [who] of myself] [set ease 20] ;; the person is two links away from the dealer
    if any? (nw:turtles-in-radius 3) with [breed = dealers and who != [who] of myself] [set ease 10] ;; the person is three links away from the dealer
  ]
  [
    ;; a turtle will calculate if it has any friends that it could do a drug with
    let amount 100 - drug-potency
    ifelse times-done <= amount [ ;; they are not addicted yet and still need friends to purchase the drug
      if my-links != nobody [
        ask my-links [
          let friend other-end
          ;; find friends and see if they will purchase a drug with you
          ask friend [
            if money > cost-of-drug [ ;; the friend can afford it
              if random 100 < risk-behavior [set ease 20]]]]]] ;; the friend is adventureous enough
    [ set ease 30 ]
  ]
  report ease
end

;; function to actually cause the person to do drugs and gain/lose the benefits
to do-drugs
  ifelse days-since-drug > 20 [ set times-done 1 ] [ set times-done times-done + 1 ]
  let tolerance 1
  let amount 100 - drug-potency
  if times-done > amount [set tolerance tolerance + 1]
  if times-done > amount * 2 [set tolerance tolerance + 2]
  if times-done > amount * 3 [set tolerance tolerance + 3]
  set money money - (cost-of-drug * tolerance)
  set dopamine-level dopamine-level + drug-potency
  set days-since-drug 0
  set times-done times-done + 1
end

;; function that calculates how much utility a person will get from making a new friend
to-report make-new-friend-change
  ;; current turtle has no friends, making a new friend will increase it's dopamine
  ;; by a lot and they are not picky about the new friends they make
  ifelse any? friendship-neighbors = false [
    if any? people in-radius 5 with [who != [who] of myself] != false [report dopamine-level + 100]
  ]
  [
    if any? people in-radius 5 with [who != [who] of myself] != false [
      let possible-friends people in-radius 5 with [who != [who] of myself]
      let temp-d 0
      let linkthere? false
      ask possible-friends [ ;; grab one of the turtles near the asking turtle
        ask my-links [
          ;; if there is already a link from this possible friend to the original turtle
          if [who] of other-end = [who] of myself [set linkthere? true]]
        ;; found a turtle that is not already connected to original turtle
        if linkthere? = false [
          ;; this turtle did the same last-action as the original turtle
          if last-action = [last-action] of myself [set temp-d dopamine-level + 50 stop]] ;; found a turtle that has the same last action
      ]
      report temp-d
    ]
  ]
  report 0
end

;; function to actually cause the person to make a new friend and gain/lose the benefits
to make-new-friend
  ifelse any? friendship-neighbors = false [
    if any? people in-radius 5 with [who != [who] of myself] != false [
      ask one-of people in-radius 5 with [who != [who] of myself] [
        create-friendship-with person [who] of myself [set weight 1 set color 15.9 set thickness .3]]
    ]
    set dopamine-level dopamine-level + 100
  ]
  [
    if any? people in-radius 5 with [who != [who] of myself] != false [
      let possible-friends people in-radius 5 with [who != [who] of myself]
      let linkthere? false
      ask possible-friends [
        ask my-links [
          if [who] of other-end = [who] of myself [set linkthere? true]]
        if linkthere? = false [
          if last-action = [last-action] of myself [
            create-friendship-with person [who] of myself [set weight 1 set color 15.9 set thickness .3 stop ]
          ]
        ]
      ]
      set dopamine-level dopamine-level + 50
    ]
  ]
  set days-since-drug days-since-drug + 1
  set times-friend times-friend + 1
end

;; function to update the social network depending on what the last actions of the turtles are
to friendships-link-change
  ;; grab friend group of turtle
  let friends (nw:turtles-in-radius 1) with [breed = people and who != [who] of myself]
  if friends != nobody [
    ask friends [ ifelse last-action = [last-action] of myself [
      ;; if the current friend did the same last activity as you, increase your friendship
      ask friendship-with turtle ([who] of myself) [ if weight + 1 <= 5 [set weight weight + 1 set color color + 1 ] ] ]
    [ask friendship-with turtle ([who] of myself) [ if weight - 1 >= 0 [set weight weight - 1  set color color - 1] ] ]
    friendship-loss
    ]
  ]
end


to friendship-loss
  let relation friendship-with turtle [who] of myself ;; grab the link between the current turtle and the current friend
  if [weight] of relation = 0 [
    ;; if the friendship has a weight of less than 0, the friendship is not strong and will end
    ask relation [die]
  ]
end



@#$#@#$#@
GRAPHICS-WINDOW
210
10
649
470
16
16
13.0
1
10
1
1
1
0
0
0
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
29
19
95
52
NIL
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
119
21
182
54
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
16
137
188
170
energy-of-job
energy-of-job
0
100
50
1
1
NIL
HORIZONTAL

SLIDER
16
180
188
213
energy-from-food
energy-from-food
0
100
20
1
1
NIL
HORIZONTAL

SLIDER
17
222
189
255
drug-potency
drug-potency
0
100
20
1
1
NIL
HORIZONTAL

SLIDER
21
305
193
338
cost-of-food
cost-of-food
0
100
10
1
1
NIL
HORIZONTAL

SLIDER
20
263
192
296
cost-of-drug
cost-of-drug
0
100
50
1
1
NIL
HORIZONTAL

PLOT
723
21
1095
205
Actions taken over time
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Drugs" 1.0 0 -16777216 true "" "plot drugs-done-total"
"Work" 1.0 0 -7500403 true "" "plot work-done-total"
"Food" 1.0 0 -2674135 true "" "plot times-eaten-total"
"Friends" 1.0 0 -11085214 true "" "plot make-friends-total"

SLIDER
21
347
193
380
num-people
num-people
0
100
100
1
1
NIL
HORIZONTAL

SWITCH
36
98
176
131
illegal-world?
illegal-world?
0
1
-1000

PLOT
744
240
944
390
Average happiness over time
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot sum [dopamine-level] of people"

MONITOR
984
262
1107
307
NIL
drugs-done-total
17
1
11

MONITOR
993
338
1120
383
Number of dealers
count turtles with [breed = dealers]
17
1
11

@#$#@#$#@
## WHAT IS IT?

This model is attempting to understand and compare the effects of drug usage in a world where drugs are legalized and purchased in stores and a world where drugs are illegal and purchased on the street. This is a popular debate amongst policy officials, with people providing data that states drug usage increases when drugs are legalized and data that states drug usage decreases when drugs are legalized.

## HOW IT WORKS
Both Worlds
At each tick, a person will attempt to choose the activity that maximizes his/her utility choosing between: working, eating, doing a drug, and creating new friendships. Working increases money but decreases happiness, eating decreases money but increases happiness, making friends with people who like the same things you do increase happiness, and doing drugs increase happiness, but decrease money factoring the level of tolerance that the individual exhibits towards the drug.

Each person is assigned a random amount of money, a random amount of happiness, and a random risk level. At the beginning of the world, no one has done drugs yet. Preferential attachment was used to create a social network between the turtles with weights on the links to symbolize how strong the friendship is.

Legal World
In the legal world, drugs are purchased from a store. In this model we are assuming that the world symbolizes a neighborhood with one drug store that is equally accessible to the entire population. Because they are so accessible, the metric that makes a person try a drug or not is based on whether or not he/she has friends who are willing (have a high enough risk tendency and can also afford it) to go into the store and purchase it with them. If they do have willing friends then they gain access to the store. As time goes on and the person attempts drugs multiple times, the need to have friends to participate in purchasing drugs decreases.

Illegal World
In the illegal world, drugs are purchased from dealers on the street. In this world, initial access to the drugs are gotten through either a direct connection between a person and a dealer, or through a person who has a friend who knows a dealer. If the number of links between a person and a dealer exceeded 3, the person did not have access to drugs.

## HOW TO USE IT

The model is pretty simple to use, you just use the sliders to set up initial parameters, press setup to initialize the world and then press go to see what happens. If you want to see the illegal world then make sure that switch is turned on!

The energy of job slider is how much energy a job takes from a person in the work function as well as how much money it will gain.

The cost of food is the cost of food.

The cost of drugs is the cost of drugs.

The drug-potency is how potent or addictive the drug you want to sell will be.

## THINGS TO NOTICE

Watch how the number of dealers changes the outputs.
Watch how the overall happiness levels change in both worlds.
Watch how the social network changes as relationships become more colorful (less strong).

## THINGS TO TRY

Suggested things to try is set up the parameters to be similar to your own hometown or a country you are interested in in order to see what the differences in drug use are

## EXTENDING THE MODEL

Add a way to implement income class.


## CREDITS AND REFERENCES

Thank you to EECS 372 for allowing me the ability to create this model
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 6.0-M5
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="High potency, varied cost, illegal world" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks = 100</exitCondition>
    <metric>drugs-done-total</metric>
    <enumeratedValueSet variable="energy-from-food">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cost-of-drug">
      <value value="0"/>
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
      <value value="35"/>
      <value value="40"/>
      <value value="45"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="drug-potency">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-of-job">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-people">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="illegal-world?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cost-of-food">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="High Potency, varied cost, legal world" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks = 100</exitCondition>
    <metric>drugs-done-total</metric>
    <enumeratedValueSet variable="energy-from-food">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cost-of-drug">
      <value value="0"/>
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
      <value value="35"/>
      <value value="40"/>
      <value value="45"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="drug-potency">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-of-job">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-people">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="illegal-world?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cost-of-food">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Low potency, varied cost, legal world" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks = 100</exitCondition>
    <metric>drugs-done-total</metric>
    <enumeratedValueSet variable="energy-from-food">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cost-of-drug">
      <value value="0"/>
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
      <value value="35"/>
      <value value="40"/>
      <value value="45"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="drug-potency">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-of-job">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-people">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="illegal-world?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cost-of-food">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="low potency, varied cost, illegal world" repetitions="10" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks = 100</exitCondition>
    <metric>drugs-done-total</metric>
    <enumeratedValueSet variable="energy-from-food">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cost-of-drug">
      <value value="0"/>
      <value value="5"/>
      <value value="10"/>
      <value value="15"/>
      <value value="20"/>
      <value value="25"/>
      <value value="30"/>
      <value value="35"/>
      <value value="40"/>
      <value value="45"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="drug-potency">
      <value value="20"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="energy-of-job">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-people">
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="illegal-world?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="cost-of-food">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
