breed [ions ion]

globals [
  cell-radius 
  k-size 
  na-size 
  ca-size 
  cl-size 
  physical-repel-distance 
  physical-repel-force
  ionic-interaction-distance
  ionic-interaction-force
  concentration-gradient
  temperature
  membrane-potential
  ]

ions-own[charge classifier location]

patches-own[voltage-gate voltage-inactivate voltage-close direction gate state selectivity]



to setup
  clear-all
  reset-ticks
  
  ask patches [set pcolor black]
  
  ;;; Constants
  set physical-repel-distance .5
  set ionic-interaction-distance 3
  set physical-repel-force -.3
  set ionic-interaction-force .1
  set temperature 20
  
  ;;; Ionic sizes proportionate  
  set k-size (1.38 / 2)
  set na-size (1.02 / 2)
  set ca-size (1 / 2)
  set cl-size (1.81 / 2)
  
  ;; Draw cell
  set cell-radius 30
  draw-cell
  
  ;; Draw ions
  make-ions "ext" "k" ext-conc-potassium * 3.0
  make-ions "ext" "na" ext-conc-sodium * 3.0
  make-ions "ext" "ca" ext-conc-calcium * 3.0
  make-ions "ext" "cl" ext-conc-chlorine * 3.0
  make-ions "int" "k" int-conc-potassium * 3.0
  make-ions "int" "na" int-conc-sodium * 3.0
  make-ions "int" "ca" int-conc-calcium * 3.0
  make-ions "int" "cl" int-conc-chlorine * 3.0
 
 
  ;; Update membrane potential
  update-membrane-potential
  tick
end

;; Moves cells, updates potential and states, sweeps cell
to go
  tick
  update-membrane-potential
  update-vg-channel-states
  sweep-cell
  ask turtles[
    move
  ]
end
  
  
;; Turtle
;;;moves turtles
to move
  ionic-attract-repel
  physical-repulsion
  patches-pull
  membrane-pass
  random-move
end

;; Turtle  
;;;; Moves away from turtles that are nearly touching. Simulates collision
to physical-repulsion
  ; Select a random neighbor that is too close and move away from it
  let too-near one-of other turtles in-radius physical-repel-distance with [location = [location] of myself]
  if too-near != nobody 
  [
    face too-near
    fd physical-repel-force
    if ([pcolor] of patch-here) != color and ([pcolor] of patch-here) != black
    [
      bk physical-repel-force
    ]
  ]
end

;;Turtle
;;; Picks in range ion and reacts accordingly
;;; Excerts force on other ion
to ionic-attract-repel
  ; Select a random neighbor and interact with it
  let near one-of other ions in-radius ionic-interaction-distance with [location = [location] of myself]
  if near != nobody
  [
    face near
    fd ionic-interaction-force * (charge * (0 - ([charge] of near))) * .5 
    ask near[ fd ionic-interaction-force * (charge * (0 - ([charge] of near))) * .5]
  
    if ([pcolor] of patch-here) != color and ([pcolor] of patch-here) != black
    [
      bk ionic-interaction-force * (charge * (0 - ([charge] of near)))
    ]
  ]
end


;; Turtle
;;;; Patches pull in turtles. Simulates ionic gradient. Increases permeability of cell
to patches-pull
  let membrane one-of patches in-radius 3 with [pcolor = [color] of myself]
  if membrane != nobody
  [
    face membrane
    fd 1    
  ]
end

;;Turtles
;;;;Goes through membrane if selectivity parameters match, turtle is on neighbor, and patch state is open
to membrane-pass
  ;; Select all transmembrane patchesb
  let membrane ([neighbors] of patch-here) with [pcolor mod 10 = 6]
  
  if any? membrane
  [
    let chosen one-of membrane with [selectivity = [classifier] of myself or (selectivity = "none")]
    if chosen != nobody
    [
      face patch-at 0 0 
      ifelse location = "ext"
      [
        if ([direction] of chosen) = "in" and ([state] of chosen) = "open"
        [
          set location "int"
          fd 2
        ]
      ]
      [
        if ([direction] of chosen) = "out" and ([state] of chosen) = "open"
        [
          set location "ext"
          bk 2 
        ]
      ]
    ]
  ]
  
  end
  
;; turtle
;;;; Moves randomly to the cell membrane
to random-move
  face patch-at 0 0
  right random 180
  left random 180
  let temp -.7
  if location = "ext"
  [
    set temp .7
  ]
  fd temp
  if ([pcolor] of patch-here) != color and ([pcolor] of patch-here) != black
  [
    bk temp
  ]
end

;;Takes cells that 'hopped' membrane without going through channel and places them on their side
to sweep-cell
 ;; Check for cells that 'hopped' membrane
 ask turtles[
   ifelse location = "ext"
   [
     if (distancexy 0 0) < cell-radius - 1
     [
       setxy 0 0 
       right random 360
       forward cell-radius + 3 + random(10)
     ]
   ] 
   [
     if (distancexy 0 0) > cell-radius + 1
     [
       setxy 0 0 
       right random 360
       forward random cell-radius 
     ]
   ]
 ]
end

;; Draws cell membrane and assigns channels to patches.
to draw-cell
  let i 0
  while [i < 720]
  [
    let membrane_pot (patches with [(abs pxcor = (floor (abs ((cell-radius + 1) * (cos(i / 2)))))) and (abs pycor = (floor (abs ((cell-radius + 1) * (sin(i / 2)))))) or (abs pxcor = (ceiling (abs (cell-radius * (cos(i / 2)))))) and (abs pycor = (ceiling (abs (cell-radius * (sin(i / 2))))))])
    if membrane_pot != Nobody
    [
      ask membrane_pot
      [
        set pcolor white
      ] 
      
      ;; Voltage-gated potassium channel
      if (random 100) < k-channel-density
      [
        ask membrane_pot
        [
          set pcolor green + 1        
          set selectivity "k"
          set voltage-gate 20
          set voltage-close -60
          set voltage-inactivate Nobody
          set gate ">"
          set state "closed"
          set direction "out"
        ]
      ]
      
      ;; Voltage-gated sodium channel
      if abs (random 100 - k-channel-density) < na-channel-density
      [
        ask membrane_pot
        [
          if pcolor = white 
          [
            set pcolor violet + 1
            set selectivity "na"
            set voltage-gate (-60 + random(2))
            set voltage-close voltage-gate
            set voltage-inactivate (20)
            set gate ">"
            set state "closed"
            set direction "in"
          ]
        ] 
      ]
      
      ;; Voltage-gated l calcium channel
      if abs random (100 - k-channel-density - na-channel-density) < l-ca-channel-density 
      [
        ask membrane_pot
        [
          if pcolor = white 
          [
            set pcolor orange + 1
            set selectivity "ca" 
            set voltage-gate 5 + random(10)
            set voltage-inactivate Nobody
            set voltage-close Nobody
            set gate ">"
            set state "closed"
            set direction "in"
          ]
        ] 
      ]
      
      ;; Voltage-gated t calcium channel
      if abs random (100 - k-channel-density - na-channel-density - l-ca-channel-density) < t-ca-channel-density
      [
        ask membrane_pot
        [
          if pcolor = white 
          [
            set pcolor orange + 1
            set selectivity "ca" 
            set voltage-gate -20
            set voltage-inactivate Nobody
            set voltage-close -20
            set gate ">"
            set state "closed"
            set direction "in"
          ]
        ] 
      ]      
      
      ;; Leak K Channel
      if abs random (100 - k-channel-density - na-channel-density - l-ca-channel-density - t-ca-channel-density) < leak-k-channel-density
      [
        ask membrane_pot
        [
          if pcolor = white 
          [
            set pcolor green + 1
            set selectivity "k" 
            set voltage-gate Nobody
            set voltage-inactivate Nobody
            set voltage-close Nobody
            set gate ">"
            set state "open"
            set direction "out"
          ]
        ] 
      ] 
      
      ;; Inward Rectifying (Nephron cells)
      if random (100 - k-channel-density - na-channel-density - l-ca-channel-density - t-ca-channel-density - leak-k-channel-density) < inward-k-channel-density
      [
        ask membrane_pot
        [
          if pcolor = white 
          [
            set pcolor green + 1
            set selectivity "k" 
            set voltage-gate -65 + random(10)
            set voltage-inactivate Nobody
            set voltage-close -50 + random(10)
            set gate "<"
            set state "closed"
            set direction "in"
          ]
        ] 
      ]   
    ]
    set i (i + 1)
  ]
end

;;Creates ions with location classifiers and number of ions

to make-ions[loc class conc]
  ifelse loc = "ext"
  [
    create-ions(conc)
    [
      right random 360
      forward (cell-radius + 3 + random(10))
      set shape "circle"
      
      set classifier class
      set location loc
      
      ifelse class = "k"
      [
        set charge 1
        set color green
        set size k-size
      ]
      [
        ifelse class = "na"
        [
          set charge 1
          set color violet
          set size na-size
        ]
        [
          ifelse class = "ca"
          [
            set charge 2
            set color orange
            set size ca-size
          ]
          [
            set charge -1
            set color yellow
            set size cl-size      
          ]   
        ]   
      ]
    ]
  ]
  [
    create-ions(conc)
    [
      right random 360
      forward random cell-radius
      set shape "circle"
      
      set classifier class
      set location loc
      
      ifelse class = "k"
      [
        set charge 1
        set color green
        set size k-size
      ]
      [
        ifelse class = "na"
        [
          set charge 1
          set color violet
          set size na-size
        ]
        [
          ifelse class = "ca"
          [
            set charge 2
            set color orange
            set size ca-size
          ]
          [
            set charge -1
            set color yellow
            set size cl-size
          ]   
        ]   
      ]
    ]
  ]
end

;; Changes patch states (open, closed, inactive) basded on voltage gating properties

to update-vg-channel-states
  carefully
  [
    ask patches with [pcolor != black][
      if pcolor mod 10 = 6 and (voltage-gate != Nobody)
      [
        ifelse gate = ">"
        [
          if state = "closed"
          [
            if membrane-potential > voltage-gate
            [
              set state "open"  
            ]    
          ]
          if state = "open"
          [
            if membrane-potential < voltage-close
            [
              set state "closed"
            ]
            if voltage-inactivate != Nobody
            [
              ;let prob abs (voltage-inactivate - membrane-potential)
              ;if random (prob * 100) < 1 [set state "inactive"]
              if membrane-potential > voltage-inactivate[set state "inactive"]
              
            ]
          ]
          
          if state = "inactive"
          [
            if membrane-potential < voltage-close
            [
              set state "closed"
            ]
          ]
        ] 
        
        ;; Else
        [
          if state = "closed"
          [
            if membrane-potential < voltage-gate
            [
              set state "open"  
            ]    
          ]
          if state = "open"
          [
            if membrane-potential > voltage-close
            [
              set state "closed"
            ]
            if voltage-inactivate != Nobody
            [
              let prob abs (voltage-inactivate - membrane-potential)
              if membrane-potential > voltage-inactivate[set state "inactive"]
            ]
          ]
          
          if state = "inactive"
          [
            
            if membrane-potential > voltage-close
            [
              set state "closed"
            ]
          ]
        ] 
      ]
    ]
  ]
  [
    ;;None
  ]
end


;; Calculates membrane potential
to update-membrane-potential
  let K-out count turtles with [classifier = "k" and location = "ext"]
  let K-in count turtles with [classifier = "k" and location = "int"]
  let Na-out count turtles with [classifier = "na" and location = "ext"]
  let Na-in count turtles with [classifier = "na" and location = "int"]
  let Ca-out count turtles with [classifier = "ca" and location = "ext"]
  let Ca-in count turtles with [classifier = "ca" and location = "int"]
  let Cl-out count turtles with [classifier = "cl" and location = "ext"]
  let Cl-in count turtles with [classifier = "cl" and location = "int"]
 
  ;;(8.314 * (273.15 + temperature) / 96485)
  carefully 
  [
    set membrane-potential (65 * (ln( (K-in + Na-in + (2 * Ca-in) + Cl-out) / (K-out + Na-out + (2 * Ca-out) + Cl-in))))
  ]
  [
    set membrane-potential "error"
  ]
end
  
  
@#$#@#$#@
GRAPHICS-WINDOW
309
15
967
694
40
40
8.0
1
10
1
1
1
0
1
1
1
-40
40
-40
40
0
0
1
ticks
30.0

BUTTON
1001
320
1064
354
NIL
setup\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
18
34
193
67
int-conc-potassium
int-conc-potassium
0
200
135
1
1
mM
HORIZONTAL

SLIDER
18
66
190
99
int-conc-sodium
int-conc-sodium
0
200
6
1
1
mM
HORIZONTAL

SLIDER
18
99
190
132
int-conc-calcium
int-conc-calcium
0
200
3
1
1
mM
HORIZONTAL

SLIDER
19
175
199
208
ext-conc-potassium
ext-conc-potassium
0
200
34
1
1
mM
HORIZONTAL

SLIDER
19
207
191
240
ext-conc-sodium
ext-conc-sodium
0
200
127
1
1
mM
HORIZONTAL

SLIDER
19
240
191
273
ext-conc-calcium
ext-conc-calcium
0
200
15
1
1
mM
HORIZONTAL

SLIDER
19
273
192
306
ext-conc-chlorine
ext-conc-chlorine
0
200
9
1
1
mM
HORIZONTAL

SLIDER
18
131
191
164
int-conc-chlorine
int-conc-chlorine
0
200
4
1
1
mM
HORIZONTAL

SLIDER
18
334
190
367
k-channel-density
k-channel-density
0
100
41
1
1
%
HORIZONTAL

BUTTON
1088
321
1151
354
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1008
53
1132
98
NIL
membrane-potential
17
1
11

SLIDER
18
432
190
465
na-channel-density
na-channel-density
0
100
25
1
1
%
HORIZONTAL

SLIDER
18
463
196
496
l-ca-channel-density
l-ca-channel-density
0
100
16
1
1
%
HORIZONTAL

PLOT
999
128
1199
278
voltage
time
voltage (V) 
0.0
200.0
-65.0
30.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot membrane-potential"

SLIDER
18
497
198
530
t-ca-channel-density
t-ca-channel-density
0
100
1
1
1
%
HORIZONTAL

SLIDER
18
367
208
400
leak-k-channel-density
leak-k-channel-density
0
100
7
1
1
%
HORIZONTAL

SLIDER
18
400
223
433
inward-k-channel-density
inward-k-channel-density
0
100
5
1
1
%
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This model simulates the electrophysiological properties of a single cell neuron. It shows how membrane potential patterns can be affected by ion concentration values and concentration densities of neuron channels.

## HOW IT WORKS

The turtles are ions. Colored patches are membrane channels. White patches are impassable membrane segments. Green patches and turtles are potassium ions and potassium ion specific channels. Purple refers to sodium ion. Orange refers to calcium ion. Yellow responds to chlorine ion.

At every tick each ion moves. Movement includes ionic repulsion/attraction from nearby ions, physical repulson from colliding with nearby ions, attraction towards ion channels, movemnt through an ion channel, and movement towards the polarized membrane.

At every tick the membrane-potential is calculatied using the Nernst equation (en.wikipedia.org/wiki/Nernst_equation). The equation calculates the logarithmic ratio of extracellular net negative charges over intracellular net negative charges (which are calculated from ion charge values).

When the membrane potential is updated, the state of every voltage gated channel is updated ("open" "closed" or "inactive"). If a channel is open, it will take ions that match its class and directional specificity and push them through to the other side of the membrane, changing the location state of the ion in the process.

If an ion "hops" through the membrane without going through a channel, it will be moved back to the other side the next tick because its location state did not change.

The membrane potential plot shows the membrane potential over time. The higher it is, the more positive ion in the cell/negative ion outside the cell.

## HOW TO USE IT

Choose the intracellular and extracellular concentrations of each ion. Choose the concentration densities of each type of channel.
- Int-conc-... refers to the intracellular concentration of the following ion
- Ext-con-... refers to the extracellular concentration of the following ion
- Channel density refers to the percentage of membrane patches that will be that type of channel. These types include.

   - K channel (outward k at ~30 mV)

   - K Leak channel (outward k)

   - K Inward Rectifyinhg Channel (inward k at ~-60 mV)

   - Na Channel (inward na at ~-30 mV)

   - Ca L Type Channel (inward Ca at ~10 mV)

   - Ca T Type Channel (inward Ca at ~-10 mV)

## THINGS TO NOTICE

The membrane potential behavior varies greatly with different parameter inputs. Notice how small things like less Na channels or relatively high intracellular potassium ion concentrations greatly affect the electrophysiological behavior.


## THINGS TO TRY

For standard, action potential eliciting behaviors, make sure intracellular k concentrations is high (~150 mM), extracellular na ion is high (~ 150 mM) and there is some presence of extracellular ca and k ion.

There are various configurations that will elicitt action potentials, and some that will come close, so try manipulating the inputs and see how the electrophysiological parameters change.

Its important to know that ions can only flow in the direction of their channels, so putting a lot of Na inside the cell will change the membrane potential but will limit the dynamics of the cell.

## EXTENDING THE MODEL

This model is incomplete and can always use more volted gated membrane channel classes (found in draw-cell).

The first major extension that will be added is ligand gating properties. A lot of channels with critical functions are opened by signaling molecules. A good place to start would be adding NMDA and AMPA receptors which open to glutamate molecule attachment, and let in Na ion until the ligand detaches.


## NETLOGO FEATURES

In vitro, ion movement is determined by a resultant force vector (ionic force, collision force, channel pull, etc.). In NetLogo it is possible to model movement by completing these force induced movements sequentially. Although it is not entirely in line with the actual principle, it is smoother in actual implementation and is super effective in creating clean movement.

## RELATED MODELS

Membrane Formation. Different scale, but useful for demonstrating ionic movement.

## CREDITS AND REFERENCES

Brendan Frick 
2015
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
